/* -*- mode: C++; c-basic-offset: 2; indent-tabs-mode: nil -*- */
/*
 *  Main authors:
 *     Christian Schulte <schulte@gecode.org>
 *
 *  Contributing authors:
 *     Samuel Gagnon <samuel.gagnon92@gmail.com>
 *
 *  Copyright:
 *     Christian Schulte, 2015
 *     Samuel Gagnon, 2018
 *
 *  This file is part of Gecode, the generic constraint
 *  development environment:
 *     http://www.gecode.org
 *
 *  Permission is hereby granted, free of charge, to any person obtaining
 *  a copy of this software and associated documentation files (the
 *  "Software"), to deal in the Software without restriction, including
 *  without limitation the rights to use, copy, modify, merge, publish,
 *  distribute, sublicense, and/or sell copies of the Software, and to
 *  permit persons to whom the Software is furnished to do so, subject to
 *  the following conditions:
 *
 *  The above copyright notice and this permission notice shall be
 *  included in all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 *  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 *  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 *  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 *  LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 *  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 *  WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */

#include <gecode/driver.hh>

#include <gecode/int.hh>
#include <gecode/minimodel.hh>

using namespace Gecode;

/// Instance data
namespace {

  /// Number of random seeds
  static const int n_seeds = 32;
  /// Random seeds (generated by hardware random generator)
  static const unsigned int seeds[n_seeds] = {
    4156683608, 97006981, 1375463525, 841714419,
    715462902, 2418870981, 1624092856, 123015042,
    2456879733, 1964065250, 2622145091, 3852245775,
    205913142, 1921802200, 3118573517, 315803625,
    1823610061, 3329660933, 3706705607, 2648334003,
    424666975, 1774959171, 4178564660, 52701009,
    2475642041, 2171371390, 1476278023, 2270518404,
    2981774515, 910724046, 92053990, 1980563460
  };


  /// %Options for %QCP problems
  class QCPOptions : public InstanceOptions {
  protected:
    /// Tie-breaking factor
    Driver::DoubleOption _tbf;
  public:
    /// Initialize options for example with name \a s
    QCPOptions(const char* s)
    : InstanceOptions(s),
      _tbf("tbf", "tie-breaking factor",0.0) {
      // Add options
      add(_tbf);
    }
    /// Return tie-breaking factor
    double tbf(void) const {
      return _tbf.value();
    }
    /// Set tie-breaking factor
    void tbf(double d) {
      _tbf.value(d);
    }
  };


  /// Instances
  extern const int* qcp[];
  /// Instance names
  extern const char* name[];

  /// A wrapper class for instance data
  class Spec {
  protected:
    /// Raw instance data
    const int* data;
    /// Access info at position \a i and \a j
    int info(int i, int j) const {
      int n = data[0];
      assert((i >= 0) && (i < n));
      assert((j >= 0) && (j < n));
      return data[1 + (i * n) + j];
    }
    /// Find instance by name \a s
    static const int* find(const char* s) {
      for (int i=0; name[i] != nullptr; i++)
        if (!strcmp(s,name[i]))
          return qcp[i];
      return nullptr;
    }
  public:
    /// Whether a valid specification has been found
    bool valid(void) const {
      return data != nullptr;
    }
    /// Initialize
    Spec(const char* s) : data(find(s)) {}
    /// Return problem size
    int size(void) const {
      return data[0];
    }
    /// Whether field is pre-assigned
    bool assigned(int i, int j) const {
      return info(i,j) > 0;
    }
    /// Return value at position \a i and \j
    int val(int i, int j) const {
      assert(assigned(i,j));
      return info(i,j) - 1;
    }
  };

}


/**
 * \brief %Example: Quasigroup completion
 *
 * \ingroup Example
 *
 */
class QCP : public Script {
protected:
  /// Options
  const QCPOptions& opt;
  /// Specification
  const Spec spec;
  /// Field elements e
  IntVarArray e;
  /// Tie-breaking factor
  double tbf;
public:
  /// Propagation to use for model
  enum {
    PROP_BINARY,  ///< Use only binary disequality constraints
    PROP_DISTINCT ///< Use distinct constraints
  };
  /// Branching to use for model
  enum {
    BRANCH_SIZE,     ///< Use smallest domain size first
    BRANCH_AFC_SIZE, ///< Use largest AFC divided by domain size
    BRANCH_CBS_MAX_SD ///< Use maximum solution density
  };
  /// Actual model
  QCP(const QCPOptions& opt0)
    : Script(opt0),
      opt(opt0), spec(opt.instance()),
      e(*this, spec.size() * spec.size(), 0, spec.size()-1) {
    // Problem size
    int n = spec.size();
    // Matrix for elements
    Matrix<IntVarArray> m(e, n);

    // Assign fields
    for (int i=0; i<n; i++)
      for (int j=0; j<n; j++)
        if (spec.assigned(i,j))
          rel(*this, m(i,j) == spec.val(i,j));

    // Post constraints
    switch (opt.propagation()) {
    case PROP_BINARY:
      for (int i=0; i<n; i++)
        for (int k=0; k<n; k++)
          for (int l=k+1; l<n; l++) {
            rel(*this, m(i,k) != m(i,l));
            rel(*this, m(k,i) != m(l,i));
          }
      break;
    case PROP_DISTINCT:
      for (int i=0; i<n; i++) {
        distinct(*this, m.row(i), opt.ipl());
        distinct(*this, m.col(i), opt.ipl());
      }
      break;
    }

    if (opt.assets() == 0) {
      // Post branchers directly, as no portfolio search requested
      switch (opt.branching()) {
      case BRANCH_CBS_MAX_SD:
#ifdef GECODE_HAS_CBS
        cbsbranch(*this, e);
#endif
      case BRANCH_SIZE:
        branch(*this, e, INT_VAR_SIZE_MIN(), INT_VAL_MIN());
        break;
      case BRANCH_AFC_SIZE:
        branch(*this, e, INT_VAR_AFC_SIZE_MAX(opt.decay()), INT_VAL_MIN());
        break;
      }
    }
  }
  /// Slave function for portfolio search
  virtual bool slave(const MetaInfo& mi) {
    if (mi.type() == MetaInfo::PORTFOLIO) {
      double tbf = opt.tbf();
      Rnd r(seeds[mi.asset() % n_seeds]);
      switch (opt.branching()) {
      case BRANCH_SIZE:
        {
          auto tbl =
            [tbf] (const Space&, double w, double b) {
              assert(w >= b);
              return b + (w - b) * tbf;
            };
          branch(*this, e, tiebreak(INT_VAR_SIZE_MIN(tbl),
                                    INT_VAR_RND(r)),
                 INT_VAL_MIN());
        }
        break;
      case BRANCH_AFC_SIZE:
        {
          auto tbl =
            [tbf] (const Space&, double w, double b) {
              assert(b >= w);
              return b - (b - w) * tbf;
            };
          branch(*this, e, tiebreak(INT_VAR_AFC_SIZE_MAX(opt.decay(), tbl),
                                    INT_VAR_RND(r)),
                 INT_VAL_MIN());
        }
        break;
      default: ;
      }
    }
    return true;
  }
  /// Constructor for cloning \a s
  QCP(QCP& s)
    : Script(s), opt(s.opt), spec(s.spec) {
    e.update(*this, s.e);
  }
  /// Copy during cloning
  virtual Space*
  copy(void) {
    return new QCP(*this);
  }
  /// Print solution
  virtual void
  print(std::ostream& os) const {
    int n = spec.size();
    Matrix<IntVarArray> m(e, n);

    for (int i=0; i<n; i++) {
      os << "\t";
      for (int j=0; j<n; j++) {
        if (m(i,j).assigned())
          os.width(2);
        os << m(i,j) << " ";
      }
      os << std::endl;
    }
  }
};

/** \brief Main-function
 *  \relates QCP
 */
int
main(int argc, char* argv[]) {
  QCPOptions opt("QCP");

  opt.branching(QCP::BRANCH_AFC_SIZE);
  opt.branching(QCP::BRANCH_SIZE, "size");
  opt.branching(QCP::BRANCH_AFC_SIZE, "afc");
#ifdef GECODE_HAS_CBS
  opt.branching(QCP::BRANCH_CBS_MAX_SD, "maxSD");
#endif

  opt.ipl(IPL_DOM);

  opt.propagation(QCP::PROP_DISTINCT);
  opt.propagation(QCP::PROP_BINARY, "binary",
                      "only binary disequality constraints");
  opt.propagation(QCP::PROP_DISTINCT, "distinct",
                      "distinct constraints");

  opt.instance(name[0]);

  opt.parse(argc,argv);
  if (!Spec(opt.instance()).valid()) {
    std::cerr << "Error: unknown instance" << std::endl;
    return 1;
  }
  Script::run<QCP,DFS,QCPOptions>(opt);
  return 0;
}

namespace {

  /*
   * Instances taken from the 2009 CSP competition.
   */

  const int d10_67_0[] = {
    // Size: 10 x 10
    10,
    // Pre-assigned fields
    2,4,0,0,5,7,0,0,0,0,
    0,6,0,4,0,0,0,3,0,0,
    3,0,9,0,0,5,0,0,8,0,
    0,0,2,10,7,0,0,0,0,0,
    0,0,0,0,0,0,5,1,0,8,
    4,0,0,0,0,0,3,6,0,0,
    0,0,0,8,0,4,0,0,0,1,
    0,5,0,3,8,0,0,0,0,0,
    0,0,1,0,0,0,4,0,5,0,
    0,0,0,0,0,6,0,8,9,3
  };

  const int d10_67_10[] = {
    // Size: 10 x 10
    10,
    // Pre-assigned fields
    0,10,0,0,0,0,4,2,0,0,
    0,0,0,3,8,0,0,0,6,0,
    0,0,0,0,0,4,0,0,1,6,
    1,6,8,5,0,0,0,0,0,0,
    0,9,1,0,0,7,0,0,0,0,
    0,0,0,0,5,0,7,1,4,0,
    3,0,2,4,0,0,8,0,0,0,
    4,0,0,0,2,0,0,0,0,5,
    0,7,6,0,0,0,0,0,0,3,
    0,0,0,0,0,10,6,4,0,0
  };

  const int d10_67_11[] = {
    // Size: 10 x 10
    10,
    // Pre-assigned fields
    7,0,0,0,1,0,0,0,2,0,
    0,0,7,0,0,0,9,0,0,6,
    0,0,0,0,0,0,6,7,5,0,
    9,3,0,7,0,0,0,0,0,2,
    0,7,0,4,8,0,0,0,0,0,
    0,1,10,0,0,0,3,0,0,0,
    2,0,0,0,0,9,7,1,0,0,
    0,0,1,0,0,7,0,3,8,0,
    0,0,0,0,0,8,0,0,1,5,
    0,9,0,6,7,0,0,0,0,0
  };

  const int d10_67_12[] = {
    // Size: 10 x 10
    10,
    // Pre-assigned fields
    0,4,0,0,0,0,8,0,2,0,
    0,5,7,0,9,0,0,0,0,0,
    6,3,0,0,0,2,0,0,0,1,
    0,0,6,0,7,0,0,8,0,0,
    3,7,0,0,0,0,0,0,4,5,
    4,0,0,0,0,0,0,2,6,0,
    0,0,0,9,2,0,0,6,0,0,
    0,0,0,1,0,7,6,0,0,0,
    10,0,0,0,6,0,0,0,0,2,
    0,0,4,5,0,6,3,0,0,0
  };

  const int d10_67_13[] = {
    // Size: 10 x 10
    10,
    // Pre-assigned fields
    0,0,9,6,0,0,0,0,1,0,
    0,3,0,0,4,0,0,0,10,0,
    0,0,0,0,6,0,10,0,0,7,
    0,0,0,7,0,8,0,10,3,0,
    6,7,10,0,0,0,0,0,0,4,
    7,0,0,10,0,0,4,0,0,0,
    0,0,0,4,0,0,6,2,0,0,
    0,0,5,0,10,0,0,3,0,0,
    4,9,0,0,0,3,0,0,0,0,
    9,8,0,0,0,2,0,0,0,1
  };

  const int d10_67_14[] = {
    // Size: 10 x 10
    10,
    // Pre-assigned fields
    7,0,10,9,0,0,0,0,0,0,
    3,0,0,8,0,0,0,0,9,0,
    0,2,0,0,7,0,0,0,0,4,
    9,0,0,10,0,0,2,0,0,0,
    0,0,9,2,0,8,4,0,0,0,
    0,0,5,0,1,0,9,8,0,0,
    0,3,0,0,0,0,0,0,8,6,
    0,0,0,0,0,5,0,7,0,9,
    0,7,0,0,0,0,0,9,5,0,
    0,6,0,0,2,1,5,0,0,0
  };

  const int d10_67_1[] = {
    // Size: 10 x 10
    10,
    // Pre-assigned fields
    9,0,10,0,0,0,0,0,2,6,
    0,8,0,1,0,0,6,3,0,0,
    0,0,0,0,0,7,0,1,0,4,
    10,0,0,0,0,3,0,0,6,0,
    6,0,0,0,0,0,9,0,0,2,
    0,0,0,4,6,2,0,0,0,0,
    0,10,8,6,0,0,0,0,0,0,
    0,0,0,0,5,0,4,0,1,0,
    8,2,0,0,0,0,10,6,0,0,
    0,0,1,0,10,0,0,0,7,0
  };

  const int d10_67_2[] = {
    // Size: 10 x 10
    10,
    // Pre-assigned fields
    0,0,2,8,0,0,6,0,1,0,
    6,8,0,0,0,0,0,4,0,0,
    9,0,0,7,0,0,0,0,0,4,
    0,0,0,6,0,0,3,0,0,10,
    0,0,0,0,7,3,10,0,0,8,
    0,0,5,0,0,2,0,3,0,0,
    0,7,0,0,8,9,0,0,6,0,
    5,0,0,0,0,0,0,6,7,0,
    7,0,3,0,5,0,0,0,0,0,
    0,2,0,0,0,8,0,0,9,0
  };

  const int d10_67_3[] = {
    // Size: 10 x 10
    10,
    // Pre-assigned fields
    0,0,0,0,10,6,0,0,0,9,
    0,0,0,1,8,0,2,0,0,0,
    0,0,9,7,0,0,1,0,0,8,
    4,9,0,2,0,0,0,0,7,0,
    0,2,6,0,0,0,0,9,0,0,
    0,0,1,9,0,0,5,0,0,0,
    0,0,0,0,0,4,0,7,8,0,
    10,8,0,0,0,0,0,0,6,0,
    9,4,0,0,0,0,0,0,5,2,
    0,0,0,0,5,8,0,3,0,0
  };

  const int d10_67_4[] = {
    // Size: 10 x 10
    10,
    // Pre-assigned fields
    5,0,3,0,0,0,0,10,0,0,
    0,0,5,0,0,0,2,0,3,10,
    0,0,0,7,0,8,1,0,0,0,
    10,5,0,0,1,0,0,0,0,0,
    0,10,8,0,0,0,0,0,1,0,
    0,0,1,3,0,0,0,9,0,6,
    0,2,0,1,0,0,0,4,0,5,
    0,0,0,0,5,9,0,0,10,0,
    0,0,0,0,3,0,6,0,0,4,
    8,0,0,0,0,6,0,0,5,0
  };

  const int d10_67_5[] = {
    // Size: 10 x 10
    10,
    // Pre-assigned fields
    7,0,0,0,0,0,0,8,1,0,
    0,0,6,1,0,0,0,0,0,2,
    0,3,0,8,9,2,0,0,0,0,
    0,0,0,0,7,4,0,0,0,10,
    0,4,0,0,0,0,10,9,0,0,
    0,0,1,0,0,0,0,0,6,5,
    9,0,4,0,0,0,1,0,0,0,
    0,5,0,4,0,8,0,0,10,0,
    0,9,0,0,6,0,8,3,0,0,
    5,0,0,0,0,3,0,0,9,0
  };

  const int d10_67_6[] = {
    // Size: 10 x 10
    10,
    // Pre-assigned fields
    6,0,10,5,0,0,0,0,1,0,
    0,0,0,0,2,5,0,0,7,0,
    0,7,0,1,10,0,0,0,6,0,
    0,0,7,0,0,0,4,0,9,0,
    1,0,0,0,0,2,0,5,0,0,
    0,8,5,0,0,0,0,0,0,7,
    0,4,0,7,9,0,0,0,0,0,
    0,0,0,0,0,7,10,6,0,0,
    0,0,0,8,0,0,0,7,0,1,
    7,0,0,0,0,0,1,8,0,2
  };

  const int d10_67_7[] = {
    // Size: 10 x 10
    10,
    // Pre-assigned fields
    0,7,0,0,6,0,0,0,0,1,
    0,1,4,0,0,0,0,3,0,6,
    1,0,0,2,3,0,0,0,0,0,
    2,0,0,3,0,8,0,0,0,0,
    0,0,0,0,0,6,0,8,7,0,
    0,0,6,0,0,0,4,0,3,0,
    0,0,0,4,0,0,10,0,6,0,
    0,10,8,0,1,9,0,0,0,0,
    0,0,0,0,0,7,9,6,0,3,
    6,0,0,0,0,0,0,7,0,4
  };

  const int d10_67_8[] = {
    // Size: 10 x 10
    10,
    // Pre-assigned fields
    0,5,0,4,0,0,3,0,0,0,
    0,0,0,0,7,0,2,0,0,4,
    0,6,10,0,0,3,0,0,9,0,
    4,0,0,0,0,7,0,0,3,10,
    0,8,2,0,0,0,4,0,0,6,
    8,0,0,10,0,0,0,6,0,0,
    0,0,4,0,0,2,0,7,0,0,
    3,0,0,0,2,0,10,0,0,0,
    0,0,0,8,3,0,0,0,0,5,
    0,0,0,0,0,4,0,5,2,0
  };

  const int d10_67_9[] = {
    // Size: 10 x 10
    10,
    // Pre-assigned fields
    2,8,0,0,0,7,0,0,0,0,
    0,5,0,8,0,2,0,0,0,6,
    0,0,0,0,0,0,2,5,0,4,
    4,0,0,0,2,0,3,1,0,0,
    0,0,10,0,0,0,0,0,2,9,
    0,0,8,9,0,0,0,0,5,0,
    0,0,0,6,0,0,0,0,9,8,
    0,0,4,0,9,6,0,0,1,0,
    8,0,0,0,0,0,6,7,0,0,
    0,2,0,0,8,0,0,9,0,0
  };

  const int d15_120_0[] = {
    // Size: 15 x 15
    15,
    // Pre-assigned fields
    0,0,0,2,0,0,11,0,0,4,15,12,13,8,0,
    0,7,2,0,0,0,0,13,15,0,1,0,4,9,0,
    0,0,6,3,0,0,12,0,8,0,13,0,0,1,2,
    0,0,1,7,0,6,3,0,0,2,0,9,0,11,0,
    0,0,0,12,0,0,4,0,0,10,9,3,0,7,6,
    13,0,0,0,7,2,6,0,5,0,14,8,0,0,0,
    7,0,14,0,12,4,15,1,0,6,0,0,0,0,0,
    15,4,8,0,9,5,0,0,0,0,0,11,0,0,10,
    11,13,0,0,2,0,0,12,0,0,3,0,6,0,14,
    0,3,10,11,0,0,14,0,9,0,0,2,1,0,0,
    0,0,0,0,11,12,0,7,4,0,0,6,0,14,5,
    0,8,11,4,0,7,0,0,0,3,0,0,2,0,15,
    2,5,0,0,4,0,0,6,10,14,0,0,8,0,0,
    5,6,0,0,0,0,0,8,0,13,7,0,9,0,4,
    3,0,5,1,13,11,0,4,0,0,0,0,0,2,0
  };

  const int d15_120_10[] = {
    // Size: 15 x 15
    15,
    // Pre-assigned fields
    0,0,8,14,1,0,7,0,4,10,0,0,0,0,9,
    0,11,0,13,0,12,10,0,5,0,0,0,4,2,0,
    0,7,11,0,0,0,8,0,3,2,0,0,0,6,10,
    1,0,0,0,2,0,0,8,0,12,9,0,14,7,0,
    2,0,0,0,0,11,1,7,0,0,14,0,12,0,15,
    4,0,0,2,14,0,0,9,0,0,11,7,13,0,0,
    12,0,0,0,10,0,0,0,14,9,4,6,1,0,0,
    10,15,0,0,3,0,0,0,0,6,5,13,0,8,0,
    0,0,2,0,0,1,0,3,0,0,15,9,0,0,11,
    13,0,15,9,0,0,0,4,10,0,0,5,11,0,0,
    0,9,12,0,0,14,0,0,0,0,0,8,10,5,4,
    3,0,0,1,0,7,5,10,0,13,6,0,0,0,0,
    0,3,7,0,9,5,15,6,1,0,0,0,0,11,0,
    0,8,10,5,0,0,11,0,15,0,0,0,0,4,1,
    0,4,0,11,8,15,0,0,0,7,0,2,0,0,5
  };

  const int d15_120_11[] = {
    // Size: 15 x 15
    15,
    // Pre-assigned fields
    0,0,9,0,10,0,14,0,3,0,8,0,0,13,6,
    7,0,0,8,0,0,0,0,1,11,9,4,14,0,0,
    0,0,0,2,0,12,13,0,4,8,11,0,0,15,0,
    0,2,0,11,6,15,0,14,0,0,0,9,0,0,1,
    0,0,11,0,9,13,8,5,0,0,0,14,12,3,0,
    13,1,12,0,0,0,6,7,0,0,0,0,0,2,3,
    0,6,0,0,3,4,9,10,5,0,0,0,0,0,0,
    4,0,0,0,0,10,0,9,8,2,15,0,0,0,13,
    1,12,10,13,0,0,0,4,0,0,0,6,15,0,0,
    8,15,0,6,0,2,0,0,0,7,0,3,1,0,0,
    0,13,2,0,8,0,0,0,0,15,4,0,9,11,0,
    3,0,0,10,7,14,2,0,0,0,1,0,0,5,0,
    0,0,5,1,0,0,0,0,0,4,14,7,10,0,12,
    2,8,0,0,0,0,11,6,14,1,0,0,0,0,15,
    0,0,13,0,12,0,0,0,9,0,0,8,11,6,10
  };

  const int d15_120_12[] = {
    // Size: 15 x 15
    15,
    // Pre-assigned fields
    0,0,6,8,9,0,7,0,10,0,0,0,2,4,0,
    1,8,0,14,0,2,0,15,0,0,0,0,0,3,6,
    0,6,9,0,0,5,3,4,0,0,11,0,0,0,7,
    0,3,13,0,0,15,0,0,0,11,14,2,0,9,0,
    10,7,0,5,1,11,0,0,0,0,0,0,14,0,9,
    3,0,11,0,0,0,0,13,0,4,6,8,0,0,10,
    0,0,12,7,5,13,14,0,0,2,0,4,0,0,0,
    0,10,4,0,0,0,9,7,0,0,15,0,8,0,11,
    0,0,0,4,0,0,6,1,8,0,3,10,7,0,0,
    0,11,0,0,6,0,0,0,1,14,0,5,15,10,0,
    2,13,0,12,0,0,0,0,11,0,4,14,0,7,0,
    5,0,0,11,10,7,0,0,2,0,0,0,4,8,0,
    14,0,0,0,0,6,0,0,4,8,0,12,5,0,0,
    4,0,0,0,2,0,5,14,13,9,0,0,0,12,1,
    0,0,1,0,13,0,4,10,0,7,8,0,0,0,15
  };

  const int d15_120_13[] = {
    // Size: 15 x 15
    15,
    // Pre-assigned fields
    0,7,0,15,10,0,0,8,9,0,12,14,0,0,0,
    0,0,5,2,0,12,8,0,0,7,0,0,11,0,4,
    7,10,0,12,0,0,0,3,0,0,0,13,9,1,0,
    0,0,9,0,14,10,3,11,0,0,0,5,0,6,0,
    12,8,11,14,0,3,0,5,0,9,0,0,0,0,0,
    6,0,12,5,3,0,0,15,13,0,2,0,0,0,0,
    0,0,0,0,1,0,14,0,10,3,9,0,0,13,6,
    0,0,4,0,0,9,1,0,12,0,0,6,2,5,0,
    2,1,0,0,0,8,6,7,0,15,0,0,0,10,0,
    0,4,14,0,0,0,0,0,2,8,5,0,12,0,11,
    10,0,0,0,5,13,15,0,11,0,0,3,0,2,0,
    15,9,2,0,0,0,0,0,8,0,11,0,14,0,1,
    0,15,0,13,11,0,12,0,0,0,8,0,4,0,14,
    0,0,0,9,4,0,0,6,0,2,0,10,0,15,13,
    8,0,0,0,0,15,0,0,0,13,14,1,6,0,10
  };

  const int d15_120_14[] = {
    // Size: 15 x 15
    15,
    // Pre-assigned fields
    1,0,9,0,0,13,2,10,0,0,0,14,0,0,0,
    0,0,2,8,6,0,13,4,0,0,0,3,0,5,0,
    8,11,5,0,0,0,0,0,0,13,6,0,0,1,3,
    13,0,10,0,0,0,4,5,0,3,0,9,0,0,6,
    5,3,0,11,13,0,12,0,7,0,0,0,0,2,0,
    0,6,13,0,0,0,0,0,15,9,14,0,12,4,0,
    11,14,0,0,12,6,8,0,2,4,0,0,0,3,0,
    0,0,0,0,0,14,0,0,6,10,15,5,13,0,9,
    0,0,0,0,4,12,0,9,8,0,0,2,15,0,7,
    12,0,0,0,0,5,3,15,9,2,0,0,4,0,0,
    0,0,0,4,1,0,0,0,0,12,9,7,2,0,15,
    0,12,0,9,14,10,0,0,0,0,4,0,6,7,0,
    15,10,0,3,11,0,0,0,0,0,0,0,9,14,12,
    0,8,1,2,0,15,0,6,11,0,3,0,0,0,0,
    0,0,11,13,0,0,7,1,0,0,5,4,0,0,14
  };

  const int d15_120_1[] = {
    // Size: 15 x 15
    15,
    // Pre-assigned fields
    4,0,0,0,8,0,2,14,0,0,10,0,12,3,0,
    7,0,0,0,13,0,14,0,3,8,0,9,0,0,4,
    6,15,0,13,11,14,4,0,0,0,12,0,0,0,2,
    5,0,0,10,0,12,6,8,0,0,0,0,7,13,0,
    11,1,13,0,3,0,15,0,0,0,0,0,0,8,6,
    0,0,14,3,0,0,0,0,1,0,0,11,10,15,0,
    0,8,7,14,9,0,0,0,0,1,0,12,4,0,0,
    15,9,0,0,10,13,0,5,6,2,0,0,0,0,0,
    0,5,15,8,0,0,0,0,14,0,11,0,0,2,12,
    0,0,12,0,0,0,7,9,0,5,0,13,2,14,0,
    0,2,1,0,0,0,0,3,15,13,14,7,0,0,0,
    0,0,4,1,0,10,0,12,0,14,9,0,0,0,11,
    14,0,0,0,0,5,0,4,7,0,0,15,9,0,10,
    0,6,0,15,0,7,0,0,0,3,2,8,0,0,13,
    0,0,0,0,6,15,5,0,10,0,4,0,11,1,0
  };

  const int d15_120_2[] = {
    // Size: 15 x 15
    15,
    // Pre-assigned fields
    5,0,0,4,0,9,10,0,2,3,14,0,0,0,0,
    14,5,2,0,0,0,0,0,11,4,3,10,0,0,0,
    11,10,0,5,0,12,7,14,0,0,0,0,13,0,0,
    0,0,0,15,12,0,0,0,7,0,2,11,0,5,6,
    0,13,0,0,0,0,0,7,3,0,8,14,0,6,9,
    6,0,0,0,5,0,3,9,15,0,0,0,0,14,2,
    3,0,0,0,0,11,4,0,13,2,0,0,1,0,14,
    0,1,13,2,3,14,0,0,0,0,5,0,0,10,0,
    0,0,5,0,9,6,0,15,0,10,0,13,3,0,0,
    7,0,1,13,0,0,9,6,0,11,0,0,8,0,0,
    2,0,11,0,7,0,0,4,6,0,0,5,0,8,0,
    0,14,0,0,10,15,0,0,0,7,13,0,0,3,11,
    0,8,9,0,4,2,12,0,0,0,0,0,15,7,0,
    0,4,15,14,0,0,0,0,0,0,7,3,6,0,13,
    0,0,0,1,0,0,13,11,0,14,0,15,9,0,12
  };

  const int d15_120_3[] = {
    // Size: 15 x 15
    15,
    // Pre-assigned fields
    0,0,13,14,0,0,0,6,0,1,9,0,0,10,2,
    2,0,10,5,0,0,0,13,14,12,6,0,0,0,0,
    0,0,0,8,4,6,9,0,0,0,0,13,0,1,10,
    0,0,0,0,7,5,1,8,0,0,12,0,10,0,14,
    11,2,0,15,0,0,0,12,0,0,0,10,13,8,0,
    10,7,0,0,0,2,0,0,0,0,0,3,12,11,13,
    14,0,0,0,3,9,5,0,10,15,0,0,0,0,7,
    9,0,5,0,0,0,0,0,15,7,1,8,0,0,12,
    1,0,4,3,0,0,15,14,12,0,0,0,0,0,6,
    5,13,3,0,0,0,8,0,0,0,0,12,1,4,0,
    0,0,0,1,10,3,0,0,0,8,14,7,4,0,0,
    0,4,0,0,0,0,0,11,2,0,5,15,14,6,0,
    0,6,12,0,1,15,10,0,9,0,13,0,0,0,0,
    0,3,0,6,14,11,0,0,4,9,0,0,2,0,0,
    0,5,11,0,6,0,7,9,0,10,0,0,0,13,0
  };

  const int d15_120_4[] = {
    // Size: 15 x 15
    15,
    // Pre-assigned fields
    15,0,0,3,2,0,4,0,0,6,0,0,0,13,9,
    12,10,0,9,0,13,0,0,0,0,1,11,15,0,0,
    0,0,9,10,1,0,8,0,0,12,0,0,7,0,5,
    5,0,3,0,9,0,0,6,7,13,0,0,8,0,0,
    0,4,0,0,3,14,0,8,0,0,12,0,0,2,6,
    0,0,0,0,8,0,9,3,0,5,15,6,0,11,0,
    0,9,0,0,0,4,1,0,0,0,2,0,14,0,10,
    0,0,6,1,0,11,3,5,0,0,0,7,0,15,0,
    0,6,1,0,10,12,0,13,14,0,7,0,0,0,0,
    0,0,0,0,0,0,5,0,12,3,8,14,4,7,1,
    6,0,14,8,0,15,0,0,5,7,0,0,0,1,0,
    14,8,0,0,0,0,11,7,13,0,0,12,1,0,0,
    8,7,0,0,0,5,0,11,0,0,14,0,2,0,15,
    13,0,2,15,5,0,0,0,11,1,0,8,0,0,0,
    0,11,13,5,0,0,0,0,10,0,0,3,0,8,12
  };

  const int d15_120_5[] = {
    // Size: 15 x 15
    15,
    // Pre-assigned fields
    7,4,0,0,0,9,0,14,0,0,2,0,13,3,0,
    0,8,4,0,0,1,10,0,0,11,0,14,0,0,13,
    2,0,15,0,1,5,7,0,8,0,0,0,0,0,6,
    8,11,0,0,0,0,12,0,3,0,9,0,5,7,0,
    0,13,5,10,0,7,3,0,0,0,0,0,11,0,9,
    0,0,0,5,6,14,9,13,15,0,0,1,0,0,0,
    0,10,14,0,15,8,0,2,7,0,0,0,0,4,0,
    0,1,0,0,0,0,0,8,0,4,7,5,0,10,14,
    0,0,0,3,5,0,0,12,9,8,10,0,0,2,0,
    15,0,0,0,3,0,0,0,2,6,5,11,12,0,0,
    13,0,0,9,0,0,6,0,0,0,0,15,2,11,12,
    0,0,0,6,14,13,15,9,0,0,0,0,4,0,3,
    5,0,13,8,4,0,0,15,0,0,0,9,1,0,0,
    12,15,7,0,0,0,1,0,11,5,3,0,0,0,0,
    0,0,1,13,0,0,0,0,0,12,8,6,0,9,10
  };

  const int d15_120_6[] = {
    // Size: 15 x 15
    15,
    // Pre-assigned fields
    13,0,7,11,0,12,0,0,3,5,0,6,0,0,0,
    0,13,0,0,3,0,0,12,11,0,7,0,1,0,15,
    0,0,3,5,0,8,7,0,0,0,2,10,0,0,4,
    15,0,1,0,11,2,9,14,0,0,0,0,0,5,0,
    4,6,0,0,0,0,0,5,0,0,0,12,8,15,10,
    0,0,10,6,0,11,0,0,0,14,0,1,3,9,0,
    0,15,0,0,12,3,0,6,8,0,9,0,7,0,0,
    7,11,0,0,0,0,0,0,14,10,5,2,0,0,12,
    11,0,0,0,0,0,4,0,10,9,3,8,0,0,0,
    0,8,0,14,10,0,0,13,9,12,0,0,15,0,1,
    0,0,0,7,15,13,3,10,0,0,12,0,6,0,0,
    0,0,0,2,0,0,5,9,1,8,0,0,0,7,14,
    0,7,15,4,2,1,8,0,0,0,0,0,0,13,0,
    2,9,6,0,0,0,0,0,0,0,4,0,10,1,13,
    12,0,2,0,5,0,6,0,0,11,0,15,0,3,0
  };

  const int d15_120_7[] = {
    // Size: 15 x 15
    15,
    // Pre-assigned fields
    10,0,0,0,0,6,4,0,0,5,0,9,0,8,15,
    0,0,11,14,0,12,3,0,0,0,0,0,6,2,8,
    0,14,2,0,0,0,0,0,6,3,0,11,8,0,13,
    9,0,0,2,0,0,8,1,0,0,4,15,11,0,0,
    3,2,0,0,4,0,0,0,0,0,0,10,15,14,12,
    12,0,0,9,5,7,0,0,0,0,6,4,0,1,0,
    0,0,4,0,0,5,15,3,2,11,0,0,9,0,0,
    0,1,0,5,10,0,14,0,0,12,0,0,0,3,9,
    0,0,0,0,0,0,11,5,12,0,1,13,3,7,0,
    1,6,0,0,14,0,0,11,8,0,12,0,0,10,0,
    4,15,0,0,0,2,0,6,0,13,7,0,0,0,1,
    0,11,5,1,8,13,0,0,3,0,0,0,12,0,0,
    0,0,8,3,1,4,0,0,0,14,13,0,0,0,10,
    0,0,3,15,0,0,13,4,5,0,11,8,0,0,0,
    5,8,12,0,7,0,0,10,14,9,0,0,0,0,0
  };

  const int d15_120_8[] = {
    // Size: 15 x 15
    15,
    // Pre-assigned fields
    0,0,5,11,0,6,0,12,0,0,0,0,3,15,9,
    8,0,9,0,0,12,0,0,0,13,5,0,0,0,2,
    7,0,0,8,9,11,3,10,0,0,15,0,0,0,12,
    0,0,1,12,0,7,9,0,0,8,0,2,0,0,10,
    4,0,8,3,0,0,7,0,0,1,0,0,0,13,11,
    0,6,7,0,0,0,11,2,8,0,1,0,0,5,0,
    0,13,0,9,0,0,10,5,6,0,4,7,0,0,0,
    5,0,0,0,2,0,0,13,3,0,0,12,15,7,0,
    0,8,6,2,10,0,0,0,4,0,0,3,1,0,0,
    11,0,0,0,0,13,0,15,0,12,0,6,0,8,14,
    0,12,0,0,14,4,0,0,5,6,7,0,10,0,0,
    0,9,0,0,8,15,0,11,0,3,2,0,0,14,0,
    0,11,0,10,5,0,8,0,9,0,0,1,4,0,0,
    13,15,11,0,0,0,0,0,10,0,0,4,7,9,0,
    15,0,0,0,6,0,5,0,0,11,13,0,2,0,4
  };

  const int d15_120_9[] = {
    // Size: 15 x 15
    15,
    // Pre-assigned fields
    0,0,1,10,3,0,0,14,0,0,0,0,5,8,12,
    9,15,0,8,0,0,0,13,7,0,0,1,0,0,6,
    6,14,0,7,0,3,0,0,0,13,0,0,4,0,11,
    12,10,0,0,0,0,0,0,0,11,3,4,2,0,15,
    0,0,0,13,9,11,10,0,2,0,12,7,0,0,0,
    0,0,10,9,0,8,0,2,11,0,0,0,3,5,0,
    0,0,0,0,14,6,5,0,8,0,13,2,12,0,0,
    0,0,11,0,0,13,3,0,5,8,0,15,0,14,0,
    0,1,0,0,5,0,8,6,0,9,2,0,0,7,0,
    4,7,0,0,0,9,1,0,0,6,0,0,0,15,5,
    0,3,9,0,0,7,0,0,1,12,0,14,0,0,10,
    0,0,15,5,13,0,14,0,3,0,0,12,0,0,1,
    10,0,6,0,0,0,4,8,0,7,14,0,0,2,0,
    15,0,0,0,2,0,0,4,0,0,1,0,7,3,13,
    0,9,2,4,1,0,0,5,0,0,10,0,15,0,0
  };

  const int d20_187_0[] = {
    // Size: 20 x 20
    20,
    // Pre-assigned fields
    10,0,0,2,15,14,16,4,19,0,17,0,0,0,0,0,0,9,18,6,
    0,5,15,19,2,1,12,0,6,0,4,0,7,0,0,0,0,20,13,0,
    0,17,9,0,0,20,14,0,7,8,6,3,0,0,11,12,0,0,0,18,
    18,0,11,8,20,0,0,0,0,17,12,0,6,2,0,16,7,0,0,0,
    0,0,16,6,0,17,3,0,0,0,0,0,19,1,9,18,10,7,0,8,
    14,15,5,0,9,0,0,0,0,16,0,11,1,4,13,0,0,0,3,0,
    6,0,7,17,5,0,18,3,0,14,0,0,10,0,19,2,0,0,0,11,
    0,19,0,0,7,6,8,12,18,0,20,0,0,0,17,0,0,0,5,16,
    0,20,0,0,11,0,0,18,16,0,13,0,4,10,8,9,17,0,0,0,
    16,0,4,0,17,0,11,0,0,20,18,19,5,15,0,0,0,3,0,1,
    15,9,0,0,0,13,0,0,0,3,0,4,14,6,5,0,12,0,10,7,
    11,8,0,3,19,4,0,14,0,15,0,0,9,7,6,0,0,12,0,0,
    0,18,17,20,0,7,0,0,0,5,0,14,0,13,2,10,3,4,0,0,
    5,3,18,0,1,0,0,19,8,0,11,0,0,0,0,14,2,0,7,0,
    20,1,0,0,0,11,6,0,13,19,0,12,0,0,7,0,0,18,16,10,
    0,0,10,11,0,12,4,8,14,9,15,17,0,0,0,6,19,0,0,0,
    0,0,14,18,0,0,15,16,0,0,0,10,12,0,0,19,9,11,17,2,
    0,12,0,9,0,0,10,0,4,0,1,6,0,0,0,17,14,13,19,0,
    0,0,0,14,12,0,0,15,10,0,0,7,0,9,1,0,0,19,8,20,
    4,0,0,0,0,2,0,1,15,12,3,5,13,19,0,0,8,0,0,17
  };

  const int d20_187_10[] = {
    // Size: 20 x 20
    20,
    // Pre-assigned fields
    4,0,1,0,9,15,0,0,5,0,14,12,18,17,0,0,13,0,19,0,
    19,0,0,20,0,11,18,15,0,0,7,8,0,9,0,6,0,10,3,0,
    0,0,10,18,13,0,0,4,14,0,0,0,17,0,3,8,16,0,6,0,
    15,1,0,8,0,0,13,0,12,11,6,0,4,0,0,0,0,3,2,20,
    10,0,11,9,14,5,2,0,0,13,15,0,6,0,0,0,0,20,1,0,
    13,10,0,0,17,0,0,0,9,1,0,0,8,0,20,16,5,0,4,0,
    0,6,0,19,18,0,0,3,10,0,12,0,0,7,16,0,14,0,17,0,
    20,0,0,11,3,19,0,12,8,0,0,16,0,15,0,13,0,0,10,1,
    9,20,0,5,0,0,10,0,0,17,0,0,16,0,8,0,4,6,0,15,
    0,17,3,15,0,2,0,6,7,8,19,0,0,0,5,0,0,18,9,0,
    2,13,0,0,0,4,3,17,0,20,9,0,1,10,0,0,15,0,0,11,
    0,0,0,0,0,13,8,11,4,7,0,1,0,20,6,12,18,0,0,16,
    0,14,4,0,0,0,0,9,20,0,0,7,12,0,13,18,0,0,5,19,
    0,0,18,3,0,20,16,7,0,19,0,2,0,0,10,14,17,8,0,0,
    18,0,13,0,10,7,0,0,3,0,0,15,9,11,0,2,0,12,0,5,
    11,19,0,12,4,14,0,0,18,0,17,0,0,0,0,20,2,5,0,0,
    0,18,12,0,11,0,1,16,0,5,4,17,0,14,19,0,0,7,0,0,
    3,0,19,10,0,17,0,0,0,16,5,11,2,12,0,0,6,0,0,7,
    0,0,14,0,7,0,5,0,0,10,0,9,0,8,15,11,0,17,0,4,
    0,8,16,0,1,0,17,14,0,0,13,20,3,2,0,4,0,0,0,10
  };

  const int d20_187_11[] = {
    // Size: 20 x 20
    20,
    // Pre-assigned fields
    8,0,0,0,9,20,0,14,5,13,0,0,15,0,0,19,11,0,18,17,
    14,0,0,2,1,0,12,0,0,19,10,0,8,0,0,17,0,9,7,16,
    0,0,12,8,0,0,2,17,0,0,18,0,9,5,19,0,0,6,0,3,
    9,11,0,19,4,0,0,0,14,5,0,0,0,0,13,1,12,0,0,7,
    15,0,0,14,12,18,0,0,0,0,0,13,19,4,0,0,8,0,11,10,
    0,14,3,15,0,11,4,0,0,0,0,12,20,8,1,0,10,0,0,13,
    17,0,0,4,2,3,7,13,0,0,0,6,0,10,0,0,0,1,0,19,
    16,0,15,0,0,0,0,20,2,4,8,19,18,0,5,0,0,0,13,6,
    3,8,0,17,19,4,14,0,10,0,7,0,13,0,0,0,16,0,12,0,
    4,7,0,0,0,10,0,6,12,0,0,2,0,9,14,8,15,0,16,0,
    10,0,17,0,0,12,0,0,0,0,19,9,4,0,6,0,1,2,14,11,
    0,0,13,0,0,5,10,11,9,20,0,7,0,19,2,15,0,0,0,12,
    0,3,6,0,0,2,0,0,16,0,13,4,0,0,0,18,7,5,1,0,
    0,18,16,0,3,0,0,8,11,0,4,0,0,13,0,0,5,10,17,0,
    0,9,0,7,20,1,0,2,15,12,16,0,0,14,11,0,0,0,0,4,
    19,20,4,18,11,0,16,10,0,0,0,0,0,6,0,3,0,17,0,0,
    0,19,5,20,0,0,13,12,0,6,0,16,0,11,18,0,2,8,0,0,
    0,0,9,12,0,8,0,15,0,18,2,0,16,0,17,14,0,19,3,0,
    5,0,14,0,7,0,20,0,8,16,0,10,11,0,0,6,3,18,0,0,
    0,4,0,0,18,0,19,0,7,3,12,11,14,0,20,2,0,0,15,0
  };

  const int d20_187_12[] = {
    // Size: 20 x 20
    20,
    // Pre-assigned fields
    0,0,16,0,0,18,13,20,0,9,14,0,0,15,4,0,0,1,0,10,
    0,0,0,5,14,7,0,9,4,8,1,16,13,0,0,0,0,0,19,17,
    0,0,11,4,17,1,0,19,0,20,7,14,10,2,0,0,0,0,0,6,
    0,14,12,16,0,19,0,0,0,0,0,0,3,8,1,11,0,7,4,15,
    18,0,0,11,13,3,5,0,17,0,16,0,14,0,12,8,20,0,0,0,
    0,17,0,10,12,6,3,0,0,0,0,2,18,14,0,16,0,0,9,4,
    11,13,0,0,0,0,0,7,15,0,0,1,0,0,5,14,6,0,18,20,
    4,11,7,15,0,0,12,0,0,0,0,10,0,1,3,6,0,2,8,0,
    12,0,4,0,0,0,16,3,14,0,0,0,20,19,0,9,0,10,15,18,
    0,0,20,0,0,4,0,14,18,1,13,11,2,0,0,0,17,19,0,0,
    10,19,0,0,2,0,1,0,0,16,3,0,0,18,7,0,0,17,12,0,
    0,8,0,1,0,5,7,0,9,15,0,0,11,20,17,0,19,0,0,12,
    1,0,3,0,6,0,0,0,0,0,20,17,12,13,8,0,5,4,0,14,
    16,0,18,6,0,17,11,0,0,3,5,0,0,4,0,10,2,0,7,0,
    5,6,0,0,16,0,0,12,0,0,0,9,0,0,18,3,7,20,2,1,
    0,0,9,0,0,16,17,0,3,13,8,0,0,7,0,4,0,18,10,0,
    8,0,0,13,0,0,9,2,10,17,0,19,7,0,11,0,15,0,0,0,
    6,12,0,17,18,14,0,11,5,0,0,4,0,0,0,19,9,0,0,3,
    0,18,0,9,4,0,15,6,0,12,0,13,0,0,0,0,11,14,3,0,
    0,9,17,0,5,0,0,10,13,0,11,0,1,0,2,7,18,15,0,0
  };

  const int d20_187_13[] = {
    // Size: 20 x 20
    20,
    // Pre-assigned fields
    1,0,8,14,0,0,0,10,0,0,19,7,0,18,0,20,16,0,5,12,
    0,19,10,0,13,7,0,0,0,0,17,12,6,0,9,0,2,18,0,5,
    4,2,0,9,20,0,0,0,13,5,16,0,0,19,0,0,8,11,0,7,
    9,0,7,19,0,8,0,0,0,0,0,0,10,17,0,0,13,20,6,4,
    11,20,0,6,0,0,18,1,16,2,13,0,9,15,0,0,0,0,0,3,
    0,11,0,0,3,9,0,14,0,0,6,0,0,1,0,19,12,17,18,0,
    2,0,0,7,0,13,0,15,0,0,18,16,3,0,14,5,0,0,9,6,
    0,0,9,0,0,0,3,6,0,16,2,0,8,0,4,10,0,0,15,18,
    6,0,1,12,0,16,0,0,15,0,10,8,14,0,5,0,19,0,4,0,
    0,13,0,0,2,6,1,0,11,0,7,4,16,0,19,0,0,15,0,0,
    17,0,12,0,0,19,13,0,18,14,0,0,0,5,0,3,1,2,0,10,
    12,4,0,0,0,15,5,3,8,0,0,1,0,10,0,18,17,0,7,0,
    0,0,0,8,0,14,4,0,0,11,0,18,0,0,20,16,15,12,3,0,
    20,6,18,13,4,0,10,12,0,8,0,0,17,0,0,0,0,3,0,19,
    0,1,0,0,11,20,17,0,6,3,0,9,19,8,18,0,0,0,12,0,
    7,14,15,16,8,0,0,18,0,20,0,0,0,11,10,13,0,0,0,0,
    0,0,19,0,0,0,16,11,7,12,20,3,13,0,0,14,5,10,0,0,
    0,18,14,2,5,0,9,0,19,0,0,11,0,0,3,0,10,7,0,8,
    0,0,0,0,19,0,15,20,9,17,0,0,0,2,16,11,0,13,14,1,
    19,16,3,20,15,0,0,0,17,0,0,14,2,4,0,1,0,0,0,0
  };

  const int d20_187_14[] = {
    // Size: 20 x 20
    20,
    // Pre-assigned fields
    13,11,3,0,0,14,16,7,20,0,0,4,0,0,0,0,0,5,12,0,
    12,0,0,18,2,0,0,0,16,19,0,0,10,5,0,4,14,11,8,0,
    15,2,0,0,7,0,0,0,0,8,0,14,0,11,6,17,16,0,20,0,
    0,0,0,0,0,0,7,15,10,17,16,0,8,0,0,11,6,2,13,0,
    20,7,0,1,16,0,15,19,0,0,0,17,0,10,0,5,0,12,0,11,
    14,6,4,0,0,17,0,0,0,2,9,0,0,0,18,0,10,0,1,5,
    19,10,0,0,0,0,2,14,0,5,0,18,0,17,0,6,1,15,0,20,
    0,14,7,20,15,0,8,0,11,13,0,0,5,12,0,0,0,19,0,0,
    0,0,13,0,20,15,3,0,0,0,18,0,0,19,4,10,17,14,0,12,
    0,18,8,0,4,13,0,0,6,0,3,12,17,16,1,0,19,0,0,0,
    0,15,0,19,17,4,0,9,0,0,1,0,6,0,0,2,0,0,18,3,
    0,0,9,6,19,5,0,2,0,1,0,13,12,14,10,0,0,0,3,0,
    3,0,0,0,0,20,1,18,9,0,19,0,0,6,0,15,11,0,16,17,
    10,3,0,0,8,0,0,4,1,6,15,0,11,0,0,13,0,0,5,2,
    0,5,12,14,0,2,0,0,8,0,10,1,13,0,20,0,0,17,19,0,
    0,0,20,10,14,6,19,0,0,18,17,0,3,0,16,7,0,0,0,4,
    0,0,14,4,0,0,0,8,18,10,0,20,0,0,11,0,7,16,2,19,
    2,0,0,0,0,18,17,11,0,7,14,9,15,0,5,19,0,0,0,10,
    5,13,11,17,0,7,14,0,15,0,0,3,0,1,19,0,0,10,0,0,
    4,0,0,9,0,0,6,13,5,0,2,0,20,0,3,0,18,1,0,0
  };

  const int d20_187_1[] = {
    // Size: 20 x 20
    20,
    // Pre-assigned fields
    0,6,18,0,16,4,17,0,0,9,3,0,0,0,0,2,0,0,19,13,
    0,7,0,13,0,20,0,0,8,16,0,1,18,0,17,14,0,0,4,15,
    6,20,3,0,0,14,0,18,17,4,7,16,0,0,0,13,0,0,0,0,
    0,0,8,15,0,16,12,19,11,0,0,2,0,0,0,18,0,17,0,1,
    1,11,20,17,0,0,3,0,0,6,0,4,0,8,9,0,0,0,16,5,
    3,17,0,0,18,7,10,0,19,0,0,0,16,0,0,8,15,1,0,4,
    8,0,15,0,0,5,0,12,0,14,11,0,10,4,0,0,18,2,9,0,
    5,0,16,0,0,15,0,17,13,1,0,0,7,3,4,0,0,19,0,0,
    15,0,0,7,0,1,0,0,6,13,16,20,0,19,0,0,11,8,0,0,
    0,18,0,0,3,0,0,0,16,0,14,0,19,0,8,4,10,9,5,0,
    0,8,2,19,4,0,9,15,5,12,10,0,14,0,11,0,0,0,0,0,
    9,0,0,14,0,8,2,4,0,0,18,0,11,20,19,0,7,0,17,0,
    0,0,19,0,8,9,0,0,20,0,0,17,15,1,6,0,4,5,0,14,
    0,0,0,12,6,11,0,2,0,5,19,15,0,9,1,0,0,0,8,20,
    0,1,9,4,12,0,0,6,0,0,0,0,20,0,0,11,19,3,7,17,
    0,4,11,18,13,0,15,0,0,20,5,14,0,0,0,16,0,7,2,0,
    17,16,0,0,0,0,0,5,0,0,2,12,0,18,10,9,0,20,6,3,
    11,0,0,6,15,0,19,14,0,0,0,3,12,7,2,0,5,16,0,0,
    16,10,5,8,1,0,4,0,18,0,0,0,9,0,0,12,3,0,0,7,
    0,0,0,0,17,0,6,7,10,0,4,18,0,5,20,0,16,0,15,0
  };

  const int d20_187_2[] = {
    // Size: 20 x 20
    20,
    // Pre-assigned fields
    0,10,6,19,0,0,7,0,18,5,0,0,14,2,0,11,0,0,0,20,
    12,0,1,4,0,7,0,0,2,0,16,0,0,0,11,8,15,0,6,3,
    0,7,19,0,0,0,17,0,0,15,1,0,20,5,0,0,12,16,4,10,
    18,0,10,11,16,0,9,7,0,0,14,2,0,0,15,0,0,0,8,4,
    0,1,5,14,0,17,0,0,9,0,0,0,4,19,7,12,0,3,0,0,
    0,0,0,0,11,0,20,9,10,6,2,5,16,7,0,0,0,0,12,19,
    0,5,0,18,0,11,0,4,0,20,0,8,3,13,0,9,2,0,0,17,
    13,0,9,3,8,15,0,0,6,0,0,18,2,0,4,20,14,0,0,0,
    0,18,0,10,15,5,3,11,0,0,19,0,0,17,2,0,0,0,0,12,
    0,15,17,0,2,19,0,0,0,9,0,3,0,11,8,18,0,0,10,0,
    3,12,13,20,0,8,0,0,0,0,0,17,0,0,0,19,1,11,5,9,
    17,13,0,5,0,18,0,0,11,0,7,4,0,15,6,0,3,10,0,0,
    0,0,0,0,18,6,10,16,20,2,11,13,0,0,5,14,0,12,0,0,
    0,0,0,0,17,0,19,15,0,1,0,0,0,0,3,16,6,14,13,2,
    6,0,0,17,19,0,16,0,0,13,20,0,10,12,0,0,4,15,1,0,
    7,0,20,0,0,13,0,18,16,0,5,0,12,0,0,6,9,2,3,0,
    19,0,8,13,20,0,1,0,0,10,15,12,0,0,0,0,5,0,14,7,
    16,11,18,0,12,0,5,8,13,19,0,14,7,0,0,0,0,6,0,0,
    20,14,0,0,0,0,13,2,4,11,18,0,6,0,17,0,0,19,0,0,
    4,2,0,0,1,14,0,3,0,0,0,0,0,9,10,17,7,0,0,13
  };

  const int d20_187_3[] = {
    // Size: 20 x 20
    20,
    // Pre-assigned fields
    6,0,0,0,0,0,3,13,5,15,9,18,0,16,2,0,0,1,0,0,
    0,13,0,10,8,16,0,0,0,9,0,4,15,0,11,0,20,14,0,5,
    8,0,2,16,0,0,0,11,1,0,15,19,0,0,0,0,6,0,17,18,
    0,3,0,0,12,2,0,10,0,0,19,0,9,4,8,15,11,16,0,0,
    7,0,0,0,11,0,1,5,3,8,0,14,2,0,0,4,13,0,0,20,
    16,0,0,0,0,4,0,12,8,14,0,2,3,11,7,10,0,0,0,9,
    11,12,6,17,0,0,0,0,0,4,10,0,0,5,0,2,0,19,0,15,
    2,17,0,5,3,11,0,0,0,16,0,0,18,0,12,8,1,0,0,0,
    19,9,1,0,0,15,14,0,12,17,6,0,0,0,18,0,4,0,3,0,
    10,20,0,9,0,0,5,16,0,0,8,13,0,7,0,1,0,4,19,0,
    0,0,0,15,19,0,0,0,11,13,7,6,0,12,0,0,10,17,2,4,
    0,15,8,12,9,0,0,18,0,7,13,0,1,0,16,17,0,3,0,0,
    0,10,13,0,0,0,0,6,0,2,0,7,14,18,20,16,0,9,8,0,
    1,0,15,4,0,0,12,8,20,0,0,0,0,10,0,3,9,0,13,16,
    3,0,9,0,20,13,8,0,0,0,0,0,11,19,0,0,14,12,18,2,
    0,0,0,18,17,9,7,0,15,0,12,8,0,0,4,6,0,0,20,0,
    0,0,20,0,5,18,15,4,16,6,0,0,0,3,14,9,0,0,1,0,
    0,0,18,0,10,0,4,9,2,0,0,12,13,20,15,0,0,0,0,19,
    13,16,0,19,7,1,0,0,0,0,0,11,6,0,0,0,17,8,15,14,
    0,8,3,0,0,6,20,0,19,0,18,0,16,0,0,0,15,11,12,0
  };

  const int d20_187_4[] = {
    // Size: 20 x 20
    20,
    // Pre-assigned fields
    0,0,0,0,20,14,12,0,11,2,0,0,9,13,1,0,0,8,19,6,
    15,20,0,0,0,3,0,0,0,9,17,0,4,12,5,14,0,16,13,0,
    0,0,17,10,0,0,2,0,1,0,0,14,8,6,0,19,7,4,0,9,
    13,7,3,0,9,0,4,18,0,5,0,0,0,19,15,0,0,2,0,0,
    16,0,0,20,0,0,19,0,5,12,2,0,6,18,0,0,0,0,9,11,
    0,10,14,11,3,0,0,8,9,19,12,0,0,4,0,15,0,0,5,0,
    0,11,0,14,4,18,0,13,17,3,0,0,0,0,7,6,0,19,0,0,
    14,0,0,0,11,17,0,1,0,0,15,5,20,0,0,4,2,0,8,0,
    6,16,0,15,0,7,0,11,0,0,1,10,12,3,0,0,4,0,0,13,
    0,0,12,1,14,0,13,2,0,0,6,9,17,0,0,0,5,15,20,0,
    0,0,13,0,10,0,15,0,8,16,20,0,19,0,9,7,18,0,0,12,
    0,3,19,9,0,16,14,0,0,10,0,8,5,0,0,0,12,7,0,18,
    1,19,6,0,0,0,9,7,0,0,0,17,14,0,8,0,0,11,16,5,
    12,0,0,4,5,2,17,0,3,0,0,7,0,0,0,13,19,0,15,0,
    17,0,0,0,18,13,0,19,12,6,0,0,0,0,2,5,16,3,0,15,
    19,0,0,17,0,0,0,10,0,18,8,11,0,1,4,0,15,0,3,0,
    4,2,9,5,12,0,0,14,15,0,0,0,0,8,10,18,0,0,0,20,
    0,15,4,0,0,0,16,17,6,0,7,13,0,10,14,20,9,0,0,0,
    0,0,8,0,0,12,0,0,0,4,5,1,16,2,6,0,0,17,11,3,
    0,14,20,16,8,11,1,0,0,0,13,3,0,0,0,0,10,0,2,0
  };

  const int d20_187_5[] = {
    // Size: 20 x 20
    20,
    // Pre-assigned fields
    18,6,3,13,0,7,20,8,0,0,11,0,0,0,0,14,0,15,0,0,
    0,0,11,0,5,3,0,17,1,0,0,12,2,7,0,6,18,0,0,20,
    0,0,4,0,19,12,9,15,0,10,0,0,14,0,11,8,6,13,0,0,
    20,0,2,0,0,0,0,0,10,4,17,8,0,0,14,3,5,12,0,16,
    0,8,0,2,0,11,0,19,0,12,3,14,15,16,0,0,0,0,20,17,
    2,4,0,0,11,0,6,0,5,0,0,0,0,1,10,17,0,19,3,0,
    6,12,10,3,0,16,0,0,0,19,15,2,0,13,5,0,0,0,14,0,
    0,0,1,0,0,8,0,10,0,13,6,7,19,17,0,0,3,0,0,14,
    0,0,0,5,20,9,0,7,0,16,0,0,13,0,18,0,10,0,15,8,
    7,0,0,11,6,19,18,5,0,0,20,17,0,0,0,10,9,0,1,0,
    0,1,0,0,4,0,0,0,8,11,7,0,0,5,17,16,0,20,12,9,
    3,0,9,8,0,6,0,0,13,0,14,0,0,18,20,15,1,0,0,0,
    13,10,0,0,12,0,2,0,14,17,0,11,7,20,8,0,0,0,9,0,
    0,3,15,0,13,0,4,6,0,14,8,19,0,0,0,20,0,10,16,0,
    8,7,17,0,0,0,0,0,9,0,0,18,0,15,3,5,19,14,10,0,
    4,17,0,10,7,0,0,1,0,3,9,16,18,0,0,0,11,0,0,12,
    0,0,13,0,0,0,14,20,15,6,0,0,3,8,16,0,0,5,0,4,
    16,0,14,17,3,0,12,0,18,0,0,0,0,19,6,0,7,2,0,10,
    0,16,0,14,17,0,1,0,0,0,10,0,8,0,0,0,20,3,4,7,
    1,2,0,12,0,10,19,4,17,0,0,3,11,0,0,7,0,0,5,0
  };

  const int d20_187_6[] = {
    // Size: 20 x 20
    20,
    // Pre-assigned fields
    1,0,0,15,3,0,0,7,6,0,0,17,13,0,4,0,20,0,19,0,
    0,7,0,0,20,2,12,0,0,0,0,9,6,18,13,0,11,0,8,14,
    18,0,16,0,11,0,0,0,1,0,10,20,15,13,14,0,0,0,4,3,
    0,0,6,17,4,5,0,8,16,11,0,0,0,3,0,1,9,2,0,0,
    0,5,11,4,12,15,0,0,2,3,17,0,20,0,0,7,0,0,0,0,
    15,9,12,16,0,0,0,19,0,0,0,0,18,14,3,2,0,1,0,7,
    0,4,0,0,7,18,0,1,0,6,0,2,3,0,0,16,0,17,10,15,
    0,15,10,0,0,4,18,0,0,1,11,0,0,0,8,5,12,0,17,0,
    13,2,19,0,0,12,9,0,0,16,20,0,0,15,6,0,0,11,0,8,
    0,12,3,14,0,17,2,6,0,13,0,0,0,19,0,0,0,4,16,11,
    9,8,20,0,0,0,4,0,7,0,18,1,0,5,0,12,10,3,0,0,
    0,14,0,6,13,0,17,0,11,0,0,19,5,8,0,0,3,0,0,4,
    20,0,18,0,0,0,3,0,8,17,1,16,4,0,0,11,0,0,0,19,
    7,0,0,20,0,14,0,18,0,9,0,10,0,11,16,0,13,15,0,2,
    0,0,0,19,5,20,0,9,0,14,2,18,0,4,7,0,0,0,12,10,
    12,0,0,10,0,0,0,2,14,0,4,8,19,0,0,13,16,5,0,0,
    0,16,0,3,19,0,0,5,18,0,15,0,0,6,0,10,0,14,7,20,
    2,0,14,0,0,0,11,10,0,12,6,0,7,0,0,19,15,8,18,0,
    11,19,0,0,0,16,20,0,0,7,0,0,14,0,12,17,8,9,13,0,
    14,0,0,0,6,10,1,17,15,0,0,12,0,0,20,0,5,0,2,0
  };

  const int d20_187_7[] = {
    // Size: 20 x 20
    20,
    // Pre-assigned fields
    3,10,14,5,0,0,0,7,0,0,9,0,1,6,13,8,0,11,0,0,
    18,0,9,20,0,14,0,10,0,0,0,7,4,0,0,0,12,0,2,15,
    0,1,0,0,12,19,0,0,13,4,0,0,0,3,10,0,14,7,15,11,
    0,14,0,0,9,0,5,0,0,15,8,0,0,12,4,18,13,1,10,0,
    11,17,18,0,1,0,0,2,19,0,0,0,13,5,0,9,6,0,0,0,
    0,11,0,1,0,0,8,4,15,0,5,0,2,17,0,0,0,13,14,6,
    1,0,0,4,6,0,17,0,10,7,19,0,9,8,0,16,2,0,0,0,
    14,0,12,19,0,10,0,0,8,20,0,0,17,0,0,4,15,6,18,0,
    4,0,11,6,0,16,2,5,0,0,0,15,0,0,0,0,8,17,13,20,
    16,20,0,13,19,3,1,0,12,8,18,17,0,0,0,7,0,0,0,0,
    0,0,0,17,0,0,3,0,4,9,11,0,0,14,0,15,0,16,5,18,
    10,2,5,0,0,0,15,20,16,17,0,19,3,13,0,1,0,0,0,0,
    0,0,16,0,0,13,0,11,0,3,14,10,7,15,0,0,5,19,0,12,
    0,0,0,18,5,17,12,0,0,13,2,0,0,16,6,3,20,0,0,7,
    15,0,6,3,0,0,13,1,0,10,0,20,16,0,11,0,0,8,7,0,
    0,9,17,0,13,0,0,0,0,0,7,0,18,19,5,6,11,0,0,4,
    0,16,3,0,11,8,7,6,0,0,0,4,0,0,17,0,0,9,20,0,
    0,0,0,16,3,20,9,17,0,0,0,1,0,0,12,19,0,0,8,5,
    19,8,0,0,14,4,0,12,5,1,0,3,0,0,7,0,0,15,0,9,
    13,18,2,0,7,0,0,0,1,0,10,5,15,0,14,0,3,0,0,0
  };

  const int d20_187_8[] = {
    // Size: 20 x 20
    20,
    // Pre-assigned fields
    7,16,8,19,0,0,0,5,4,0,0,0,11,3,12,9,0,0,0,18,
    0,0,7,18,0,0,0,20,0,3,0,8,10,9,0,0,13,14,1,17,
    18,0,13,0,9,0,8,4,11,0,0,0,3,12,0,5,16,6,0,0,
    14,3,2,9,0,0,0,10,0,0,0,4,0,0,1,0,11,18,17,6,
    0,0,12,8,0,0,9,3,1,0,0,15,13,19,14,6,0,0,0,0,
    4,0,18,0,13,0,6,14,7,0,0,12,16,0,20,3,0,0,8,0,
    0,13,0,15,0,3,10,0,8,0,5,0,0,0,9,0,4,20,7,1,
    10,19,3,0,0,9,0,0,14,2,7,18,0,4,17,0,0,0,0,20,
    0,0,10,0,12,18,0,7,16,17,0,0,0,0,0,13,6,2,19,11,
    6,18,0,11,0,0,0,12,0,0,0,13,17,15,2,7,0,0,20,5,
    0,4,5,2,15,12,14,0,0,0,3,0,0,18,0,1,19,7,0,0,
    0,2,0,20,0,4,0,0,17,7,13,0,5,16,11,0,15,0,9,0,
    9,6,0,12,5,13,15,0,0,20,4,0,0,0,0,0,7,0,14,0,
    11,0,6,0,4,8,17,0,0,9,16,14,1,2,0,19,0,0,0,0,
    20,0,0,14,0,17,0,8,10,0,12,1,0,0,0,0,9,11,3,19,
    0,8,0,10,14,0,2,9,0,12,0,3,0,0,16,0,0,0,5,4,
    0,0,0,0,17,6,7,18,0,10,15,0,19,5,0,0,0,1,13,0,
    5,0,17,0,0,0,13,0,0,14,1,20,8,0,0,2,12,0,0,3,
    17,0,0,0,20,0,5,0,18,1,0,7,15,0,0,8,14,13,0,0,
    0,9,0,0,1,16,0,0,2,0,20,0,0,7,15,14,0,4,0,8
  };

  const int d20_187_9[] = {
    // Size: 20 x 20
    20,
    // Pre-assigned fields
    0,6,0,0,1,0,12,0,5,7,19,0,0,15,9,0,17,0,14,0,
    4,0,0,2,7,0,19,0,0,0,18,6,8,5,0,12,0,0,20,0,
    5,0,16,18,0,8,0,13,0,12,14,0,0,0,10,0,0,11,9,0,
    0,9,0,0,17,16,6,0,0,0,0,11,5,3,8,0,0,10,12,4,
    0,10,2,11,0,18,0,3,8,0,0,1,0,0,15,0,9,5,0,16,
    0,5,0,10,9,0,0,17,7,0,3,18,11,16,0,2,0,0,0,15,
    13,3,20,0,11,0,0,14,0,5,0,2,0,8,0,0,15,4,0,0,
    1,0,0,0,0,20,11,12,4,0,2,0,9,19,14,0,18,0,17,0,
    0,14,19,0,13,6,0,0,0,15,0,0,0,7,20,4,3,1,0,8,
    0,2,0,12,0,7,5,0,0,3,0,15,14,0,16,0,19,20,1,0,
    6,4,0,0,12,0,10,0,0,0,11,17,18,0,0,20,5,0,0,3,
    0,0,12,15,0,0,14,7,11,17,0,0,4,0,0,0,20,8,6,1,
    2,0,4,13,0,0,0,5,1,10,17,0,16,18,0,19,0,0,0,9,
    10,0,14,0,0,2,18,20,0,6,9,0,0,0,17,8,0,3,4,0,
    0,0,17,9,0,11,8,15,3,13,4,0,0,0,18,10,2,0,0,0,
    0,19,6,4,0,5,15,0,9,8,0,0,20,0,0,7,0,0,11,0,
    0,0,15,1,20,0,0,16,17,0,0,7,10,13,0,0,6,0,18,2,
    9,7,0,0,8,0,13,0,6,0,1,4,0,0,11,3,0,12,0,0,
    15,0,1,16,4,14,0,19,0,0,6,12,0,9,0,13,0,0,0,5,
    11,15,0,0,0,1,0,0,16,0,0,9,0,0,12,14,10,2,5,20
  };

  const int d25_264_0[] = {
    // Size: 25 x 25
    25,
    // Pre-assigned fields
    6,13,17,0,0,15,10,0,0,4,7,3,0,2,0,0,22,25,14,0,0,1,0,18,19,
    22,0,7,4,0,0,0,0,14,17,0,25,0,0,0,0,15,20,3,1,11,0,12,5,13,
    0,4,12,0,18,24,17,0,2,11,20,19,0,0,15,0,0,0,0,8,22,25,0,10,16,
    20,0,15,19,0,0,5,1,10,2,0,11,0,24,25,14,16,21,0,0,13,0,0,0,18,
    2,9,14,0,22,0,0,20,23,24,0,0,0,25,16,4,0,12,17,10,6,0,0,11,0,
    3,2,11,20,7,0,6,5,12,0,23,0,18,0,0,0,0,0,0,17,0,24,16,0,4,
    0,0,0,6,12,0,7,13,0,0,0,16,0,14,22,5,25,0,8,2,19,21,0,9,10,
    14,8,0,5,0,7,0,22,13,0,3,0,23,0,19,0,21,0,0,16,0,11,1,0,17,
    0,20,9,12,8,11,1,7,4,0,0,13,0,0,10,0,0,0,6,0,5,0,21,2,0,
    0,0,0,0,0,0,0,0,18,1,6,0,25,0,8,20,4,14,21,3,15,9,22,19,2,
    0,7,0,0,1,0,21,0,0,0,0,5,0,0,11,16,2,10,0,13,14,17,15,8,22,
    1,11,0,0,15,17,23,0,0,3,8,22,0,21,7,24,13,0,10,6,9,0,0,0,0,
    11,0,4,0,5,21,2,24,0,18,19,6,0,13,0,1,0,9,0,14,25,0,0,0,0,
    5,1,22,11,23,16,0,0,0,0,0,15,0,3,0,9,20,0,25,0,0,2,0,14,8,
    12,0,24,0,3,6,0,10,1,16,25,0,13,0,0,21,8,23,0,5,18,22,0,0,0,
    8,0,21,0,0,5,14,0,0,15,17,0,20,11,0,0,6,0,18,22,24,4,2,1,0,
    0,22,0,0,14,20,16,0,7,12,13,4,21,18,23,11,0,0,0,0,0,0,9,24,0,
    0,0,0,15,19,0,0,16,22,0,0,14,11,20,0,8,12,4,0,0,17,3,25,0,21,
    16,0,0,8,20,0,0,23,0,13,2,0,12,19,14,22,0,17,0,0,3,0,0,7,9,
    7,6,0,14,0,4,15,3,9,21,0,0,10,17,20,0,0,0,12,18,0,0,19,0,0,
    10,24,0,25,0,18,0,0,15,19,0,0,2,0,5,13,0,8,23,0,0,12,6,0,3,
    0,23,8,0,0,1,0,21,6,0,9,24,16,5,13,0,14,11,0,0,0,19,0,4,0,
    0,25,19,23,16,0,4,12,0,0,21,0,7,8,3,0,18,13,2,0,10,0,5,0,0,
    4,0,3,17,0,8,25,0,0,0,11,0,1,6,0,23,0,16,19,15,0,5,7,0,24,
    0,0,0,13,0,25,0,11,5,20,1,18,8,0,0,10,0,2,4,19,0,0,24,21,0
  };

  const int d25_264_10[] = {
    // Size: 25 x 25
    25,
    // Pre-assigned fields
    0,24,0,0,4,5,0,21,0,20,9,19,1,0,0,6,0,0,0,16,0,25,3,8,12,
    22,17,25,0,0,2,19,0,20,5,1,16,0,18,6,23,0,0,0,24,8,0,15,0,0,
    0,22,0,0,0,16,0,17,24,7,10,0,25,15,0,0,21,1,13,8,0,19,0,18,23,
    10,4,0,15,9,0,12,7,0,0,3,13,0,0,11,17,25,0,21,0,0,2,18,0,22,
    19,5,10,0,18,23,0,0,13,16,0,6,0,17,0,0,2,21,8,7,0,0,0,9,0,
    0,0,8,19,2,0,24,3,11,0,0,0,4,0,16,5,0,23,0,12,0,18,7,0,20,
    0,14,0,25,17,9,0,0,0,10,0,8,12,0,0,18,22,0,6,3,4,16,0,0,13,
    1,0,14,0,0,0,20,0,0,17,4,15,5,13,23,8,0,3,0,0,11,0,9,16,2,
    8,20,0,17,0,0,0,11,18,0,0,23,0,4,0,16,7,10,24,0,0,1,13,3,6,
    21,15,3,0,0,0,6,2,16,0,17,0,0,9,0,0,12,25,0,0,14,7,0,24,10,
    0,19,0,24,11,20,4,0,15,0,7,12,13,16,0,9,18,0,25,0,0,22,0,2,0,
    6,0,4,0,0,0,1,0,22,25,20,0,8,0,14,11,5,0,0,0,19,10,12,17,0,
    3,0,20,1,7,10,0,15,0,11,0,0,19,0,0,0,0,2,0,14,5,21,24,6,9,
    0,0,11,12,24,0,22,23,19,0,25,2,9,14,18,0,17,0,16,1,0,0,0,0,8,
    16,0,0,21,22,4,0,0,3,8,0,0,0,1,5,10,6,0,23,0,18,17,0,20,19,
    25,21,7,14,0,0,16,10,0,9,6,20,0,0,15,0,0,24,4,0,1,0,19,0,0,
    0,2,0,20,0,0,14,6,0,12,19,0,11,0,10,7,8,17,3,0,9,0,0,25,0,
    0,0,17,5,0,14,3,0,7,18,0,22,0,10,20,0,15,0,0,11,0,0,6,12,24,
    0,0,1,0,0,8,0,19,17,3,16,4,0,21,12,20,24,14,0,0,13,0,22,0,0,
    18,0,0,16,1,6,0,0,4,0,0,10,14,23,24,0,0,12,15,25,21,0,5,0,0,
    4,0,16,23,10,0,0,0,0,0,2,0,15,0,25,24,0,18,22,0,7,14,0,13,21,
    23,12,13,0,0,22,10,0,14,0,8,7,2,5,0,0,19,0,17,20,25,0,21,0,0,
    0,1,0,2,23,15,0,5,25,4,21,24,0,3,22,0,0,6,0,13,0,12,0,0,0,
    7,16,18,0,12,0,17,8,0,0,0,0,0,0,0,4,11,22,0,6,20,9,10,21,25,
    0,0,9,0,20,25,8,18,0,0,0,0,21,2,19,15,0,0,5,10,23,3,0,22,0
  };

  const int d25_264_11[] = {
    // Size: 25 x 25
    25,
    // Pre-assigned fields
    0,7,1,23,18,22,0,0,19,0,12,25,0,16,0,4,8,0,0,0,5,0,13,21,0,
    15,14,0,0,0,10,2,22,4,19,7,0,9,13,20,18,5,0,0,0,0,0,23,0,0,
    0,1,3,0,20,18,0,19,0,8,2,0,15,0,0,25,24,11,6,22,0,9,0,23,0,
    22,0,14,0,0,3,18,13,0,9,1,0,12,0,0,17,0,25,19,7,0,20,24,2,0,
    2,16,20,4,5,14,11,0,21,15,0,0,17,19,0,0,13,0,0,1,0,7,0,0,0,
    0,10,6,25,0,0,8,17,5,0,20,0,0,0,1,2,16,15,0,0,0,0,7,9,23,
    0,9,4,8,25,0,19,24,0,18,6,7,0,17,22,0,11,0,0,13,16,0,12,0,0,
    1,0,0,0,0,2,7,0,3,0,0,13,0,15,8,20,0,14,23,9,12,11,4,5,0,
    9,21,25,0,10,16,20,0,0,0,0,11,0,0,24,0,6,19,14,4,13,0,22,0,0,
    23,0,0,16,0,17,1,14,10,12,0,15,18,20,0,0,0,0,0,2,21,6,0,0,13,
    24,0,0,2,13,8,25,0,0,0,0,17,0,4,18,9,0,20,0,0,0,19,11,6,16,
    0,0,0,20,9,6,0,7,15,0,3,14,0,11,0,8,10,0,5,21,0,13,19,0,1,
    0,0,0,3,2,0,4,9,14,0,17,16,0,0,21,24,0,7,22,0,6,0,0,8,11,
    10,0,0,0,0,0,12,21,16,0,0,2,7,0,0,3,22,23,4,0,8,0,25,20,19,
    19,0,2,0,23,0,22,0,20,10,24,6,11,0,0,12,0,0,0,3,0,5,0,13,4,
    0,0,13,11,15,0,0,4,0,7,16,0,0,3,14,0,19,1,0,0,22,2,0,25,21,
    0,18,7,6,17,12,0,0,0,4,0,8,22,2,5,0,0,13,16,10,0,25,0,0,0,
    11,6,0,0,0,15,0,2,0,0,18,0,0,0,10,1,25,24,0,5,23,3,9,22,17,
    16,0,0,0,0,0,10,1,22,0,0,0,21,25,23,0,12,8,13,0,2,18,3,0,5,
    13,3,9,19,0,11,0,5,18,22,0,23,14,0,2,16,0,0,8,0,7,0,0,0,0,
    3,0,5,0,1,0,0,18,0,11,0,10,13,14,7,0,0,4,21,15,17,0,0,12,20,
    0,4,22,7,0,0,17,0,0,24,0,18,0,9,13,6,2,0,15,25,0,14,10,0,8,
    0,8,0,22,21,0,0,0,13,3,11,20,16,1,0,0,0,12,9,0,14,0,18,10,25,
    20,5,0,0,16,13,0,15,24,2,22,0,1,0,3,0,18,0,0,0,19,17,0,7,9,
    25,15,8,17,3,24,9,0,0,14,23,0,10,0,0,5,0,18,12,16,0,0,0,19,0
  };

  const int d25_264_12[] = {
    // Size: 25 x 25
    25,
    // Pre-assigned fields
    11,0,0,22,18,0,0,5,0,8,3,0,4,2,23,0,15,0,25,20,0,14,19,10,0,
    3,8,22,20,0,0,2,0,17,1,13,12,23,18,0,0,4,0,0,0,10,5,16,0,0,
    24,9,21,0,15,3,0,0,10,0,17,0,13,12,0,6,0,0,22,2,0,23,20,0,0,
    0,0,0,23,20,25,6,1,16,19,0,14,0,0,18,8,24,0,21,0,12,0,3,0,13,
    0,1,24,0,13,12,18,0,11,0,0,3,0,0,0,0,0,25,0,6,14,10,4,2,23,
    16,24,0,12,8,19,0,21,0,6,0,17,5,0,14,0,0,13,15,0,0,0,23,0,20,
    0,0,5,6,0,0,23,0,13,0,4,15,10,0,0,0,14,0,19,24,16,12,0,25,18,
    0,0,16,0,14,0,11,0,22,0,24,0,1,8,21,3,20,19,0,13,7,2,0,0,25,
    5,13,3,0,16,2,17,0,25,23,0,0,19,6,0,21,7,0,0,0,0,0,0,14,9,
    23,0,15,0,12,14,20,0,0,2,0,24,18,0,0,0,0,4,11,8,5,6,21,0,0,
    21,0,0,0,3,15,7,17,0,0,0,0,0,0,19,14,11,8,12,0,20,16,13,0,2,
    19,16,14,0,0,0,0,20,0,24,21,7,0,23,13,15,3,6,9,0,0,25,0,0,17,
    14,4,23,5,10,0,0,18,3,0,0,22,0,0,17,9,8,0,16,0,0,15,0,19,1,
    6,0,12,15,1,16,0,19,20,4,0,13,21,0,0,0,0,0,10,5,17,0,0,0,11,
    0,12,25,0,6,11,0,0,15,3,10,2,22,16,8,0,0,5,0,7,21,0,0,13,0,
    12,5,18,0,0,1,0,0,0,0,14,10,0,0,4,11,22,0,0,15,8,24,2,7,0,
    0,0,0,10,7,4,1,13,0,18,0,0,0,15,6,24,23,22,2,0,25,0,0,20,0,
    0,18,0,3,0,24,0,4,7,16,0,19,6,13,5,12,0,0,0,0,0,1,10,15,22,
    0,6,0,8,0,18,0,22,12,0,16,0,2,21,11,0,0,0,5,10,0,7,1,23,0,
    0,0,2,0,0,0,5,25,1,10,22,0,0,7,9,23,21,12,0,14,0,0,17,8,0,
    0,25,0,0,0,0,0,11,0,14,7,0,0,17,24,18,16,10,3,21,2,0,12,1,15,
    0,22,0,14,17,21,12,0,0,0,0,9,0,5,0,20,0,24,0,0,3,19,7,18,16,
    13,0,0,25,0,0,24,6,0,0,8,5,17,10,0,16,0,11,0,22,15,0,0,21,14,
    10,19,13,11,23,0,3,0,14,0,12,8,15,25,0,0,0,20,0,4,0,22,0,0,21,
    15,21,19,2,0,0,14,8,0,9,5,0,0,0,16,4,1,18,13,25,22,0,0,0,0
  };

  const int d25_264_13[] = {
    // Size: 25 x 25
    25,
    // Pre-assigned fields
    17,15,0,6,0,4,0,23,0,8,0,0,1,21,16,0,12,2,10,18,0,0,0,24,0,
    9,0,4,3,22,6,0,21,17,10,0,24,15,0,0,13,0,1,0,11,20,7,0,0,0,
    0,11,13,0,10,21,18,0,0,15,0,7,0,16,0,17,0,5,0,0,2,25,0,3,1,
    23,16,9,0,0,0,24,14,0,0,0,11,2,7,0,20,8,6,0,12,17,0,5,0,0,
    0,25,22,1,6,5,15,0,0,11,2,12,0,3,0,8,0,16,17,20,0,0,0,0,10,
    21,13,0,0,0,2,0,0,6,7,0,0,20,18,0,0,17,24,1,0,0,16,14,23,25,
    20,2,12,0,0,25,0,0,21,0,0,5,11,17,3,15,23,0,0,19,0,24,9,0,0,
    24,0,0,0,23,10,0,2,4,0,0,18,16,9,0,0,0,3,11,1,5,21,8,15,0,
    0,12,0,0,0,7,6,20,15,0,0,0,23,0,9,25,13,19,22,14,0,0,16,0,21,
    7,4,6,0,0,14,13,9,0,19,16,0,0,0,15,0,22,0,0,21,0,17,0,5,11,
    2,0,25,0,0,23,3,8,18,0,17,0,0,0,0,22,0,9,0,10,24,20,15,7,12,
    18,10,8,0,16,0,0,0,12,6,3,0,0,0,19,0,11,23,15,7,13,0,22,0,0,
    5,0,17,24,9,0,16,19,0,22,4,25,21,0,7,10,0,0,0,0,3,0,6,0,0,
    14,0,23,11,12,0,4,0,1,17,20,0,24,2,0,0,0,22,19,0,18,0,0,0,7,
    11,0,0,16,1,0,0,0,7,0,22,23,9,0,0,21,18,0,3,13,0,15,4,6,5,
    0,0,0,4,25,1,11,0,0,18,12,0,0,22,14,5,2,17,20,6,19,0,0,9,0,
    12,17,11,7,14,19,20,0,0,0,15,0,0,25,0,6,0,13,24,0,10,2,0,4,0,
    0,0,0,25,3,0,0,0,14,12,0,17,13,8,20,18,9,0,2,0,0,0,1,11,19,
    0,22,2,13,18,0,0,7,19,25,14,0,4,0,21,0,10,0,5,0,0,23,0,20,8,
    0,20,0,17,8,3,5,24,0,0,21,0,0,12,25,9,0,0,14,0,22,0,18,10,6,
    0,0,0,18,0,0,17,1,3,24,0,14,0,13,5,12,0,8,9,0,25,11,0,21,4,
    19,14,0,12,7,0,0,25,9,0,0,22,17,0,6,0,0,0,0,0,15,10,2,16,18,
    0,0,0,10,17,0,22,5,24,3,1,15,0,0,23,0,21,0,6,0,9,13,0,19,20,
    6,7,5,0,0,12,23,22,8,4,25,16,0,0,2,0,24,0,0,3,0,0,21,0,0,
    0,0,3,0,0,11,0,0,0,0,19,20,8,6,17,23,7,12,0,25,14,5,24,0,0
  };

  const int d25_264_14[] = {
    // Size: 25 x 25
    25,
    // Pre-assigned fields
    0,0,11,0,0,18,22,0,24,0,23,0,0,10,17,3,4,6,8,20,2,0,9,25,0,
    9,0,13,24,10,20,19,0,0,0,0,0,11,0,25,16,17,0,18,21,0,12,3,0,0,
    0,5,0,15,7,0,1,16,13,0,0,17,24,0,22,12,0,2,0,0,0,9,8,19,11,
    3,11,0,18,0,0,15,5,23,0,12,0,16,0,0,25,14,4,0,24,13,6,0,0,8,
    22,0,8,0,4,0,0,0,0,18,0,0,2,0,21,24,1,25,12,19,3,0,20,5,0,
    8,16,6,0,18,2,11,0,0,0,13,7,0,24,12,22,0,21,0,25,23,19,0,0,0,
    0,9,0,22,11,0,8,0,20,17,15,4,0,0,10,5,0,0,0,0,18,14,1,6,0,
    0,0,10,0,0,17,0,9,0,7,0,11,23,20,15,2,0,0,22,18,19,0,13,0,5,
    24,13,0,0,0,15,0,10,17,0,20,21,18,5,4,0,25,0,0,0,0,3,0,11,2,
    14,15,23,0,0,0,0,0,4,13,24,0,0,9,0,0,19,10,0,17,11,25,12,22,0,
    0,0,0,11,2,23,0,1,15,0,3,16,9,0,0,18,0,24,0,12,25,21,0,20,6,
    25,21,1,0,17,5,14,0,8,0,11,0,6,0,0,20,0,0,15,10,0,0,18,4,7,
    13,0,0,4,0,3,9,0,21,0,0,2,17,0,14,0,6,0,11,0,20,7,19,1,0,
    4,0,0,9,5,1,10,23,0,14,0,0,0,15,0,0,11,13,20,0,21,0,7,17,12,
    5,2,0,7,0,8,6,25,12,16,9,18,0,1,0,0,21,0,17,23,0,0,15,0,0,
    0,24,0,0,15,10,0,20,0,25,8,1,5,0,19,4,0,14,0,9,6,0,0,0,13,
    21,0,7,3,0,0,18,24,1,15,0,0,0,19,0,0,23,16,0,6,10,13,0,0,17,
    0,0,22,10,24,0,0,0,0,12,0,20,3,6,9,13,0,11,0,14,16,1,21,0,0,
    15,0,12,19,25,0,0,0,0,0,0,6,10,16,18,0,22,9,13,0,8,2,14,0,1,
    0,18,21,0,9,0,2,0,25,1,7,24,22,17,0,23,20,0,0,16,0,0,0,8,0,
    0,0,0,8,6,0,0,2,11,20,18,15,19,25,0,0,5,0,16,0,4,24,0,23,0,
    6,4,0,5,0,0,13,8,0,22,1,9,0,0,0,0,2,18,24,0,0,20,23,16,25,
    0,6,3,0,0,19,23,13,22,8,16,12,0,21,7,17,0,0,9,0,0,0,2,0,15,
    0,22,2,20,14,9,0,6,7,19,0,0,8,4,0,0,0,17,10,0,0,0,0,3,21,
    2,19,16,0,1,7,17,15,0,5,0,0,0,18,13,0,0,8,14,0,0,4,0,0,3
  };

  const int d25_264_1[] = {
    // Size: 25 x 25
    25,
    // Pre-assigned fields
    7,0,0,10,0,5,9,23,17,0,0,12,18,0,4,15,19,0,0,13,22,0,0,20,24,
    0,0,0,21,1,8,14,22,0,0,10,6,7,9,17,25,0,0,13,0,0,16,15,0,5,
    14,24,3,18,0,9,0,0,0,17,6,10,0,0,23,0,22,19,15,0,16,7,0,0,1,
    0,0,0,3,8,22,10,9,0,0,0,24,20,0,0,21,0,0,4,17,5,1,6,13,0,
    18,0,8,0,7,11,0,20,0,0,21,17,0,23,0,1,0,0,10,5,0,14,13,2,19,
    15,0,5,0,0,6,23,19,0,0,0,20,24,14,12,0,7,18,22,0,25,2,0,0,0,
    2,9,18,25,13,0,0,7,5,0,0,0,0,0,8,0,14,6,0,22,1,3,0,17,0,
    0,16,0,0,11,4,19,0,1,13,0,5,15,0,0,23,24,0,0,0,3,0,9,6,7,
    8,10,4,22,3,0,0,0,0,21,5,15,0,20,0,6,11,23,0,14,24,0,0,12,0,
    13,21,25,0,0,0,0,17,2,7,1,14,0,18,19,20,4,0,0,0,0,8,24,0,22,
    0,0,20,14,12,0,13,10,11,24,0,0,23,3,0,17,0,0,7,9,0,21,4,0,18,
    9,25,15,0,0,1,0,0,0,23,16,0,17,22,10,3,12,0,5,0,21,6,0,0,0,
    6,5,2,23,0,21,0,0,0,0,0,3,4,1,18,0,0,10,24,11,0,0,17,0,9,
    12,22,0,0,0,0,5,18,4,10,0,0,19,25,13,0,6,14,0,23,2,0,0,16,0,
    0,14,21,2,25,0,0,11,6,3,4,0,9,0,24,0,0,8,0,0,19,0,16,18,15,
    16,17,11,4,0,0,22,24,9,0,2,0,0,8,0,0,3,0,0,10,23,25,0,0,12,
    0,12,0,0,23,13,2,25,15,16,14,4,0,24,0,10,0,0,11,8,0,0,0,1,0,
    1,0,19,0,6,0,12,0,20,11,7,0,25,17,5,24,0,15,0,0,10,0,0,0,8,
    0,7,22,24,0,0,0,13,19,12,0,2,1,0,0,0,9,3,20,16,0,15,25,0,11,
    0,0,14,0,4,24,0,0,8,0,11,18,0,19,0,5,17,21,16,0,0,13,2,9,0,
    0,0,0,15,0,0,21,8,0,18,3,7,11,5,0,0,0,1,0,4,13,9,10,14,0,
    3,0,0,0,21,18,8,0,0,22,9,0,5,0,16,0,0,11,19,0,17,0,20,7,23,
    17,6,0,7,20,0,1,0,3,25,19,23,0,16,0,0,0,9,21,15,18,0,0,10,0,
    25,8,0,5,0,20,4,0,21,0,12,0,2,0,15,13,18,0,17,0,0,0,1,11,16,
    0,0,0,16,18,15,3,6,0,2,0,0,0,0,14,7,5,24,0,20,0,19,12,23,0
  };

  const int d25_264_2[] = {
    // Size: 25 x 25
    25,
    // Pre-assigned fields
    0,0,0,0,0,15,0,16,0,2,0,12,23,0,21,19,6,13,24,10,0,0,9,7,5,
    22,0,0,7,18,21,9,2,0,3,1,0,0,11,0,0,0,10,6,13,15,19,0,0,0,
    4,13,0,10,0,22,17,0,18,0,0,14,8,0,0,0,15,7,21,0,0,23,12,1,3,
    11,9,0,0,15,0,0,0,0,22,0,7,18,25,0,0,4,1,17,12,20,16,0,14,8,
    0,5,24,14,0,3,0,10,0,18,17,15,0,0,7,0,13,21,0,1,22,0,4,11,0,
    9,18,10,13,0,1,0,7,11,14,20,0,0,23,15,8,3,0,0,0,0,22,0,0,0,
    0,25,21,0,3,0,24,18,16,0,0,5,13,0,2,20,0,0,0,8,17,0,0,4,12,
    13,3,20,0,2,0,0,0,6,9,12,0,25,19,0,0,18,0,4,0,21,0,0,16,17,
    18,21,13,0,16,17,10,20,0,1,7,0,0,0,19,4,25,0,0,0,8,0,0,15,22,
    0,0,0,3,5,0,13,0,8,21,10,0,0,0,0,25,14,12,15,24,16,0,7,0,1,
    0,8,0,0,11,19,0,0,9,15,23,21,5,2,0,0,0,24,14,4,0,20,10,18,0,
    16,17,12,15,6,0,7,3,0,0,0,0,0,13,4,0,0,8,25,11,0,24,22,0,0,
    0,0,22,0,21,0,5,0,1,0,16,4,6,18,23,17,0,25,0,7,0,10,0,0,20,
    15,2,0,0,7,16,0,0,0,23,0,9,0,0,14,22,19,0,5,0,4,25,20,6,21,
    0,0,18,25,0,9,19,12,15,0,13,22,0,17,0,11,23,2,16,21,7,0,0,0,0,
    0,0,0,6,24,12,14,0,2,0,0,16,19,9,20,1,10,17,7,0,0,5,8,0,0,
    2,11,0,24,19,0,23,15,17,0,25,1,3,0,6,5,0,14,0,0,0,18,0,0,4,
    6,0,2,23,0,0,0,11,19,0,0,20,15,22,0,0,16,5,0,0,9,0,1,25,24,
    21,20,14,18,0,0,15,0,24,4,0,0,0,8,0,3,0,0,23,22,0,11,13,0,19,
    0,0,25,4,22,0,0,0,14,8,11,0,0,12,10,16,24,0,9,15,0,0,0,2,18,
    0,16,4,8,14,13,2,17,0,6,0,18,20,0,0,10,1,0,0,23,0,12,0,3,0,
    0,4,0,0,0,24,0,21,0,16,6,0,17,1,0,18,0,0,0,3,14,7,11,13,23,
    23,0,0,0,25,2,3,13,4,0,8,10,0,5,22,0,0,9,19,0,6,0,18,0,0,
    8,0,9,0,0,14,16,6,25,19,24,0,2,0,3,0,0,0,0,0,23,1,17,22,0,
    17,1,7,9,0,0,18,0,0,0,0,3,14,0,11,0,5,4,2,0,12,0,16,21,13
  };

  const int d25_264_3[] = {
    // Size: 25 x 25
    25,
    // Pre-assigned fields
    0,2,3,0,11,9,0,15,18,0,17,13,0,16,7,6,4,0,5,1,0,0,0,0,25,
    21,0,0,0,3,0,14,0,11,18,0,10,0,15,0,19,0,2,20,4,8,9,25,0,12,
    5,0,14,0,20,0,16,18,17,10,0,2,12,0,0,9,0,24,0,0,0,19,15,0,7,
    19,12,0,3,0,25,6,24,16,0,4,20,7,0,9,1,18,0,0,0,14,23,0,0,0,
    0,0,6,1,0,12,0,22,0,0,23,24,2,19,11,5,21,0,3,16,0,0,0,17,0,
    0,20,0,0,0,10,9,0,12,0,8,14,22,0,0,0,1,6,21,18,2,24,0,11,0,
    23,0,9,2,14,8,24,11,22,0,0,0,0,6,0,18,17,0,0,12,25,0,1,0,0,
    9,3,0,19,1,0,0,8,6,0,7,23,0,0,24,0,0,0,11,22,0,16,0,25,15,
    0,0,5,0,0,24,17,9,7,3,0,0,0,22,16,0,11,0,14,8,12,0,2,21,10,
    0,15,0,5,0,0,10,0,21,0,18,0,3,2,19,25,24,8,7,0,17,11,20,0,0,
    8,25,4,13,24,0,2,0,0,12,11,0,18,5,21,0,0,1,0,0,0,0,7,0,16,
    18,1,0,24,10,0,0,0,0,13,0,0,0,7,0,14,0,3,8,0,22,20,6,23,11,
    7,0,10,12,0,0,0,6,19,4,0,0,0,0,8,0,13,0,25,21,18,0,14,15,9,
    11,21,24,8,0,2,7,19,0,5,15,9,0,4,0,0,0,0,0,3,0,10,0,16,13,
    0,0,13,23,17,22,0,14,0,0,0,21,16,0,4,3,0,25,2,0,15,1,0,6,20,
    15,10,0,0,8,4,22,0,20,23,0,11,0,0,0,2,0,19,9,6,0,7,0,0,18,
    2,16,15,17,21,7,0,0,0,22,25,19,0,0,0,20,9,0,13,0,6,0,0,24,23,
    0,0,0,18,0,19,0,17,0,15,13,22,6,10,25,12,16,20,23,14,0,8,0,0,0,
    0,0,23,21,0,0,4,0,1,24,10,0,20,0,0,0,5,15,0,7,0,25,12,13,2,
    0,11,21,0,22,15,0,3,0,17,0,7,8,12,0,0,0,23,0,5,20,0,19,9,0,
    10,0,0,0,19,0,5,7,23,21,0,6,13,0,2,15,20,0,0,0,16,0,11,3,0,
    0,0,0,4,5,23,11,0,0,0,22,0,10,0,17,0,2,18,6,24,0,0,3,19,14,
    1,23,0,0,0,0,19,0,8,25,6,0,0,9,18,24,3,11,0,0,13,22,4,12,0,
    24,22,2,0,15,0,23,12,14,8,0,0,5,13,0,10,0,7,0,0,21,4,0,0,3,
    20,14,7,0,23,5,0,0,0,0,9,0,25,3,15,0,6,10,19,0,0,17,21,0,0
  };

  const int d25_264_4[] = {
    // Size: 25 x 25
    25,
    // Pre-assigned fields
    8,12,0,0,0,6,2,0,11,0,0,0,16,3,0,21,25,13,15,0,24,10,5,0,7,
    23,0,0,20,17,0,22,13,0,19,25,4,14,18,15,0,0,0,0,0,5,0,21,7,16,
    10,7,18,6,1,0,9,2,0,22,17,0,0,14,0,0,16,23,0,0,0,21,24,0,0,
    0,0,16,12,0,17,21,0,15,0,20,0,24,0,10,23,4,22,8,0,0,14,13,1,0,
    5,16,19,0,12,8,0,0,24,11,7,0,0,21,0,0,2,9,0,4,23,1,0,17,0,
    0,0,0,0,6,0,24,4,25,21,0,0,0,20,12,9,0,19,0,8,10,5,7,23,11,
    6,0,0,22,2,4,7,0,0,18,0,0,25,12,0,0,0,0,16,9,20,8,19,21,5,
    22,11,0,7,0,16,19,0,0,0,0,18,5,0,25,10,0,8,0,0,15,3,20,0,2,
    0,15,0,9,0,0,0,14,0,3,0,16,0,23,20,12,21,0,1,2,19,0,0,10,6,
    17,20,14,0,10,12,5,0,21,7,0,0,6,0,0,25,18,0,0,13,0,16,23,0,0,
    0,6,0,18,16,23,8,1,0,4,0,5,10,25,0,0,0,0,21,12,11,0,0,0,22,
    0,0,0,0,15,18,25,23,2,0,10,17,0,0,1,22,24,0,11,16,0,0,6,4,9,
    0,0,12,21,0,0,0,24,0,0,18,20,1,17,9,0,23,15,3,0,6,4,0,2,0,
    0,0,22,19,24,0,0,9,23,8,13,12,0,15,0,1,7,17,0,11,4,2,0,0,0,
    0,19,25,2,0,21,0,22,17,0,8,9,0,0,0,0,3,18,23,24,1,20,0,0,0,
    0,0,7,14,0,3,0,18,6,24,0,13,15,0,16,0,11,0,10,0,0,19,0,20,4,
    16,17,5,10,23,0,0,0,1,0,12,0,13,22,0,2,0,25,7,19,0,0,0,15,0,
    14,0,0,5,9,0,10,0,13,0,0,25,11,0,19,24,20,2,0,0,12,0,22,6,0,
    18,24,13,11,4,1,0,12,0,0,5,0,0,0,14,0,17,6,0,10,0,15,0,0,21,
    2,0,0,0,13,0,15,8,3,17,0,0,18,19,0,5,0,0,14,1,7,0,0,9,20,
    0,4,11,0,20,24,17,3,18,0,9,2,0,0,6,13,0,0,0,23,0,0,8,0,25,
    21,14,2,0,0,0,0,0,0,16,11,6,7,24,13,19,12,0,20,5,0,25,0,0,23,
    0,23,3,0,0,2,0,16,7,10,4,15,0,0,17,8,0,0,5,6,0,0,11,24,1,
    24,0,6,0,0,20,0,0,0,14,0,0,17,4,11,0,15,3,22,0,8,7,9,0,13,
    13,5,0,8,25,19,4,10,0,1,24,14,0,0,2,0,0,12,17,0,0,0,3,11,0
  };

  const int d25_264_5[] = {
    // Size: 25 x 25
    25,
    // Pre-assigned fields
    14,0,0,20,16,24,10,0,0,0,0,15,0,21,23,3,6,0,12,19,0,2,9,0,0,
    12,6,0,15,0,22,0,14,8,0,4,5,0,0,21,16,0,0,20,0,13,7,0,1,10,
    0,9,17,4,1,0,20,0,0,0,0,0,21,12,0,0,0,13,0,16,22,23,24,14,5,
    16,12,20,7,14,2,0,0,0,0,0,23,0,25,0,15,18,0,21,0,5,24,0,17,9,
    0,21,25,1,8,3,23,12,10,0,0,0,17,0,22,0,13,24,0,0,0,0,5,20,0,
    0,0,4,12,0,0,21,18,24,2,0,6,0,22,0,11,0,20,0,17,0,0,8,13,16,
    0,0,0,0,0,0,15,24,0,1,6,0,0,10,11,12,8,9,19,7,3,0,14,5,23,
    20,14,10,22,12,0,0,0,16,13,11,7,6,17,0,5,0,0,0,1,0,0,0,15,0,
    2,22,18,21,15,0,0,1,4,23,0,0,0,9,10,19,0,0,17,11,0,14,0,0,0,
    0,25,0,0,0,5,17,4,19,10,0,0,18,0,6,13,15,0,16,22,12,0,23,0,3,
    0,23,15,0,0,4,0,21,1,0,0,25,11,6,18,0,9,2,0,10,19,13,3,0,0,
    21,16,19,0,5,6,0,17,18,25,0,0,1,8,0,0,20,0,3,0,2,22,0,10,0,
    0,0,0,0,0,0,9,13,2,15,14,0,20,0,8,22,11,23,0,0,1,6,19,25,0,
    9,0,16,0,18,15,11,0,13,0,17,21,0,0,12,0,25,10,0,5,14,0,7,0,1,
    0,0,24,0,21,12,5,0,0,0,8,11,0,0,0,23,3,19,10,4,0,18,2,0,6,
    1,0,0,0,0,0,22,23,6,20,18,14,10,3,0,0,0,16,11,24,9,0,0,0,21,
    8,0,0,0,17,11,24,0,0,22,19,18,23,0,0,0,21,14,4,13,6,20,0,0,0,
    15,0,6,24,20,8,0,0,3,11,5,0,0,7,0,2,4,0,0,0,0,21,12,18,0,
    11,24,0,0,19,1,6,2,0,7,0,10,3,13,9,0,0,0,0,0,23,0,0,22,25,
    19,0,1,0,22,16,0,0,20,0,0,24,8,15,3,0,0,17,0,2,18,0,25,0,13,
    5,0,12,8,6,0,0,0,0,0,25,0,4,18,15,7,2,21,22,0,0,16,0,3,19,
    4,17,8,6,23,0,2,15,0,0,9,0,24,0,5,1,0,3,25,0,0,10,0,12,0,
    6,5,11,13,0,23,1,0,0,9,15,0,0,0,16,25,19,0,8,0,0,17,0,7,4,
    0,2,0,11,0,7,18,8,9,6,20,22,0,0,0,0,10,1,0,14,4,0,16,0,0,
    0,20,0,10,0,0,0,22,7,17,21,19,15,0,25,4,0,0,6,0,24,12,18,0,8
  };

  const int d25_264_6[] = {
    // Size: 25 x 25
    25,
    // Pre-assigned fields
    14,23,12,4,0,21,16,0,0,10,0,0,0,1,0,0,18,9,15,2,3,0,20,0,7,
    10,18,0,0,0,0,9,0,20,2,0,0,11,0,0,0,7,25,1,24,19,13,0,12,5,
    0,10,0,11,0,17,18,20,0,15,24,0,0,0,9,0,3,0,7,0,13,23,14,0,12,
    22,3,25,0,0,4,0,10,0,23,0,0,17,9,0,12,16,5,0,0,0,2,15,0,13,
    12,8,0,13,0,14,0,3,15,4,19,23,0,2,0,0,25,0,0,7,0,0,11,5,6,
    0,0,7,0,16,13,0,18,25,0,10,11,14,0,0,3,0,1,4,0,24,12,22,0,20,
    0,7,24,14,0,9,3,1,0,17,0,18,0,0,22,5,23,4,0,0,0,21,0,0,8,
    6,12,13,0,0,0,17,9,0,0,18,14,0,0,0,10,19,24,8,0,22,5,2,21,0,
    3,0,0,23,13,16,20,4,10,0,8,0,18,0,0,0,22,12,0,0,0,11,0,24,9,
    0,4,2,0,25,0,10,8,6,0,0,12,13,24,0,0,0,0,0,21,16,14,0,15,17,
    0,13,10,12,7,5,24,0,19,20,15,22,0,0,18,21,1,0,16,14,0,0,0,0,0,
    1,14,0,9,11,0,23,0,0,12,20,6,15,21,0,0,10,22,0,0,0,19,17,0,4,
    0,0,18,0,0,12,0,0,2,0,21,0,10,7,4,15,0,13,5,25,20,0,24,9,0,
    5,0,0,25,22,11,0,0,8,0,9,1,19,10,21,6,0,0,23,13,18,0,0,0,0,
    20,6,0,16,8,0,2,25,0,0,3,4,24,11,0,0,0,0,19,0,17,0,5,18,14,
    0,0,8,20,12,0,0,0,9,13,6,0,4,14,3,25,15,0,2,10,0,22,7,0,0,
    25,21,20,3,19,0,0,17,14,0,0,0,1,0,23,22,0,8,0,6,0,0,10,11,0,
    8,17,5,6,0,19,4,0,0,0,0,21,0,18,14,2,0,7,0,0,15,20,0,13,0,
    0,0,0,0,14,0,6,13,24,3,0,9,2,5,15,16,0,0,18,0,8,0,12,20,0,
    0,0,15,0,0,0,14,24,0,6,4,3,0,0,16,13,9,21,11,5,0,0,18,1,0,
    16,0,11,0,0,0,0,23,0,9,0,2,0,22,12,0,8,19,25,17,6,4,13,0,18,
    13,0,0,15,21,3,0,19,0,0,12,0,0,17,11,7,20,23,0,22,2,0,0,0,16,
    0,0,0,7,4,0,13,0,1,19,0,15,25,0,17,14,0,0,0,20,9,10,8,16,3,
    0,1,4,22,10,25,0,0,11,24,16,7,6,19,0,9,13,0,20,0,0,0,0,17,0,
    15,25,0,0,18,23,0,6,17,22,5,0,0,0,1,0,0,0,13,12,4,8,0,14,0
  };

  const int d25_264_7[] = {
    // Size: 25 x 25
    25,
    // Pre-assigned fields
    0,0,0,0,0,10,24,0,2,0,18,8,0,7,19,6,0,12,11,0,0,9,1,25,16,
    20,3,2,0,19,5,23,4,8,17,16,0,0,0,14,0,6,13,21,0,0,0,7,0,0,
    0,0,0,0,16,21,13,7,5,0,0,12,0,25,24,3,10,0,0,2,4,20,23,0,0,
    22,12,0,0,10,0,3,0,23,19,5,0,0,17,11,0,0,0,4,18,13,1,9,0,0,
    0,7,17,0,0,0,0,0,10,15,0,11,4,3,1,8,22,18,5,16,0,0,0,20,25,
    15,18,0,16,24,12,0,11,0,9,20,7,0,0,8,0,0,22,2,0,23,3,0,6,0,
    13,21,9,0,20,0,0,3,0,12,0,14,22,0,0,0,7,4,17,6,11,2,0,0,19,
    0,24,0,6,12,1,0,23,14,10,4,0,13,0,9,11,18,0,0,0,0,0,16,19,0,
    11,0,20,7,0,0,5,10,13,24,0,0,8,0,25,0,0,14,0,1,0,6,3,12,15,
    21,0,18,8,0,0,0,0,0,0,25,23,15,0,13,1,5,19,16,14,9,0,0,0,17,
    12,0,0,0,9,7,0,0,15,4,11,0,2,20,22,5,0,0,0,0,14,0,17,8,13,
    0,0,6,14,0,0,0,22,0,0,13,10,12,24,0,23,0,21,0,19,8,15,25,11,2,
    0,10,11,0,0,15,0,13,1,22,23,0,0,21,0,0,0,24,7,17,0,19,0,16,6,
    0,20,0,18,0,16,1,14,0,0,0,0,0,23,10,12,17,8,24,0,22,11,21,0,0,
    19,0,0,24,1,0,2,0,3,0,21,15,20,0,18,0,11,0,22,13,0,10,0,14,23,
    14,6,3,5,0,8,16,0,21,0,0,2,11,9,0,17,0,10,18,0,0,25,0,13,0,
    4,19,21,0,22,11,0,0,0,25,0,0,9,0,0,24,0,0,3,0,5,18,15,2,10,
    0,0,0,23,17,25,20,0,0,7,12,0,5,8,15,10,0,11,0,3,0,0,0,4,22,
    24,9,22,21,0,0,19,25,4,16,0,1,14,6,0,0,12,0,0,5,7,0,0,0,0,
    17,5,10,15,0,0,25,19,0,11,2,13,7,1,0,22,21,0,0,24,6,0,0,0,0,
    0,0,12,0,2,4,0,20,0,14,0,0,23,22,17,0,16,0,10,15,24,13,8,0,18,
    7,0,4,2,11,19,0,0,0,0,1,9,0,0,0,13,25,0,0,0,15,12,6,18,5,
    1,14,0,3,5,0,7,0,0,0,24,22,25,2,4,0,15,0,0,0,0,0,18,21,12,
    0,2,0,17,0,23,9,21,22,0,3,5,0,15,0,19,24,20,6,0,0,0,12,0,0,
    10,23,24,12,13,2,4,1,6,3,0,0,0,0,0,16,8,25,0,11,18,0,0,0,0
  };

  const int d25_264_8[] = {
    // Size: 25 x 25
    25,
    // Pre-assigned fields
    0,0,22,25,0,0,16,9,4,0,12,0,0,0,0,2,18,0,15,14,5,11,3,13,24,
    21,0,0,0,0,0,0,2,7,20,0,0,9,1,10,16,15,0,0,6,23,12,25,0,4,
    11,21,6,0,0,5,0,19,18,0,13,10,0,25,0,0,0,12,24,8,3,0,23,2,0,
    19,4,0,0,8,0,9,0,12,23,15,0,1,0,6,0,16,10,5,0,0,20,14,25,0,
    24,0,0,19,16,21,2,0,0,0,3,0,14,0,18,10,0,1,25,23,0,22,4,0,20,
    0,0,17,0,24,20,19,0,10,13,2,25,0,0,9,3,11,0,14,7,0,0,0,0,16,
    0,22,24,16,11,14,0,20,15,9,23,0,0,0,0,0,8,13,0,0,25,7,1,6,0,
    12,9,13,21,0,16,15,0,0,0,17,18,0,5,19,0,6,0,1,4,0,10,0,0,0,
    0,0,0,14,2,0,18,0,0,17,7,0,0,13,4,19,22,21,0,3,0,15,5,0,1,
    15,18,3,10,7,0,0,8,0,2,0,21,0,0,14,1,0,0,0,24,11,23,12,0,0,
    0,0,11,8,20,10,0,0,21,0,0,5,25,7,23,0,0,3,2,22,0,4,0,12,19,
    9,0,18,12,19,2,0,17,14,16,0,0,0,0,20,4,7,22,3,0,13,0,0,0,0,
    4,5,0,18,0,13,0,6,0,8,20,16,17,0,12,0,10,23,0,0,1,0,22,24,0,
    0,11,0,1,0,8,0,21,2,14,10,13,18,19,0,22,0,6,0,16,0,0,0,20,0,
    0,0,5,7,9,0,1,0,0,10,0,4,0,11,13,23,24,16,17,0,12,0,0,0,2,
    10,0,14,0,3,6,25,16,0,0,9,0,5,24,17,0,0,0,0,0,18,1,7,21,0,
    0,19,0,9,4,24,17,0,5,0,0,1,22,23,15,0,0,0,21,2,16,0,0,3,11,
    16,12,0,0,0,25,13,7,3,5,0,0,4,0,21,17,19,0,23,0,0,14,0,0,22,
    0,25,1,11,0,0,21,0,0,18,0,14,7,15,0,0,4,19,20,5,6,0,17,8,0,
    22,20,15,0,0,4,0,18,6,0,1,0,10,17,0,21,0,2,0,0,0,19,24,0,9,
    2,24,0,0,23,0,0,4,0,21,0,9,0,12,0,5,0,0,6,15,20,25,19,1,17,
    0,0,12,0,13,7,14,25,22,0,6,2,23,3,1,0,0,0,18,0,0,0,15,5,0,
    5,23,0,0,17,22,6,0,9,0,0,0,8,18,0,7,25,20,11,1,0,0,0,0,3,
    6,0,10,17,0,0,0,13,0,12,4,7,11,0,0,9,3,24,0,0,15,21,0,18,8,
    3,7,0,5,0,0,23,1,0,25,11,19,24,0,0,20,17,0,0,0,4,0,0,9,14
  };

  const int d25_264_9[] = {
    // Size: 25 x 25
    25,
    // Pre-assigned fields
    8,0,1,21,0,0,0,7,13,0,9,0,24,0,4,3,16,6,15,0,0,20,0,22,0,
    0,0,0,0,20,7,0,0,15,8,16,0,19,10,18,4,1,0,17,5,9,0,0,24,0,
    23,0,9,19,0,21,5,12,22,18,0,6,0,0,0,20,0,3,7,8,0,0,16,0,14,
    22,8,0,0,0,0,0,9,0,20,3,14,21,23,10,0,0,0,0,6,11,24,0,4,16,
    25,0,10,7,0,15,0,11,18,0,0,8,5,13,6,0,22,9,0,3,24,0,0,14,0,
    0,16,4,8,18,0,7,22,0,25,0,0,15,0,23,6,0,0,0,0,19,0,24,17,11,
    0,18,2,0,19,8,17,0,16,0,10,0,13,21,22,0,0,0,0,11,1,14,6,0,25,
    20,0,11,0,5,9,2,0,23,4,1,0,0,22,0,7,6,0,0,0,21,10,15,0,12,
    16,0,0,12,0,0,0,21,7,15,0,24,3,11,9,10,5,23,0,14,0,0,22,0,0,
    0,25,17,0,0,0,14,0,0,22,7,1,0,3,24,18,23,0,0,12,15,16,5,0,0,
    11,23,22,0,10,16,21,14,9,0,0,15,0,0,0,0,4,8,1,17,0,0,0,13,0,
    0,20,0,14,24,5,3,6,10,7,0,23,4,0,12,0,0,0,16,0,0,0,17,8,18,
    0,0,0,16,8,3,0,1,0,5,6,4,22,0,19,0,15,14,0,0,0,0,20,10,24,
    15,0,0,0,2,0,24,0,0,0,20,0,1,7,0,0,10,16,9,22,25,5,0,12,21,
    0,7,12,0,23,13,0,24,11,21,0,0,10,0,0,16,18,19,2,9,0,15,4,0,0,
    17,3,0,0,1,10,4,0,0,0,22,21,7,24,5,0,2,0,18,0,0,6,0,25,8,
    0,0,8,20,13,25,16,0,6,23,0,0,0,5,15,2,0,24,3,1,17,0,0,0,0,
    6,13,23,25,17,1,0,4,0,0,19,11,0,0,0,14,7,0,0,20,10,22,0,0,0,
    13,4,0,22,9,0,0,0,17,0,23,0,0,14,0,1,0,25,12,19,3,21,0,0,5,
    0,0,20,10,0,19,6,0,12,0,13,22,0,8,0,0,0,5,11,2,0,4,9,18,23,
    7,19,0,13,11,0,0,3,2,0,18,10,25,0,20,0,17,0,0,0,0,0,12,9,15,
    10,17,7,0,0,4,15,25,8,9,0,0,0,0,13,0,0,1,24,0,14,19,0,21,0,
    21,0,15,3,0,0,0,0,0,14,24,13,0,18,0,11,0,4,20,0,5,8,23,16,2,
    0,11,0,18,0,22,13,8,0,0,15,0,9,6,0,23,25,7,0,0,2,17,19,0,4,
    0,24,16,17,22,0,10,2,0,19,0,5,0,4,14,15,12,0,8,0,0,0,18,0,6
  };


  /*
   * Instances taken from examples that ship with the generator
   * "lsencode" by Carla Gomes <gomes@cs.cornell.edu>.
   */

  const int d30_316[] = {
    // Size: 30 x 30
    30,
    // Pre-assigned fields
    30,24,2,18,0,15,20,0,5,0,7,0,0,0,21,0,0,23,0,0,17,26,0,25,4,19,12,11,14,0,
    0,20,0,0,9,30,28,0,0,21,2,0,0,22,17,0,6,1,27,0,0,0,10,14,24,18,0,8,19,12,
    25,16,24,14,4,0,2,0,0,29,11,0,30,17,0,0,21,0,1,0,0,0,0,10,0,0,23,26,9,5,
    0,5,30,27,25,24,23,0,0,8,0,0,0,12,2,6,18,0,28,0,19,22,7,17,0,16,4,0,0,1,
    28,0,4,3,14,0,0,12,0,2,17,22,0,8,11,20,26,0,21,0,0,0,19,27,0,10,0,0,0,23,
    0,10,0,30,0,1,14,0,0,0,21,0,19,11,18,23,9,7,13,0,0,27,0,0,0,0,28,20,24,29,
    0,0,0,0,0,16,0,2,0,30,19,28,25,26,6,14,1,27,0,0,0,0,24,0,21,0,0,23,17,7,
    10,0,0,15,21,28,19,23,0,26,9,12,1,0,24,22,0,4,0,0,5,29,0,20,17,11,0,14,27,18,
    5,0,10,0,19,12,3,7,25,0,6,14,21,0,0,29,8,18,16,4,24,9,0,0,26,2,22,15,0,13,
    7,17,14,19,27,20,0,0,29,3,0,0,0,9,0,0,28,13,0,1,15,0,0,4,10,0,26,24,30,0,
    0,4,13,20,0,0,0,28,12,19,27,6,0,7,26,5,24,2,11,9,22,14,0,18,0,1,8,17,0,0,
    26,14,22,8,0,10,0,6,0,4,0,0,0,0,0,19,0,15,29,3,20,7,9,13,0,0,11,0,0,24,
    24,0,11,0,0,18,0,0,14,5,22,8,0,0,0,0,12,17,9,23,6,0,0,0,30,28,0,0,21,2,
    16,27,15,24,12,26,25,17,0,1,0,0,23,0,0,30,0,0,10,0,21,28,18,3,29,14,9,7,6,22,
    18,3,26,0,7,0,4,0,0,10,29,0,22,23,0,1,0,19,5,28,9,17,25,0,11,0,0,0,20,14,
    0,0,20,0,0,3,0,29,10,28,30,13,26,24,0,16,0,21,25,8,23,19,1,9,14,4,18,27,0,17,
    19,0,7,25,20,9,16,15,3,14,0,10,17,30,29,27,4,0,23,21,18,0,11,0,28,0,6,12,2,8,
    0,2,19,29,26,0,11,10,0,16,8,23,0,21,5,24,0,0,20,0,7,0,3,0,0,9,25,0,4,28,
    9,23,0,7,6,0,29,0,22,24,10,0,0,1,4,26,11,5,19,0,12,2,13,16,3,27,0,30,0,20,
    2,0,29,16,10,0,0,11,18,15,1,5,0,4,13,17,20,0,6,24,0,21,14,12,8,30,0,0,22,0,
    6,29,1,0,28,17,27,0,0,0,0,20,0,14,7,0,3,0,0,2,8,0,4,22,15,26,21,5,16,30,
    0,0,18,0,22,29,0,0,0,17,0,25,20,10,19,0,5,0,15,27,0,30,2,6,0,0,24,28,1,9,
    0,12,21,0,18,27,30,0,0,0,28,4,2,0,9,25,23,14,0,0,0,11,0,29,0,6,0,10,0,3,
    0,11,28,0,29,22,17,0,24,6,0,9,0,16,20,21,0,25,0,12,30,0,0,0,2,7,0,1,5,19,
    29,22,6,21,13,11,26,0,1,0,20,19,24,2,0,8,0,16,17,0,27,3,0,28,0,25,10,0,23,0,
    0,26,9,22,24,4,5,20,11,27,18,17,0,13,25,28,19,12,14,6,2,0,8,30,0,29,3,21,15,10,
    8,0,16,9,0,0,0,21,27,25,24,7,12,19,0,18,0,0,2,0,1,4,0,0,22,20,30,0,13,0,
    22,30,8,0,0,0,0,0,21,0,0,27,11,0,1,10,7,28,4,17,14,6,29,0,9,24,20,25,0,16,
    0,19,27,28,17,6,0,0,4,0,25,21,15,20,30,2,29,8,26,0,0,0,0,24,5,23,16,22,3,0,
    1,15,17,0,0,19,24,0,9,23,14,3,6,0,0,4,16,0,0,20,11,12,0,0,0,0,7,13,0,26
  };

  const int d30_320[] = {
    // Size: 30 x 30
    30,
    // Pre-assigned fields
    0,2,3,0,5,6,7,8,9,0,11,0,0,0,0,16,17,18,0,20,21,22,0,24,0,0,0,28,0,30,
    2,3,4,0,6,7,0,9,10,0,12,0,14,15,0,17,18,0,20,21,22,0,24,0,26,27,0,29,30,0,
    0,4,5,6,7,8,9,10,0,0,0,14,15,16,0,18,19,20,21,22,23,24,0,26,27,28,0,30,1,2,
    0,5,0,7,8,9,10,11,12,0,14,15,16,0,18,19,20,21,22,23,0,25,26,27,28,29,30,1,2,0,
    0,0,7,8,0,0,0,12,0,0,15,16,17,18,19,0,21,22,0,24,25,0,27,28,29,30,0,0,3,4,
    6,7,0,0,0,11,12,13,14,15,16,17,0,19,20,21,22,0,24,0,0,0,0,29,0,1,2,0,0,5,
    0,0,9,10,11,12,13,14,15,0,17,18,19,0,21,0,0,0,25,26,27,0,0,0,1,2,3,4,0,0,
    0,9,0,11,12,13,14,15,0,17,18,0,20,21,0,0,24,25,26,27,28,29,0,1,0,3,4,0,0,0,
    9,10,0,12,13,14,15,16,0,18,19,20,21,22,23,24,25,0,0,28,29,30,0,0,3,4,5,6,7,0,
    0,11,0,0,0,0,0,17,18,19,20,21,22,0,0,25,26,27,0,0,30,0,2,3,4,5,0,0,8,9,
    11,12,13,14,15,0,17,0,19,20,21,22,0,0,25,26,0,28,29,30,1,2,3,4,0,6,7,8,0,10,
    12,13,0,15,16,0,18,0,20,0,22,23,0,25,26,27,0,0,0,1,2,3,4,0,6,7,0,9,0,11,
    0,0,15,16,0,0,0,20,21,22,23,0,25,0,27,28,29,0,0,2,3,4,5,6,7,8,0,10,11,12,
    0,0,16,17,0,19,20,0,22,0,0,25,0,27,28,29,0,1,2,3,4,0,0,7,8,9,10,11,0,13,
    0,16,17,0,0,20,0,22,23,24,0,26,27,0,0,30,0,2,0,4,0,6,7,0,0,10,11,12,13,14,
    0,17,18,0,0,21,22,23,24,25,26,27,0,29,30,1,2,0,0,0,6,0,0,9,10,0,0,13,0,15,
    0,0,19,20,21,22,23,0,25,26,27,0,29,30,0,0,3,4,5,6,7,8,0,0,11,12,0,0,15,16,
    0,19,20,0,0,23,0,0,26,27,28,29,30,1,2,0,4,0,0,7,8,9,10,11,0,13,14,0,16,17,
    19,0,0,0,0,24,25,26,0,0,0,0,1,0,3,0,5,0,0,8,9,0,11,12,13,0,15,16,0,0,
    0,0,22,23,0,0,26,0,28,29,30,1,0,0,4,5,6,0,8,9,10,0,12,0,14,0,16,0,0,19,
    21,22,23,0,25,26,27,0,29,30,0,2,3,4,5,6,0,8,9,0,0,0,13,0,0,16,17,18,19,0,
    22,23,24,25,26,27,28,29,0,1,2,3,0,5,6,0,0,9,0,11,12,13,0,15,0,17,18,19,0,21,
    23,24,25,0,0,28,0,0,0,0,0,4,5,0,7,0,0,0,0,12,13,0,0,0,0,18,19,0,0,0,
    24,0,0,27,0,0,30,1,2,0,0,5,0,0,8,0,0,11,0,13,14,15,16,17,0,19,20,21,0,0,
    25,0,27,28,29,30,1,2,3,4,5,6,7,8,0,0,0,12,0,14,0,16,17,18,19,20,0,22,23,24,
    26,27,28,29,0,1,0,3,4,0,6,0,8,9,10,11,12,0,14,15,0,17,0,0,20,21,0,0,24,25,
    27,0,0,0,0,2,3,4,0,6,7,8,9,10,0,0,13,14,15,16,17,0,19,0,21,22,0,24,25,0,
    28,29,30,0,2,3,4,0,0,0,0,0,10,11,12,13,14,15,0,0,18,19,20,21,0,23,24,25,26,27,
    0,0,1,0,0,4,0,0,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,0,23,24,25,26,27,0,
    30,1,0,3,4,5,6,7,0,9,0,11,12,0,14,0,0,0,18,19,0,21,22,23,24,0,26,27,28,29
  };

  /*
   * 40 hard 30x30 instances with 42% holes generated with "lsencode" by Carla
   * Gomes <gomes@cs.cornell.edu>. Those instances where first used in
   *   A. Zanarini, G. Pesant, Solution counting algorithms for
   *   constraint-centered search heuristics, CP 2007.
   *   Pages 743-757, LNCS 4741
   *
   */

  const int d30_374_1[] = {
    30,
    0,19,0,29,0,15,0,25,0,4,7,21,0,14,0,24,22,0,0,0,5,23,10,26,0,16,9,2,0,8,
    0,18,0,5,29,22,0,17,0,7,30,6,21,0,24,0,26,1,13,11,23,0,15,14,19,0,0,0,0,0,
    15,0,30,0,4,6,19,0,0,14,0,0,0,0,10,27,0,0,12,20,25,18,22,0,13,7,0,0,29,17,
    0,0,20,0,9,17,6,0,16,0,23,18,12,0,13,0,0,11,26,0,29,0,0,0,2,19,15,7,3,0,
    0,0,4,3,0,25,0,0,0,0,22,8,0,0,20,14,24,6,0,29,30,7,0,27,12,0,0,5,19,23,
    0,0,0,0,17,12,0,24,15,16,4,20,14,0,0,0,0,18,0,0,19,2,27,30,0,6,0,26,8,29,
    14,12,6,23,0,0,0,4,0,2,13,24,9,0,0,3,17,30,8,0,7,0,0,0,20,18,0,11,0,5,
    12,0,0,7,0,1,9,30,28,25,0,14,20,0,19,22,10,0,0,26,27,0,0,11,0,5,0,0,6,24,
    0,0,28,0,11,0,0,7,0,0,0,27,3,17,18,21,0,14,23,24,12,0,25,9,26,22,0,0,16,2,
    2,3,21,0,26,0,5,1,14,0,0,17,0,6,0,23,18,10,25,28,0,15,9,16,27,0,0,0,0,0,
    16,28,10,0,19,11,0,0,24,0,6,0,0,29,23,30,0,21,2,3,1,22,0,5,0,13,0,0,12,0,
    0,15,13,0,5,0,20,0,26,27,0,7,22,1,0,4,19,0,14,10,0,0,11,0,8,24,0,0,25,0,
    4,0,12,21,0,23,3,0,7,17,0,0,16,5,29,9,0,0,0,27,0,0,8,10,28,1,2,14,0,0,
    0,0,16,0,0,21,12,19,0,26,0,13,8,22,2,0,25,5,15,0,20,0,24,0,0,0,17,6,23,0,
    18,2,29,0,0,9,10,28,30,0,12,22,23,0,0,0,14,0,5,0,0,20,7,0,0,15,26,0,17,0,
    5,24,0,6,25,0,0,0,10,19,1,0,29,2,8,0,30,20,0,12,0,28,3,0,0,0,11,27,15,0,
    30,8,5,0,7,0,16,14,21,3,0,0,11,9,26,29,12,13,1,0,0,0,0,4,0,0,0,22,0,0,
    23,0,9,27,21,0,0,20,25,28,0,0,24,0,4,17,13,0,0,0,26,0,0,18,29,2,14,0,0,22,
    0,0,19,28,0,0,15,21,5,0,26,29,0,20,0,0,16,0,0,1,0,12,0,0,9,8,7,17,24,30,
    6,0,0,0,2,24,0,0,8,0,3,15,0,0,11,20,0,0,16,9,0,27,30,17,0,14,10,1,13,4,
    28,14,22,30,0,3,0,0,2,0,19,0,17,10,15,0,0,29,0,23,11,0,26,0,0,20,12,0,21,1,
    0,11,0,19,24,0,30,8,0,1,0,9,0,26,0,0,0,12,3,0,0,16,18,22,21,0,25,15,20,7,
    13,23,15,17,0,28,21,26,0,0,11,0,2,0,27,8,0,0,9,4,0,14,12,0,10,0,0,0,0,19,
    0,10,11,13,0,0,0,0,0,22,24,23,26,25,12,15,1,0,7,0,8,6,0,0,16,17,18,28,0,0,
    29,25,0,0,12,5,8,11,0,9,10,3,0,0,0,26,7,0,0,0,14,1,0,2,15,0,6,16,0,13,
    21,9,0,26,15,0,25,0,22,23,0,0,0,27,0,0,0,4,18,0,0,11,13,7,5,0,3,29,0,10,
    10,29,0,14,0,26,4,0,9,5,0,0,15,0,6,0,11,7,24,18,0,8,0,28,0,0,19,0,2,21,
    26,6,0,0,22,0,0,16,19,0,18,11,0,12,21,0,3,17,0,2,13,25,0,0,23,27,0,4,0,0,
    0,0,0,22,0,13,1,2,29,0,15,0,19,28,0,11,4,27,0,16,21,0,6,12,0,0,20,0,0,9,
    0,13,24,16,30,0,14,0,0,18,0,0,0,15,0,0,0,9,10,0,3,26,2,25,0,29,5,20,11,6
  };

  const int d30_374_2[] = {
    30,
    15,4,20,0,14,10,23,0,0,7,0,8,0,0,26,0,0,21,3,0,0,11,17,13,9,0,12,0,6,16,
    0,11,29,0,0,0,0,0,17,0,15,27,24,18,25,4,0,0,0,9,23,19,0,0,3,5,16,0,26,6,
    29,0,0,0,8,0,0,5,0,3,0,24,20,9,0,17,0,0,18,2,11,0,14,21,0,25,0,28,16,30,
    6,5,22,20,9,29,0,30,19,0,0,25,0,15,0,0,23,14,0,0,0,0,0,1,16,13,2,26,0,0,
    0,0,27,0,24,6,7,29,28,0,1,0,3,0,16,0,13,0,2,26,15,20,4,0,10,0,0,0,17,0,
    0,10,0,9,12,2,0,0,15,24,17,0,0,4,0,28,0,20,0,0,30,0,3,0,27,29,23,11,0,8,
    7,20,12,0,0,13,4,18,29,22,16,0,2,0,0,0,5,0,0,23,8,21,0,19,26,0,0,0,25,11,
    0,0,0,13,0,0,22,0,9,27,25,4,14,0,30,5,24,16,0,28,0,10,11,0,19,0,8,0,1,0,
    24,30,13,26,25,0,0,4,27,5,21,0,0,0,0,3,0,0,0,0,6,28,20,16,0,22,0,29,19,7,
    0,0,1,21,0,0,6,24,14,28,3,30,22,2,4,26,0,0,27,19,0,0,7,0,0,0,0,0,5,15,
    5,0,0,12,26,0,9,0,3,0,0,0,0,21,0,0,17,0,7,13,4,0,29,25,8,0,11,27,14,20,
    27,0,16,0,11,0,0,12,0,0,14,0,0,20,8,0,0,0,30,24,26,0,13,5,4,1,18,0,22,19,
    10,0,17,0,2,0,0,0,5,21,0,28,26,0,7,0,14,9,0,30,0,18,1,4,13,16,0,8,20,0,
    17,13,19,0,0,25,11,2,8,0,0,22,0,3,27,0,1,30,0,20,0,14,10,6,0,0,15,16,0,0,
    30,23,0,0,13,0,0,20,10,15,0,7,16,17,0,0,12,1,21,6,0,0,0,29,14,0,9,25,0,0,
    0,26,14,4,7,20,0,22,6,0,28,0,12,0,3,21,2,27,0,10,0,8,0,0,0,24,17,0,0,13,
    0,0,0,0,0,8,0,27,0,19,13,0,9,1,29,12,0,7,16,0,0,2,22,28,5,11,20,18,0,10,
    8,18,0,30,27,4,19,17,0,25,0,29,11,23,5,16,0,26,0,0,0,13,21,12,0,0,0,14,0,0,
    0,22,2,0,0,26,0,0,0,10,0,11,21,12,28,15,16,0,8,0,5,17,30,0,1,0,0,23,0,9,
    19,0,0,3,0,0,5,23,2,0,6,0,30,0,9,7,4,10,11,8,21,27,0,0,0,17,0,22,0,28,
    1,29,5,8,0,0,0,25,23,12,26,10,19,0,0,13,0,28,0,0,7,24,9,0,0,3,6,0,21,0,
    23,15,0,28,0,14,21,0,26,29,0,20,7,0,11,6,0,0,25,0,9,30,0,0,0,2,5,10,0,22,
    0,0,10,1,0,18,30,0,0,0,7,0,0,26,0,27,0,19,28,3,0,0,0,8,22,12,21,17,15,25,
    12,0,3,22,29,16,15,0,0,0,19,9,8,27,17,25,26,0,14,0,13,0,0,0,0,0,1,2,0,24,
    0,0,4,29,28,3,2,0,12,6,0,0,0,0,0,19,30,11,0,1,20,0,0,15,18,21,27,0,10,14,
    11,1,0,16,20,28,0,13,0,14,5,0,6,8,0,10,15,0,29,0,3,12,0,26,0,0,0,9,7,0,
    4,28,30,18,0,0,26,15,0,0,24,1,0,10,2,9,7,17,13,22,0,0,0,3,0,20,0,0,0,29,
    0,0,0,6,0,0,3,0,11,0,10,13,0,28,1,0,20,5,15,16,14,0,0,17,23,18,0,24,30,0,
    16,19,0,0,18,11,14,28,0,0,0,26,25,0,0,0,27,0,17,0,12,9,8,0,0,15,3,30,2,0,
    0,7,15,0,4,17,8,16,0,9,18,6,0,0,21,0,29,13,0,14,10,0,19,11,24,27,0,0,0,0
  };

  const int d30_374_3[] = {
    30,
    0,6,26,0,0,16,2,14,5,19,7,0,0,0,0,8,9,0,13,22,0,0,0,21,11,3,12,0,0,29,
    0,2,0,19,0,12,21,0,1,14,0,0,0,0,4,0,0,23,0,18,17,0,3,8,16,0,11,24,22,13,
    24,26,0,0,2,0,0,0,27,17,0,8,0,20,12,29,28,0,18,0,0,21,25,0,1,30,0,0,11,22,
    0,0,5,16,0,27,1,0,0,3,0,11,2,0,13,28,4,0,0,19,20,10,0,29,0,22,14,0,17,0,
    3,0,0,22,16,0,15,0,0,23,24,27,30,1,0,18,12,0,0,25,14,28,5,0,0,8,20,6,0,0,
    30,0,0,0,10,15,0,1,0,21,25,0,24,22,0,20,0,0,0,0,8,26,0,27,7,0,3,2,9,12,
    0,0,2,11,0,30,0,0,16,1,0,21,4,6,0,25,0,0,0,27,24,14,0,3,12,23,0,0,19,18,
    29,8,0,28,12,0,0,25,21,0,0,2,11,0,0,0,26,22,6,17,0,9,19,20,0,0,0,3,18,30,
    0,5,28,0,3,0,19,0,23,25,18,0,0,30,0,17,6,13,0,16,0,1,20,0,26,0,0,4,0,24,
    18,0,15,4,6,5,0,16,19,8,0,3,27,11,10,0,21,0,9,0,22,0,13,0,0,20,0,0,0,1,
    9,14,3,0,0,8,0,21,29,0,0,5,0,26,15,0,2,16,0,0,0,11,7,0,17,0,1,0,25,10,
    0,0,1,0,0,29,0,10,0,24,5,7,0,17,25,23,14,0,0,2,6,0,15,0,9,0,16,11,0,8,
    6,7,0,0,22,2,0,0,25,0,0,14,12,24,5,0,0,19,29,4,16,0,28,17,0,26,10,9,0,0,
    0,13,0,5,27,0,3,30,7,0,19,26,0,0,23,15,29,14,21,12,0,6,0,0,0,0,0,10,0,16,
    0,25,30,17,0,0,18,11,13,0,4,6,0,0,0,27,23,5,20,0,0,0,2,26,0,14,22,16,0,0,
    0,16,13,0,0,0,11,2,9,0,17,0,15,0,8,0,3,1,4,24,19,20,0,0,0,21,0,25,0,5,
    11,0,0,24,18,21,0,12,0,0,0,0,0,13,0,16,0,29,8,0,10,15,23,0,30,6,9,20,2,26,
    14,24,6,30,9,17,7,0,0,27,23,18,0,0,20,0,5,15,1,0,0,0,0,28,29,0,0,21,0,4,
    23,0,0,1,30,25,4,5,12,0,10,19,18,0,0,0,0,3,15,0,0,0,0,7,20,28,21,0,13,9,
    0,0,16,10,19,0,0,3,17,28,11,9,1,29,0,0,20,4,26,0,15,23,22,0,0,12,0,0,14,0,
    5,0,19,0,13,28,25,0,2,0,0,0,0,18,24,22,7,0,0,0,0,17,0,14,21,0,23,12,20,27,
    12,23,22,0,0,20,6,0,0,29,2,16,10,15,28,1,0,21,27,0,9,0,0,19,3,0,0,0,30,0,
    15,0,9,25,29,1,0,0,14,11,0,0,0,0,18,24,22,0,0,0,13,0,4,12,23,16,7,0,6,21,
    22,3,0,15,5,23,17,0,11,20,0,0,29,25,0,0,0,6,7,8,26,0,0,10,28,9,13,0,0,0,
    0,10,0,0,0,0,14,24,28,9,1,0,0,5,7,0,18,0,16,6,27,19,30,2,8,0,17,0,12,0,
    0,19,24,14,25,0,0,6,0,4,29,0,17,8,11,13,0,0,23,20,7,16,0,1,0,15,0,5,0,0,
    27,28,25,26,0,0,16,0,0,0,13,0,19,4,0,21,0,11,0,7,1,0,9,0,18,24,15,14,10,0,
    16,0,17,0,23,0,5,22,0,0,12,13,20,0,3,10,0,25,24,15,0,8,14,0,0,0,0,27,29,0,
    0,0,21,0,0,10,20,7,0,0,0,17,5,0,9,0,0,30,22,1,23,13,16,25,2,0,24,29,3,0,
    19,18,0,20,0,0,26,15,0,0,21,30,6,23,14,0,27,12,0,3,0,0,1,9,0,25,29,0,5,0
  };

  const int d30_374_4[] = {
    30,
    0,0,20,16,26,22,0,0,14,19,0,24,5,11,0,13,27,0,28,12,0,6,0,0,23,0,17,0,15,18,
    0,0,26,6,1,7,28,0,19,0,0,0,0,0,24,0,15,23,21,8,11,0,12,0,18,27,14,9,0,30,
    0,0,0,19,6,0,8,10,17,30,24,14,0,27,0,0,7,0,18,0,9,23,0,15,0,0,22,0,11,16,
    14,21,0,7,0,0,15,0,0,0,0,0,4,24,3,0,6,0,20,30,23,10,19,13,16,0,29,2,0,9,
    0,0,0,0,10,2,0,3,8,9,23,4,28,0,0,5,17,21,26,0,0,0,18,12,19,0,0,14,25,0,
    0,22,1,24,25,0,3,18,0,0,11,13,0,0,10,28,8,0,9,0,2,19,0,29,20,0,7,0,4,0,
    17,0,0,20,0,21,30,15,22,5,0,27,11,0,0,1,9,18,24,0,29,0,2,0,0,19,0,0,13,10,
    19,0,22,0,30,11,0,7,5,24,14,0,0,0,18,29,13,1,0,0,28,0,15,0,0,17,0,8,16,0,
    0,23,5,12,0,14,22,16,0,0,27,20,0,25,0,0,0,0,0,15,13,9,3,0,0,2,6,21,10,19,
    4,0,24,0,28,5,0,2,0,0,13,0,21,3,0,11,0,8,19,10,14,29,30,7,0,25,0,0,26,0,
    0,0,0,10,3,15,0,17,0,21,7,0,1,13,0,0,30,0,2,25,16,26,4,0,9,8,5,0,0,0,
    15,30,0,17,0,23,9,29,0,0,20,28,0,4,13,24,0,7,0,1,0,0,0,25,5,0,21,18,0,0,
    20,0,16,1,8,0,2,21,0,22,9,0,15,29,23,0,12,0,10,0,0,0,24,30,0,0,13,25,19,0,
    0,12,0,0,15,0,0,5,0,13,0,18,27,0,20,7,29,4,23,22,26,0,28,0,24,9,0,30,6,0,
    0,5,0,0,20,25,27,0,16,3,0,30,13,0,21,18,0,0,0,28,17,12,0,0,29,0,23,0,24,4,
    18,13,27,0,0,0,0,14,0,26,3,10,7,0,0,30,23,20,0,0,0,8,5,0,0,0,15,4,1,2,
    26,0,21,27,0,0,25,12,13,7,0,6,0,9,0,0,0,3,0,0,10,0,0,20,0,23,1,15,8,22,
    25,7,30,0,9,0,13,0,0,0,28,0,2,0,0,6,0,0,0,17,0,22,27,10,12,0,11,3,18,26,
    30,9,4,0,2,19,0,0,21,16,0,3,0,0,25,0,11,22,0,0,15,0,7,17,1,0,8,6,0,29,
    0,3,0,0,0,29,10,26,20,11,25,23,30,15,9,0,21,16,12,0,7,0,8,22,0,0,0,0,0,24,
    9,10,0,8,5,4,11,0,12,28,0,17,16,14,0,0,20,0,0,23,0,25,0,0,26,7,0,24,0,0,
    0,28,2,5,19,17,0,0,0,6,1,0,9,30,0,26,0,13,15,7,8,14,0,0,0,4,0,0,0,11,
    16,8,19,0,0,0,0,0,26,0,2,29,14,20,7,21,0,28,25,24,6,13,0,0,0,22,0,12,3,0,
    12,16,0,0,27,0,0,0,0,0,10,0,0,22,17,15,5,0,3,13,0,18,9,14,28,24,0,0,30,23,
    3,1,17,26,0,0,6,11,4,0,15,0,0,0,22,20,0,19,0,0,0,28,0,18,0,30,25,5,27,0,
    6,15,13,0,4,9,26,0,3,27,0,1,0,0,16,25,0,17,0,0,0,0,29,24,10,12,30,0,0,0,
    0,18,3,28,0,0,19,0,7,10,0,0,0,2,29,0,0,5,8,11,0,21,14,1,0,13,4,16,0,20,
    7,0,29,0,0,0,0,0,18,0,0,12,25,1,15,27,28,26,11,0,20,0,0,4,6,14,0,17,5,21,
    28,0,0,11,24,26,12,30,1,0,18,9,22,17,8,0,0,0,0,29,0,4,25,2,13,0,0,20,0,0,
    0,4,14,25,0,6,0,1,0,15,0,0,0,7,0,23,10,0,30,27,3,24,0,26,22,5,2,0,0,12
  };

  const int d30_374_5[] = {
    30,
    25,18,16,19,4,0,0,7,0,0,0,0,29,15,14,17,0,13,0,24,3,2,0,0,27,0,30,21,0,0,
    5,7,6,0,0,14,28,17,0,0,4,0,13,9,20,23,2,29,1,0,0,0,0,0,11,12,0,0,22,19,
    14,0,11,0,12,0,0,27,23,18,24,9,2,30,17,19,21,0,0,16,0,7,20,28,0,0,0,25,0,0,
    9,0,13,4,0,5,0,0,15,0,0,28,26,0,0,25,22,24,2,11,0,0,23,0,0,0,19,10,21,1,
    0,11,24,23,0,17,21,9,0,12,27,25,6,0,15,0,0,0,10,8,0,0,19,30,0,26,0,0,7,0,
    24,0,0,0,16,0,27,22,5,13,30,10,0,19,28,4,11,21,0,2,0,0,8,20,17,0,7,0,0,0,
    0,0,10,27,17,0,0,1,0,24,3,0,25,0,18,9,0,6,14,23,19,13,15,0,2,0,4,0,12,0,
    0,0,0,25,13,0,19,0,1,23,17,3,0,5,6,0,10,27,0,14,0,20,30,15,26,0,0,0,16,28,
    15,0,7,0,3,0,25,0,21,2,20,0,8,11,26,0,30,0,16,0,0,18,12,0,0,29,27,28,0,22,
    10,8,18,12,0,22,26,13,0,0,7,15,0,1,0,0,0,9,23,0,0,29,0,6,20,0,21,11,0,0,
    0,23,0,15,0,0,0,0,13,0,0,22,0,27,4,12,29,2,20,0,0,0,17,3,0,5,8,1,11,25,
    20,16,27,0,0,30,12,0,26,0,14,0,0,0,9,0,4,8,24,0,0,28,0,13,5,18,0,23,0,15,
    18,3,9,26,1,0,0,19,0,29,0,0,0,8,0,0,0,14,0,12,10,17,0,27,13,20,28,22,30,0,
    0,2,26,0,8,21,1,16,0,5,12,27,22,10,0,0,0,4,0,0,11,0,29,7,18,0,0,14,0,20,
    6,19,0,22,21,13,0,20,8,0,26,0,0,18,0,0,0,11,9,0,28,15,4,0,30,0,25,24,29,0,
    28,21,5,30,26,11,0,0,0,22,15,0,17,0,0,0,0,0,29,0,12,14,0,25,0,16,23,0,4,13,
    0,0,0,0,20,27,11,26,29,10,0,1,0,0,0,0,25,28,8,7,9,12,0,24,0,13,14,0,0,30,
    7,0,0,9,22,0,24,30,19,0,0,0,5,20,27,14,16,17,0,3,6,10,0,0,0,0,0,0,2,12,
    8,26,0,0,19,0,0,18,14,0,11,20,0,0,13,0,23,25,27,1,21,0,22,0,0,0,3,30,17,9,
    27,0,21,0,0,0,17,5,0,30,0,4,0,0,29,22,0,10,25,0,13,9,7,26,12,0,0,0,28,8,
    0,25,1,0,0,15,0,0,0,16,0,21,9,3,7,0,0,0,17,19,26,24,5,12,10,2,0,18,13,0,
    23,0,0,21,0,0,4,0,6,0,0,0,0,24,0,3,8,15,0,20,30,27,16,17,28,1,12,5,19,0,
    3,12,29,28,0,0,0,0,11,6,16,26,0,7,0,20,0,0,0,0,25,0,2,0,1,9,17,0,10,4,
    0,27,25,5,15,12,30,24,18,17,19,0,4,0,0,10,0,0,26,0,2,0,0,0,0,23,0,9,0,6,
    0,13,0,0,2,29,7,14,0,28,0,11,18,16,0,8,6,0,5,9,24,0,0,19,3,25,0,0,0,27,
    0,0,0,0,30,18,20,12,22,0,0,0,28,4,2,26,7,0,19,29,0,0,0,11,23,15,6,0,14,17,
    13,29,0,6,23,19,2,21,0,11,0,14,30,0,0,15,18,0,12,0,4,22,0,16,0,0,0,17,0,0,
    0,0,8,0,25,1,9,0,12,0,0,13,20,0,30,0,5,0,0,4,27,16,14,0,19,6,10,15,0,0,
    4,9,19,7,0,20,23,0,2,27,28,0,24,0,5,21,1,26,0,15,0,0,0,0,0,11,16,0,3,0,
    0,0,0,2,0,28,0,0,0,0,25,19,21,0,22,24,12,0,0,13,15,3,1,18,0,14,20,29,26,16
  };

  const int d30_374_6[] = {
    30,
    2,0,10,0,0,0,26,0,21,7,8,22,9,6,0,0,14,16,0,15,4,23,0,25,13,12,0,0,0,29,
    0,30,28,26,0,0,9,11,2,20,24,0,21,15,0,0,16,0,14,17,0,0,10,0,0,0,5,4,29,27,
    0,0,16,3,0,0,0,2,0,0,20,8,12,19,27,22,28,0,7,0,0,0,24,23,14,26,21,0,17,5,
    0,10,0,4,8,12,0,29,0,0,0,0,22,0,0,0,0,13,25,16,5,20,2,1,7,23,0,15,19,6,
    26,12,24,0,9,18,16,23,0,0,10,21,11,29,0,0,0,15,17,22,0,0,20,0,5,0,0,28,0,0,
    8,23,0,0,13,7,0,18,0,26,0,0,5,2,0,20,17,22,1,14,12,0,25,15,11,0,0,0,0,0,
    0,16,22,14,4,0,0,19,0,13,0,15,1,23,0,11,0,26,0,0,6,30,0,21,0,9,12,18,0,0,
    1,24,9,0,28,10,12,5,20,0,0,0,0,27,21,14,0,0,11,0,23,15,30,0,19,0,17,0,0,18,
    5,25,26,0,0,9,4,0,28,10,21,0,0,0,0,0,24,0,0,0,0,13,29,14,16,0,23,12,22,3,
    0,1,4,0,30,24,29,0,17,0,0,7,14,0,0,8,2,9,18,25,27,0,0,0,10,6,0,0,15,11,
    30,27,0,2,0,19,3,0,24,0,12,18,0,0,13,23,0,11,0,9,15,0,1,0,0,0,4,5,26,25,
    0,0,0,0,12,0,24,0,0,18,11,17,26,0,7,0,4,29,15,10,9,2,8,0,0,16,27,0,23,0,
    24,0,20,0,7,8,0,0,13,16,2,0,0,26,14,29,0,12,4,0,0,1,0,0,3,11,28,0,27,15,
    0,0,0,6,0,23,0,0,19,29,0,0,0,17,5,24,18,27,3,30,16,0,11,12,4,13,0,21,0,7,
    0,0,8,11,0,28,0,0,0,0,0,19,10,0,23,18,25,0,24,26,21,0,5,6,0,0,30,3,16,20,
    22,0,0,17,0,25,11,10,0,28,13,16,27,0,24,1,5,0,2,6,19,26,0,0,0,0,9,0,7,0,
    17,20,18,0,0,0,0,3,11,0,0,9,25,0,26,15,23,28,10,0,0,21,4,0,30,0,29,7,0,1,
    0,0,0,28,6,15,0,30,10,12,16,11,17,0,8,13,27,4,0,0,0,19,0,0,0,22,0,29,20,0,
    3,0,12,0,29,4,21,6,26,22,23,27,0,0,2,0,0,19,8,5,0,18,16,30,0,0,0,17,0,0,
    9,8,2,27,3,0,10,0,0,0,0,29,0,25,17,0,7,21,13,0,24,28,0,20,18,1,0,0,11,0,
    23,0,14,24,27,29,20,28,8,4,0,0,0,0,16,12,0,7,0,0,0,0,15,18,1,17,0,6,0,30,
    27,0,5,7,21,22,0,20,0,30,14,0,0,13,0,6,0,0,9,11,1,0,0,10,0,4,26,23,0,0,
    0,0,13,29,23,2,22,0,0,17,0,6,7,16,0,0,9,0,0,18,28,0,0,24,0,0,14,10,25,21,
    21,14,6,0,0,0,7,8,0,0,15,2,0,0,9,26,0,0,16,13,29,12,0,5,0,20,1,0,4,23,
    0,22,0,21,0,0,0,25,3,5,0,0,8,28,0,0,11,0,0,24,30,7,19,16,17,15,0,9,2,0,
    16,15,17,19,0,13,0,1,27,0,9,0,28,18,0,0,0,0,23,29,25,4,0,0,12,5,0,0,14,0,
    11,3,0,0,19,0,17,15,14,27,1,0,0,0,20,0,26,10,0,8,0,0,9,2,23,18,0,16,0,0,
    0,19,0,30,0,0,1,0,29,0,28,0,0,12,22,2,20,24,0,0,14,8,0,0,0,3,11,27,13,26,
    14,26,0,22,25,0,0,0,6,0,7,1,19,8,0,0,3,5,0,0,0,11,0,0,20,28,2,13,18,16,
    0,0,0,23,18,0,2,9,16,25,0,12,24,4,11,27,0,0,0,0,0,29,13,7,6,0,22,0,0,28
  };

  const int d30_374_7[] = {
    30,
    0,26,16,0,15,14,0,0,0,11,23,19,21,3,0,5,20,17,2,8,0,6,0,0,0,0,7,24,0,12,
    0,16,10,0,0,0,30,0,0,5,0,0,28,9,0,26,8,19,0,6,7,4,1,14,0,11,0,0,29,17,
    0,10,9,2,1,5,0,30,17,0,13,7,0,23,3,0,0,0,0,0,28,19,8,0,22,0,16,20,0,0,
    18,17,5,0,0,3,15,23,20,6,0,11,0,7,0,0,16,0,0,0,0,27,0,8,0,0,22,28,12,1,
    7,0,11,4,0,30,0,16,0,14,0,0,6,0,26,0,17,0,20,0,0,12,0,5,10,29,9,22,13,0,
    0,0,17,15,0,0,0,24,27,9,0,22,4,2,23,29,0,0,16,0,11,5,0,0,0,18,26,3,0,28,
    0,15,0,0,0,0,8,0,0,25,0,1,18,19,21,7,23,0,28,22,27,0,14,0,13,10,0,17,30,0,
    15,0,22,0,12,0,0,28,25,0,11,0,20,0,0,14,18,13,17,24,8,30,0,6,0,3,0,16,0,23,
    0,21,0,24,20,9,29,0,26,2,15,12,0,5,14,0,4,23,0,0,0,0,0,3,30,28,17,0,19,0,
    0,0,14,0,16,2,0,13,10,28,0,25,24,0,19,3,0,29,0,30,20,0,21,0,0,26,23,0,1,8,
    0,18,13,0,11,0,28,17,0,0,27,3,8,30,4,19,0,6,7,0,0,0,0,22,14,16,0,0,25,21,
    14,0,18,25,0,0,6,21,2,0,20,0,0,12,0,16,0,28,13,0,0,0,5,1,0,15,29,0,24,27,
    0,2,0,12,29,27,0,25,4,0,0,0,7,28,6,0,30,0,0,11,16,8,18,0,1,20,0,23,0,0,
    27,23,28,20,0,0,18,26,7,13,0,0,0,0,17,0,1,0,0,0,12,25,15,0,6,0,5,19,8,10,
    16,4,0,0,21,11,14,0,1,0,26,0,17,10,28,13,0,5,12,0,25,0,29,23,0,8,0,0,9,0,
    0,0,27,0,13,1,12,22,0,26,30,29,0,24,0,21,0,0,18,0,0,14,4,2,17,5,11,8,0,0,
    4,0,25,27,0,12,7,29,11,10,21,0,30,0,0,0,0,22,0,28,18,0,9,0,23,0,14,0,0,2,
    17,13,0,7,14,16,0,0,0,8,2,0,29,0,0,0,0,0,0,23,15,3,10,30,0,4,21,0,5,20,
    9,0,8,28,0,13,21,0,16,18,4,24,0,6,12,20,15,1,10,29,0,0,19,0,0,0,0,25,0,0,
    29,20,0,0,28,0,0,0,0,0,8,27,23,0,0,30,6,0,24,10,0,7,3,15,12,2,0,13,11,9,
    0,9,0,0,30,17,24,0,5,27,0,20,14,4,2,0,10,0,0,3,13,15,0,19,16,0,0,1,0,7,
    24,28,15,6,22,0,3,0,0,17,0,26,16,0,0,10,5,8,21,0,0,2,0,0,19,0,0,11,18,0,
    8,11,0,13,19,6,0,0,0,0,9,15,0,0,18,22,0,4,0,17,5,29,0,7,0,24,25,0,3,0,
    0,14,0,19,18,10,11,0,0,0,16,17,2,0,22,0,0,21,8,25,0,0,26,0,4,0,13,9,28,6,
    0,5,0,0,0,0,0,8,0,15,19,6,22,0,10,25,2,18,26,21,0,0,13,16,27,0,28,0,20,14,
    21,0,0,8,0,0,0,7,28,23,24,0,0,0,15,2,14,0,5,20,9,18,0,11,3,17,19,27,0,0,
    19,0,20,21,10,29,0,0,14,0,0,8,0,0,0,24,11,0,0,4,3,0,2,13,7,9,0,12,27,18,
    28,19,0,0,23,22,4,11,15,0,25,16,5,26,0,0,0,24,30,0,0,0,20,17,0,0,1,6,0,0,
    1,0,4,16,0,7,25,14,29,0,0,0,0,15,8,0,27,9,11,2,6,22,0,0,20,12,3,0,0,0,
    2,0,23,17,24,0,16,18,0,3,14,0,0,27,30,4,0,25,22,5,21,0,0,0,15,0,0,0,0,26
  };

  const int d30_374_8[] = {
    30,
    0,28,0,0,18,22,0,29,24,5,0,7,0,4,2,3,14,0,0,0,17,15,0,0,1,13,0,10,0,11,
    11,0,16,0,6,0,0,19,15,0,17,29,9,0,0,0,8,27,0,0,13,1,2,24,0,0,20,18,0,5,
    6,0,30,17,24,0,0,11,23,0,25,2,0,19,0,10,0,0,15,8,20,0,27,0,5,0,26,14,12,0,
    13,17,0,0,0,5,0,0,11,0,0,10,24,14,0,26,28,22,7,1,0,16,0,15,12,0,25,27,8,0,
    0,27,5,0,0,3,9,0,22,8,0,30,26,0,13,19,0,0,6,0,29,0,24,0,0,11,7,21,4,14,
    9,0,18,27,4,0,12,0,0,0,13,11,15,7,10,14,17,16,0,2,3,0,0,0,8,24,0,0,19,0,
    0,0,4,0,13,10,0,27,28,6,5,0,0,0,1,7,0,15,11,0,0,26,21,22,17,0,0,9,3,18,
    0,0,23,18,10,0,4,0,26,30,0,21,2,20,6,16,0,29,28,0,0,27,0,13,0,15,3,11,0,0,
    25,7,0,1,0,12,15,0,0,3,0,0,0,16,0,24,6,19,4,13,26,0,0,0,18,0,22,0,11,27,
    0,20,0,19,30,14,0,6,2,11,0,9,13,0,24,8,0,0,0,0,4,0,5,16,21,0,0,3,10,25,
    0,11,7,20,0,18,0,1,0,19,21,15,22,27,28,0,0,6,0,25,0,12,14,0,23,8,0,0,29,0,
    28,21,3,0,2,17,0,20,0,12,0,1,0,0,26,15,0,5,0,18,0,13,0,25,11,16,24,0,0,0,
    0,2,0,11,1,0,21,0,18,0,16,26,30,12,22,27,5,24,0,0,0,0,0,4,14,25,0,17,0,13,
    21,0,8,0,25,20,29,22,1,24,30,19,3,10,27,0,0,0,17,0,0,0,0,23,0,6,0,0,0,12,
    0,0,0,12,17,0,26,14,0,0,18,22,0,23,16,0,15,8,25,6,0,9,19,10,24,30,2,0,0,0,
    30,1,0,0,0,0,0,26,4,17,15,3,0,0,5,0,29,0,24,0,2,0,16,14,22,18,0,0,13,19,
    26,6,10,0,0,0,14,25,19,7,4,0,0,13,0,17,0,28,0,22,0,0,1,0,30,27,0,0,23,9,
    2,0,22,0,0,8,3,0,21,1,0,0,14,26,0,0,0,0,9,24,0,29,13,17,0,7,11,0,6,20,
    15,0,26,5,0,21,28,0,12,2,19,0,0,0,3,0,4,30,0,16,11,23,0,0,6,1,0,8,7,0,
    0,23,0,0,5,26,7,16,0,0,2,18,4,21,0,9,10,12,0,3,19,0,28,20,0,14,0,13,0,0,
    14,0,29,24,11,15,30,0,0,0,22,0,0,18,23,21,0,0,20,28,0,4,8,5,0,0,9,0,0,10,
    1,0,13,16,0,25,27,24,0,0,9,0,0,0,20,0,12,0,14,26,30,22,15,8,0,0,0,19,18,6,
    8,24,6,25,29,0,0,0,9,14,11,0,16,0,0,0,22,18,0,0,21,0,30,28,0,3,13,20,15,0,
    0,0,12,8,0,11,0,2,10,0,6,0,5,0,21,0,0,0,30,17,15,0,26,0,13,0,16,25,20,1,
    27,14,0,30,15,0,8,0,0,0,0,0,23,6,0,29,20,0,1,7,0,19,0,2,0,21,12,26,28,17,
    29,0,0,2,27,1,0,0,3,4,0,0,21,0,0,6,19,0,0,12,8,28,0,0,25,5,18,16,0,15,
    18,25,0,15,0,23,24,5,0,21,0,13,10,29,9,0,30,0,0,11,0,0,0,7,0,28,27,0,2,8,
    0,19,11,26,0,0,0,0,0,25,12,0,18,0,0,4,23,9,27,0,10,21,17,0,20,0,1,15,24,0,
    0,29,0,0,16,0,17,3,20,9,0,0,12,25,0,22,11,1,21,0,5,2,4,19,0,0,28,0,0,0,
    0,12,0,4,22,0,11,7,0,0,3,5,0,28,8,0,18,25,2,15,27,14,9,0,0,0,0,24,0,0
  };

  const int d30_374_9[] = {
    30,
    0,27,0,0,12,0,28,0,29,16,3,25,0,5,18,26,10,30,11,2,22,0,0,4,0,21,0,14,0,0,
    26,13,18,0,0,5,0,9,0,12,0,0,28,0,0,14,16,23,0,11,8,0,25,20,30,27,2,0,0,1,
    18,29,13,0,10,28,1,0,0,2,22,0,9,0,0,0,0,0,19,21,16,0,0,27,0,26,0,15,20,11,
    7,0,0,20,8,0,4,0,0,27,30,28,0,15,0,3,12,0,0,14,9,6,23,5,0,18,21,0,16,0,
    11,0,5,2,30,14,0,0,16,0,20,23,0,0,21,0,4,10,6,0,0,22,19,0,0,9,3,13,0,0,
    29,2,0,0,19,13,8,0,4,11,5,0,0,0,20,22,27,7,0,0,0,14,10,23,0,0,1,0,25,0,
    0,0,0,0,9,26,3,21,0,6,19,0,8,0,30,17,28,15,29,25,0,0,0,1,11,7,0,16,0,2,
    0,15,23,0,0,17,11,19,0,0,0,0,1,26,27,0,25,28,0,0,20,13,22,29,6,12,4,0,2,0,
    0,0,0,29,0,22,0,0,21,20,26,0,30,2,10,0,0,1,0,0,5,9,6,17,25,19,0,24,13,27,
    0,30,0,9,22,0,12,15,17,23,10,6,0,0,0,0,26,0,0,5,0,0,8,28,19,2,0,29,3,13,
    24,23,2,0,0,29,25,0,3,0,0,14,4,0,0,9,0,0,0,0,6,21,0,18,0,30,26,19,17,5,
    15,21,11,24,0,0,29,0,26,0,1,17,3,23,25,30,7,8,0,0,0,5,0,22,0,0,0,0,10,19,
    3,0,0,23,0,16,17,28,0,19,6,21,18,13,0,8,0,0,0,0,11,1,0,14,15,0,29,9,0,0,
    27,14,0,19,24,0,2,0,15,13,8,7,22,0,23,0,0,0,17,0,1,18,0,0,0,16,10,0,26,0,
    0,19,0,30,6,12,0,20,2,24,0,22,16,0,14,10,0,11,15,0,0,7,29,0,1,28,0,0,0,0,
    0,7,28,0,0,0,13,16,6,0,9,10,0,30,0,20,23,0,0,18,0,11,17,0,29,0,14,0,5,21,
    20,0,0,7,17,0,27,13,0,0,0,0,25,1,3,0,0,22,0,28,30,16,5,0,12,15,0,23,18,8,
    0,5,0,12,0,21,0,11,0,26,4,0,0,29,0,0,17,2,3,13,15,0,20,10,7,0,25,0,23,6,
    17,4,30,25,13,0,0,18,24,0,0,9,0,11,0,0,19,0,0,27,29,0,28,0,0,1,6,20,22,23,
    6,0,14,0,18,2,0,8,25,15,0,0,26,0,29,0,0,21,23,0,0,10,27,0,28,3,0,4,0,12,
    13,0,16,10,25,23,22,4,11,0,7,19,12,8,0,0,0,0,1,24,14,0,0,0,0,0,27,0,21,18,
    0,0,3,18,0,24,0,27,12,0,14,11,0,0,8,1,0,6,0,30,17,26,0,0,10,0,28,5,0,7,
    19,16,15,0,27,0,0,0,8,25,0,0,0,7,2,6,11,4,5,23,0,29,3,0,26,10,0,1,0,0,
    0,3,0,0,23,0,26,1,0,0,28,13,0,10,0,12,6,25,9,0,2,15,0,19,27,0,0,21,0,20,
    4,17,25,0,0,11,14,0,0,0,0,20,10,0,13,28,3,0,18,19,7,0,30,26,0,22,8,12,0,0,
    0,0,24,11,0,6,23,3,22,0,0,1,0,0,12,0,8,0,25,10,0,4,16,0,14,0,19,2,0,9,
    0,0,12,15,5,0,0,26,27,10,0,8,20,21,7,2,1,19,22,0,0,28,0,25,24,0,0,0,30,0,
    21,20,29,8,1,18,0,0,0,17,0,26,6,9,5,0,0,0,2,0,12,0,7,0,0,0,13,28,27,15,
    9,25,0,0,0,7,0,0,10,3,0,0,13,27,15,4,0,0,20,12,21,0,14,6,5,0,22,0,11,0,
    0,0,27,22,0,0,0,6,0,7,24,0,23,12,0,29,0,20,30,17,0,0,0,13,9,14,0,10,15,16
  };

  const int d30_374_10[] = {
    30,
    25,9,24,0,26,5,0,27,0,18,21,0,0,15,13,0,7,28,3,0,1,0,14,2,0,6,0,0,20,0,
    0,0,29,22,2,10,0,0,0,20,23,0,1,0,21,16,30,0,4,0,0,14,8,0,0,12,0,6,5,25,
    0,0,0,1,25,0,0,4,29,30,0,22,3,11,0,0,23,8,18,0,15,0,9,0,26,13,17,0,14,2,
    0,20,0,9,6,28,0,15,21,23,0,13,0,0,0,0,0,0,0,3,5,16,25,22,4,24,11,2,19,0,
    19,3,9,14,0,0,29,0,16,7,15,0,18,8,0,13,24,12,0,26,27,0,11,0,5,0,0,0,0,21,
    0,14,0,28,9,0,4,0,13,0,26,25,29,21,27,22,0,0,0,12,0,5,24,0,0,16,15,23,0,30,
    20,16,0,5,15,0,8,24,17,0,13,28,30,0,0,18,0,11,1,25,0,0,6,0,0,22,0,0,12,4,
    29,19,10,0,0,9,13,3,23,22,5,0,0,0,0,14,0,18,0,2,0,20,0,27,0,0,6,8,25,24,
    0,5,0,16,0,0,18,22,0,9,0,0,10,20,19,0,6,1,27,4,2,11,0,0,17,0,0,24,7,23,
    0,26,16,6,11,17,0,0,0,15,10,0,12,4,0,30,25,0,23,28,0,18,19,0,24,0,20,0,0,0,
    16,0,11,21,0,27,0,25,20,28,19,0,0,30,14,0,8,23,0,0,9,0,22,0,15,17,0,29,0,0,
    0,12,19,0,20,24,1,0,14,0,8,0,27,2,0,0,18,0,29,0,0,7,0,30,3,28,13,15,6,0,
    11,1,13,8,10,14,0,0,0,0,0,16,26,9,30,20,29,0,0,23,3,0,0,5,0,0,12,7,0,22,
    3,0,2,0,0,0,0,0,28,6,11,20,14,13,29,0,21,16,7,1,22,0,0,4,0,0,0,26,0,18,
    0,21,5,27,0,22,24,18,0,0,0,0,0,1,26,9,13,19,0,29,6,12,0,0,20,11,0,0,8,0,
    23,0,0,0,28,0,25,26,6,27,30,18,7,0,24,0,0,29,22,0,21,0,15,0,14,5,0,17,0,0,
    14,30,0,0,23,0,7,10,0,0,0,2,22,27,8,25,5,0,20,0,0,3,0,18,29,0,0,1,17,26,
    0,0,21,23,4,0,14,0,0,1,18,29,0,0,16,0,0,26,28,19,10,0,0,7,0,2,8,30,27,0,
    0,23,3,0,18,6,0,0,0,12,0,21,0,0,15,7,22,0,11,0,28,30,26,16,0,19,29,10,0,0,
    28,13,0,0,0,19,21,0,3,0,1,0,15,0,0,26,0,0,25,16,0,8,10,6,23,0,5,0,18,27,
    6,18,30,15,0,1,10,14,2,13,0,3,0,19,0,0,4,9,26,0,0,0,7,21,0,8,0,0,24,0,
    0,29,12,0,0,4,0,19,11,8,0,15,0,0,0,0,0,2,5,27,14,0,0,1,7,9,22,25,0,13,
    5,0,0,0,0,15,0,6,19,0,0,8,23,0,25,10,0,22,13,21,0,27,17,9,1,0,24,0,11,7,
    4,0,26,17,0,0,27,29,0,0,24,10,0,28,18,2,19,7,0,0,23,15,0,25,22,30,16,0,0,0,
    17,6,1,18,3,0,12,0,0,24,0,0,0,0,0,29,28,0,16,0,0,4,27,0,8,14,9,21,0,19,
    8,0,0,25,0,0,16,9,26,0,3,27,0,12,17,0,20,0,6,10,0,0,0,24,0,7,30,18,29,0,
    30,17,0,0,22,0,11,0,0,0,0,4,0,0,3,21,0,5,0,0,29,2,18,23,10,0,7,16,28,1,
    0,0,17,0,14,21,23,16,9,4,0,12,28,5,2,19,0,0,0,13,8,22,0,0,18,29,0,0,0,20,
    7,0,28,0,19,3,0,8,27,0,14,0,2,10,0,0,1,0,0,0,30,25,23,15,0,0,0,22,26,6,
    0,0,0,7,21,16,6,30,5,3,20,23,17,0,0,28,0,24,0,14,0,0,2,0,25,0,19,0,1,29
  };

  const int d30_374_11[] = {
    30,
    21,28,27,0,0,22,29,6,3,11,23,0,0,0,0,0,15,0,18,9,26,8,0,0,12,30,0,4,1,0,
    19,0,3,0,24,30,0,18,0,1,14,8,6,26,22,0,0,16,0,0,0,0,4,0,0,20,17,0,7,9,
    30,0,20,14,0,17,25,3,22,2,0,9,8,24,0,0,28,15,0,23,0,0,29,4,26,0,0,27,0,0,
    5,21,23,0,27,0,1,0,0,12,0,0,19,0,0,6,0,20,16,30,15,10,0,14,11,0,0,0,22,4,
    0,3,0,23,21,0,0,0,0,28,24,13,5,7,11,16,20,6,0,0,25,9,0,18,0,8,1,0,0,0,
    22,0,4,1,0,6,0,12,18,0,29,23,30,13,0,24,25,0,28,0,10,0,0,0,21,0,26,0,15,2,
    23,14,19,24,0,16,21,5,0,15,0,0,4,0,6,0,0,0,0,25,2,13,3,0,20,17,8,22,0,0,
    6,23,0,0,20,29,26,0,10,27,19,12,17,18,0,0,0,0,0,5,7,22,24,30,0,0,2,0,13,0,
    9,29,7,11,17,0,0,27,0,0,30,25,0,5,0,0,6,0,21,20,4,0,0,8,19,13,0,0,23,18,
    0,0,12,5,25,0,17,2,7,26,9,0,28,0,1,0,24,3,0,0,0,30,0,21,0,0,14,18,0,11,
    0,26,29,15,0,1,0,19,9,0,0,11,12,27,23,17,0,21,0,0,16,7,5,0,0,22,0,0,28,0,
    28,0,0,0,13,0,4,0,0,0,0,17,1,0,0,25,0,29,30,21,19,24,15,3,6,0,18,14,5,0,
    0,5,0,21,0,18,3,20,6,0,7,0,23,25,0,10,16,0,0,11,0,15,9,2,27,0,0,0,0,29,
    0,27,0,10,1,0,0,23,0,0,22,20,11,0,9,14,13,0,3,0,24,0,21,29,0,18,19,15,0,26,
    0,0,17,4,0,0,9,7,11,3,0,16,10,21,26,0,0,19,0,0,0,0,27,5,0,1,23,20,8,28,
    20,0,0,0,2,0,13,0,8,0,0,5,0,14,3,15,0,24,11,17,0,25,16,23,0,21,0,29,18,6,
    0,0,28,22,6,11,10,21,12,0,25,0,3,0,2,0,0,0,0,0,27,4,1,26,0,9,24,0,20,15,
    0,16,18,0,0,0,27,0,30,8,4,15,0,6,0,22,0,17,26,0,0,20,0,1,23,29,10,9,0,0,
    0,0,0,0,26,0,0,0,14,17,0,6,2,3,21,13,27,0,23,0,22,18,0,0,29,24,12,0,9,7,
    1,12,0,30,22,0,0,0,4,29,16,0,24,0,0,19,8,0,25,14,0,0,0,0,13,6,9,5,2,23,
    25,0,9,0,0,28,6,29,0,30,0,18,16,11,0,27,10,7,5,0,0,0,0,15,4,0,20,0,0,13,
    2,0,21,7,0,24,20,26,0,5,0,0,29,9,0,0,23,28,8,0,30,11,10,25,3,0,0,0,17,0,
    14,22,0,0,19,20,0,0,25,0,5,2,0,0,0,0,18,10,1,4,8,27,26,6,0,0,0,24,21,0,
    0,9,10,0,3,0,19,0,16,0,18,4,0,1,29,21,22,5,7,27,0,0,0,0,17,0,11,12,0,0,
    0,4,13,29,0,15,0,1,0,0,26,0,0,30,12,8,7,0,0,0,0,28,0,17,5,16,0,19,3,20,
    0,11,0,20,16,2,22,24,0,10,1,0,0,4,8,7,0,23,6,18,28,0,0,0,25,0,0,17,19,0,
    10,17,14,0,5,23,0,30,21,18,0,0,0,0,27,0,4,0,0,7,0,1,19,0,0,2,29,3,0,12,
    0,2,0,0,18,8,0,0,0,13,21,0,0,28,20,11,0,12,22,3,9,6,25,10,0,5,0,16,0,27,
    15,20,0,12,14,7,0,0,23,19,0,3,0,0,13,30,11,8,4,16,0,0,17,0,28,25,0,0,0,24,
    24,0,0,9,0,26,11,0,29,0,3,10,0,0,14,0,19,25,0,22,6,0,30,0,18,7,27,1,0,16
  };

  const int d30_374_12[] = {
    30,
    30,24,0,0,0,7,0,17,0,27,23,3,6,2,10,18,0,0,29,13,0,0,26,11,0,9,0,0,1,21,
    13,0,0,22,3,10,18,24,0,30,0,25,16,19,14,11,0,0,21,1,0,0,0,9,0,6,0,17,27,0,
    0,30,11,28,16,0,25,0,22,0,0,0,12,0,19,5,0,0,0,0,20,18,17,2,8,3,0,9,0,6,
    0,6,24,0,0,17,5,0,26,3,25,2,21,0,0,0,27,0,0,29,28,0,10,0,0,11,16,7,18,0,
    7,14,10,27,0,1,15,0,0,17,0,13,0,6,29,0,3,0,8,20,0,21,30,0,0,0,26,18,9,0,
    0,16,0,0,0,0,0,0,11,15,2,26,29,25,5,0,0,7,20,17,9,10,0,0,19,0,13,28,0,12,
    0,0,0,0,0,16,29,6,14,0,0,15,8,0,24,28,0,0,13,22,7,0,0,12,9,17,0,25,19,23,
    0,23,0,12,11,0,0,0,10,0,14,20,0,13,17,9,16,6,0,0,18,1,0,7,0,0,8,0,26,19,
    1,0,0,14,29,0,0,15,4,0,17,0,25,3,22,12,0,10,0,0,21,0,0,24,20,0,23,30,6,26,
    16,28,0,7,14,0,11,13,15,25,6,5,26,20,0,0,0,27,0,0,0,0,8,23,10,0,0,0,29,0,
    23,17,9,5,22,19,21,0,0,0,0,18,0,16,27,0,15,12,0,26,13,3,0,28,0,0,7,0,11,0,
    20,0,8,0,30,5,24,0,7,28,0,0,0,0,18,17,0,0,3,0,14,0,25,0,15,19,11,10,23,9,
    9,25,20,0,0,8,0,0,2,0,0,0,5,0,3,0,23,1,7,0,0,22,24,0,26,30,27,12,0,11,
    8,15,21,13,0,6,0,19,16,10,0,23,0,4,7,0,29,26,0,0,0,0,0,1,0,20,0,5,30,3,
    0,26,5,29,13,0,0,7,21,9,0,0,24,0,0,0,20,19,16,0,10,27,0,8,0,28,14,0,12,25,
    0,4,7,0,24,26,8,9,13,18,10,29,22,27,0,23,0,0,0,0,1,0,0,20,0,25,21,2,0,0,
    12,0,14,0,7,0,0,0,0,24,15,0,18,17,0,19,26,30,22,9,2,25,28,0,16,0,3,0,0,27,
    11,0,0,19,17,0,16,23,3,0,0,12,0,14,8,15,24,13,0,0,0,28,20,18,0,0,6,1,0,0,
    0,9,0,1,0,21,14,0,0,0,0,0,30,28,11,20,7,23,0,24,3,13,19,0,29,8,12,0,16,0,
    0,0,19,26,0,0,27,5,18,0,9,0,2,0,0,7,22,0,17,11,0,8,3,0,0,23,20,14,0,4,
    14,0,0,18,0,2,26,8,25,5,29,10,0,0,0,6,13,0,9,0,4,7,11,0,17,0,1,0,0,0,
    6,1,12,0,28,3,0,30,0,0,18,0,0,29,23,25,0,4,2,16,0,0,21,14,27,0,0,0,22,5,
    0,11,23,24,5,0,0,29,0,0,0,7,13,9,0,0,25,0,28,19,0,16,1,17,3,0,0,21,20,2,
    0,7,29,0,10,0,30,0,6,23,27,0,28,24,13,0,17,9,19,25,0,5,16,4,0,0,0,0,0,18,
    5,0,0,11,8,30,20,0,28,13,1,0,3,0,12,0,9,0,27,6,22,0,29,16,25,0,0,0,0,0,
    29,12,6,20,9,0,0,22,19,0,7,0,0,0,0,0,4,3,0,27,30,11,5,0,13,26,0,15,0,0,
    0,0,22,9,18,12,0,2,0,0,20,14,0,8,0,0,1,5,26,30,11,6,0,0,23,24,0,4,0,0,
    0,0,16,0,19,4,28,27,0,14,30,0,0,0,0,21,0,15,5,3,25,26,0,29,6,2,0,0,17,24,
    22,19,0,0,0,29,23,12,0,1,0,17,14,30,21,26,0,8,0,0,0,20,0,0,18,15,10,0,7,28,
    27,0,26,3,0,9,0,28,0,16,21,24,0,0,0,4,10,0,0,0,5,0,13,15,0,12,2,29,8,0
  };

  const int d30_374_13[] = {
    30,
    0,13,6,0,16,0,0,17,12,19,0,5,20,0,0,3,0,22,25,27,0,18,1,0,10,30,11,23,0,0,
    8,5,19,21,0,2,0,0,0,0,0,0,18,12,23,14,7,0,0,0,27,3,0,0,15,13,0,26,29,17,
    0,0,14,0,27,6,1,20,26,0,4,29,12,0,19,0,13,0,0,0,0,16,5,2,7,0,0,0,10,21,
    24,0,0,22,0,25,0,4,1,0,0,0,21,11,0,13,10,0,27,6,0,29,26,0,3,15,18,2,30,0,
    0,20,8,0,26,30,3,0,0,0,0,6,14,0,0,0,18,13,23,1,0,21,0,10,0,11,19,5,24,16,
    30,0,27,4,0,0,5,9,0,1,16,0,0,8,20,0,0,10,19,15,6,0,0,17,13,0,0,24,12,28,
    0,9,10,26,20,7,15,18,0,0,0,0,24,0,17,4,1,12,3,0,16,28,0,5,0,0,0,0,23,0,
    0,1,0,15,0,23,18,16,6,13,0,0,7,0,30,9,27,11,28,0,0,0,2,14,0,5,22,0,0,26,
    28,0,0,10,1,0,0,15,2,6,22,30,9,4,21,5,0,0,0,0,26,0,0,12,0,0,24,3,16,8,
    0,23,0,0,0,27,6,24,0,2,0,16,0,20,11,12,0,0,22,0,0,0,13,0,17,7,1,15,26,30,
    0,12,24,30,0,1,10,0,23,0,25,0,8,16,0,0,17,5,6,11,0,0,15,27,19,0,0,0,9,14,
    4,2,0,23,24,16,30,5,0,21,11,10,0,17,3,28,0,0,14,19,0,0,0,0,0,0,29,22,15,0,
    21,19,0,0,14,24,0,2,4,0,18,0,0,27,22,0,5,17,0,13,7,25,0,0,30,29,0,16,0,0,
    22,24,29,0,3,0,23,28,19,5,10,0,0,7,27,0,0,0,0,4,8,0,21,0,2,0,16,14,18,0,
    0,0,20,9,19,0,8,0,18,0,24,1,0,30,0,6,21,0,10,5,3,15,7,0,0,0,14,0,0,27,
    17,3,12,24,0,14,0,19,0,23,0,18,15,22,0,20,0,29,9,26,5,0,0,8,21,0,0,0,0,13,
    1,10,0,0,8,28,29,0,30,18,5,27,0,0,2,0,15,3,0,0,22,0,0,13,0,0,25,0,7,11,
    0,0,0,0,2,10,7,0,11,9,14,0,0,0,0,29,19,0,18,0,23,4,27,25,12,17,21,20,0,5,
    5,0,15,7,0,8,25,0,0,4,0,9,0,0,24,0,0,26,0,0,1,27,19,21,14,0,0,18,11,3,
    3,16,7,0,0,19,0,27,0,8,30,20,0,26,0,0,22,6,0,24,10,23,11,4,0,0,0,0,14,0,
    20,17,2,6,9,0,12,0,13,0,0,26,1,0,0,30,0,8,0,0,0,10,0,18,25,16,0,11,0,15,
    0,15,0,0,0,0,9,3,5,7,6,19,10,14,25,0,0,0,24,29,0,0,17,0,16,28,0,8,0,23,
    14,0,13,27,10,0,0,0,20,0,3,0,0,2,12,25,8,18,0,0,28,0,24,1,0,23,15,30,0,0,
    2,0,0,0,0,0,20,0,27,0,0,14,17,9,6,21,0,24,4,25,18,19,8,0,22,1,0,0,0,29,
    16,0,9,28,0,0,24,25,0,26,0,22,29,0,0,11,0,0,8,21,14,30,12,20,0,6,13,0,5,0,
    9,7,11,8,25,0,0,23,0,0,20,0,27,13,5,0,24,21,2,0,30,0,0,0,28,4,0,0,1,18,
    0,0,18,0,0,0,0,30,16,28,29,4,0,25,0,0,0,15,1,10,0,0,22,6,0,26,5,17,2,7,
    0,0,0,16,15,18,0,10,29,30,7,21,0,0,4,0,14,1,0,17,0,2,0,0,11,25,3,13,20,0,
    11,6,1,25,21,0,13,0,0,15,27,0,30,0,0,23,2,19,0,22,0,17,20,29,18,0,4,0,0,0,
    0,0,0,0,4,9,0,0,14,11,1,3,23,6,13,10,30,0,15,20,12,26,25,0,0,8,28,0,0,0
  };

  const int d30_374_14[] = {
    30,
    0,15,20,7,9,16,12,2,5,10,13,29,0,28,0,24,25,0,26,19,0,14,0,22,0,0,0,0,0,0,
    20,0,8,0,21,1,3,0,0,0,7,19,0,0,14,0,4,23,12,27,15,18,0,0,0,0,6,0,9,2,
    8,0,28,0,11,15,29,0,3,0,0,0,0,0,16,0,0,0,19,6,9,27,22,24,23,10,20,4,0,0,
    21,17,0,28,0,25,0,30,0,0,0,2,19,0,4,10,0,5,11,0,18,1,8,0,16,0,26,0,7,24,
    0,0,0,14,20,10,0,1,16,25,18,0,12,11,23,28,0,22,0,8,2,3,0,17,0,0,29,9,0,0,
    0,0,0,3,16,0,26,0,0,0,20,8,25,0,13,0,9,2,0,0,10,11,27,5,24,28,19,6,0,0,
    27,9,7,15,0,0,1,16,0,30,0,0,13,18,24,0,0,11,0,0,0,22,0,0,3,5,8,2,14,23,
    0,28,16,6,0,0,0,0,12,23,24,14,0,25,30,0,11,29,9,0,19,0,0,2,0,15,7,5,20,0,
    30,12,6,26,0,13,0,0,24,2,10,21,11,0,0,23,0,0,0,28,14,0,25,0,9,0,3,22,0,4,
    16,1,0,30,0,9,5,19,7,11,21,0,28,0,0,29,0,0,0,12,17,6,0,0,0,0,4,10,27,18,
    0,11,0,29,0,0,24,23,9,27,0,22,20,26,3,0,15,14,0,13,0,0,0,6,2,17,25,0,0,12,
    0,0,13,0,0,27,15,25,18,0,14,0,24,0,19,26,17,0,0,1,0,0,12,4,0,8,21,28,0,10,
    29,27,3,12,5,0,0,0,0,0,0,28,22,14,0,6,16,0,25,0,8,0,7,20,13,0,0,21,17,15,
    0,10,19,0,0,2,4,0,0,0,11,15,0,17,25,0,6,12,24,0,0,9,0,21,5,27,30,29,16,0,
    28,0,24,0,0,12,0,0,4,7,0,27,1,0,9,22,18,0,0,26,3,25,0,0,30,0,17,15,19,0,
    3,0,30,23,0,0,18,28,2,0,0,16,0,0,1,0,27,0,10,20,0,21,11,0,8,19,0,0,6,13,
    12,0,0,0,0,0,21,17,29,15,26,0,2,16,0,18,8,28,1,30,20,0,4,23,0,22,0,0,0,0,
    26,0,1,10,19,0,0,11,0,0,9,12,15,8,5,3,0,0,17,0,21,2,24,7,0,0,27,13,0,0,
    6,0,0,22,29,0,20,18,0,3,0,7,0,0,27,13,0,0,0,24,23,17,0,12,25,14,10,0,8,21,
    0,7,10,17,3,23,22,0,13,16,25,0,21,27,18,20,0,15,0,0,0,0,0,8,26,0,0,0,24,6,
    5,0,0,16,7,30,9,0,28,14,0,0,0,29,0,1,10,26,3,0,0,20,23,0,6,11,0,19,0,0,
    0,13,4,9,0,17,0,22,15,8,29,0,0,1,0,0,0,0,18,11,0,24,3,0,0,30,0,20,10,7,
    18,20,0,0,15,29,0,0,0,0,4,25,23,0,0,2,5,0,14,17,12,13,9,0,10,0,24,0,0,1,
    0,24,9,0,22,0,0,8,11,29,19,0,17,23,0,0,2,25,0,18,27,0,21,28,0,13,0,0,26,5,
    19,0,22,0,6,14,0,13,0,4,0,23,0,2,0,0,12,24,21,0,0,7,0,15,17,0,0,30,1,29,
    17,6,0,0,18,0,16,24,0,0,0,30,0,0,7,12,22,10,8,23,0,0,0,0,15,21,9,14,2,0,
    0,25,23,13,2,3,14,0,6,17,0,0,30,0,0,8,1,27,7,29,0,0,10,0,19,26,0,0,0,9,
    25,14,0,8,0,0,0,20,30,0,1,3,18,10,0,17,0,0,22,5,0,0,28,19,0,2,0,16,15,0,
    0,23,21,0,27,7,28,3,8,24,22,0,0,12,0,0,0,19,0,0,1,0,17,14,11,4,0,0,13,20,
    0,0,0,0,13,8,0,0,0,0,2,0,10,4,21,0,19,16,23,0,6,5,26,9,0,0,14,7,25,11
  };

  const int d30_374_15[] = {
    30,
    0,2,0,1,0,10,0,17,16,11,27,0,21,0,3,0,4,0,24,0,0,19,7,14,0,30,15,22,0,12,
    0,0,9,15,6,24,4,16,20,0,25,7,0,23,1,28,0,0,27,0,22,0,3,0,26,0,8,0,0,0,
    4,0,0,14,19,28,29,0,0,0,0,0,1,5,13,17,16,0,0,9,0,10,30,6,7,2,21,20,0,0,
    14,0,5,22,4,23,8,0,0,16,0,26,6,0,0,2,12,19,9,0,0,21,15,0,0,28,0,0,1,20,
    0,20,8,0,13,9,3,0,30,12,17,1,15,26,16,0,0,27,29,6,0,0,0,24,0,0,0,0,0,5,
    0,24,0,5,7,19,25,12,29,0,4,0,0,0,0,21,23,0,30,26,0,0,0,28,2,22,13,0,15,18,
    22,1,4,0,23,0,5,0,14,0,7,24,0,18,0,25,20,12,0,13,11,0,0,19,27,0,0,2,16,0,
    0,16,24,13,0,0,0,0,0,23,8,0,0,21,6,0,2,5,0,22,0,3,12,0,20,9,7,11,0,17,
    6,13,0,0,24,5,12,23,0,7,21,0,0,0,0,15,11,30,0,3,29,0,0,2,0,0,10,1,0,4,
    18,0,13,0,0,30,11,0,5,25,23,20,0,24,0,3,0,14,0,0,12,0,29,26,15,19,1,4,0,0,
    0,0,20,2,0,16,17,0,0,0,0,29,11,1,19,27,0,0,0,12,30,4,0,5,21,0,0,24,25,15,
    24,0,7,16,14,0,0,2,23,0,3,30,0,29,0,0,17,26,5,0,0,15,22,0,0,0,0,8,10,11,
    0,26,0,0,10,27,2,30,0,13,29,25,0,12,0,19,0,7,1,11,5,18,16,0,0,0,0,0,23,28,
    16,18,15,6,11,17,0,0,4,20,2,0,25,0,0,0,19,0,0,0,27,29,0,0,23,8,3,30,28,0,
    0,3,0,7,0,0,24,29,0,0,19,0,8,6,23,0,0,4,11,20,15,16,18,9,14,0,0,0,17,0,
    9,6,0,11,0,29,20,7,0,18,0,0,12,0,17,0,8,0,28,1,0,13,19,10,25,0,26,0,21,0,
    15,0,26,0,9,0,0,0,7,29,22,0,2,28,4,12,10,3,8,0,0,0,0,23,1,27,0,18,13,0,
    3,8,0,20,0,0,0,24,6,0,0,0,16,27,18,14,9,15,25,28,10,11,0,0,0,1,0,0,0,22,
    23,0,29,21,0,0,0,10,22,1,15,14,0,0,7,0,3,0,0,2,18,0,17,20,12,0,5,19,6,0,
    17,29,23,0,2,0,13,1,0,4,0,6,20,0,0,0,0,0,3,5,14,0,11,16,0,24,25,10,0,0,
    0,0,0,0,0,8,0,3,13,28,0,19,27,0,22,24,0,21,0,0,1,9,0,12,0,6,16,7,11,26,
    0,23,0,17,1,0,28,19,9,0,0,4,18,0,0,5,26,6,20,10,0,0,21,3,0,7,27,0,0,13,
    12,0,10,9,0,0,21,5,19,0,0,16,22,14,27,20,0,0,17,0,0,28,0,8,0,13,4,0,0,25,
    21,14,28,0,0,15,1,4,0,27,0,8,7,30,11,0,5,0,0,0,25,12,2,0,18,29,22,0,0,0,
    0,0,17,0,27,0,0,25,0,0,16,13,0,9,10,11,0,22,14,29,3,0,4,7,0,20,0,5,24,2,
    28,0,0,0,0,22,0,0,11,0,18,23,19,4,0,0,0,25,0,24,2,5,0,17,13,15,30,6,0,16,
    30,4,21,0,20,0,18,0,27,9,0,0,0,25,0,0,0,0,26,19,8,0,5,0,6,17,2,16,22,3,
    0,12,18,30,15,20,0,28,2,0,0,0,0,0,8,0,7,29,4,0,17,0,1,0,19,0,0,9,14,21,
    1,0,0,3,0,13,14,0,8,10,5,0,29,0,28,18,15,24,6,23,0,17,0,0,4,26,0,0,12,0,
    0,10,25,0,16,1,0,11,0,24,0,22,23,2,9,26,14,0,0,0,21,27,0,0,0,0,18,28,5,0
  };

  const int d30_374_16[] = {
    30,
    26,0,0,0,13,5,9,4,0,0,0,1,3,0,0,6,17,0,30,19,29,0,25,12,20,21,0,10,22,0,
    10,0,22,18,26,0,0,21,0,0,20,0,30,5,7,19,0,0,0,0,1,3,23,0,24,9,0,17,4,0,
    6,21,0,0,19,11,28,0,0,10,0,0,0,8,0,16,2,3,5,13,14,0,26,0,22,0,0,30,12,0,
    29,23,0,6,8,0,0,14,24,0,0,0,0,26,0,25,27,22,20,7,0,0,4,28,0,0,11,21,18,30,
    0,8,2,19,16,6,7,0,0,0,0,0,0,30,9,0,23,24,13,18,26,0,0,4,5,11,0,1,27,0,
    0,0,30,22,29,3,17,0,19,0,24,0,0,23,4,18,10,0,0,6,0,0,21,13,0,1,28,0,7,5,
    0,11,0,28,0,0,29,0,0,9,30,16,7,4,21,0,0,18,2,3,0,22,24,0,17,6,5,0,0,25,
    2,0,4,29,0,20,14,0,17,7,0,0,25,0,18,0,0,26,10,0,3,0,0,0,27,13,21,8,23,1,
    15,16,19,26,0,0,0,5,0,29,0,4,22,0,8,0,0,28,0,1,23,13,17,0,0,3,12,0,0,27,
    20,0,0,0,0,13,0,8,26,30,0,25,2,14,0,10,7,17,0,0,18,28,11,19,0,0,0,12,29,23,
    0,0,5,0,0,7,20,15,6,0,3,18,24,28,19,0,8,27,14,0,10,0,22,25,0,0,4,0,17,0,
    11,0,9,23,0,10,0,12,8,18,1,0,26,7,0,29,0,0,4,0,0,16,0,0,0,15,25,0,24,19,
    0,0,0,0,1,0,0,10,25,2,27,0,0,20,23,24,4,21,0,14,0,9,0,30,8,0,6,15,13,11,
    0,0,11,0,25,0,23,27,28,4,14,22,13,0,0,0,15,0,0,9,0,29,3,0,19,17,2,6,0,18,
    21,0,0,30,28,8,13,23,0,20,0,0,0,0,27,4,25,0,0,16,0,0,29,14,15,0,0,18,3,9,
    1,27,29,0,0,21,3,0,14,24,0,17,8,0,0,0,9,0,25,22,5,20,10,0,0,0,7,0,0,13,
    24,19,0,15,18,0,0,22,0,26,7,0,12,13,3,0,0,0,8,2,27,0,1,6,25,4,0,0,0,0,
    0,14,0,0,24,15,16,0,0,0,10,0,1,3,0,27,0,13,0,0,9,2,0,8,0,5,29,22,25,26,
    19,12,0,0,0,24,1,16,22,11,6,10,28,2,29,15,0,0,0,5,25,0,9,0,0,0,27,0,14,0,
    28,15,0,1,14,22,2,18,0,0,23,9,0,17,0,0,0,19,12,26,4,0,0,0,0,24,30,16,6,0,
    9,0,15,14,2,0,19,0,0,28,17,13,20,0,0,12,18,0,0,0,7,0,30,21,3,8,0,0,0,22,
    0,0,18,0,0,0,0,0,12,25,26,23,21,0,20,5,13,7,0,29,0,19,27,11,9,10,17,0,0,3,
    25,13,20,12,0,17,0,30,0,0,0,28,0,10,0,0,5,1,0,24,0,4,0,29,14,0,23,3,0,8,
    14,26,3,0,0,16,0,0,0,5,0,0,0,0,28,0,19,15,21,23,0,8,20,18,12,30,10,0,11,0,
    0,7,6,0,3,27,0,2,18,0,0,8,0,0,15,1,0,9,22,17,20,30,0,23,16,12,0,24,0,0,
    0,0,0,3,10,26,27,13,1,0,19,30,0,15,5,0,0,4,23,0,11,14,0,16,7,0,0,9,0,2,
    0,2,16,25,12,0,18,0,5,3,13,0,10,22,14,9,0,0,29,0,0,26,0,0,28,0,15,4,0,20,
    8,24,7,2,30,0,0,0,13,0,12,27,4,0,6,17,21,0,26,0,28,0,0,3,0,18,0,0,1,0,
    0,18,0,0,4,28,0,0,16,12,8,21,0,0,0,23,6,30,11,0,0,10,0,5,0,27,1,2,15,0,
    0,20,13,4,0,0,12,3,9,17,0,15,0,24,2,22,11,16,7,0,6,27,0,0,0,0,0,5,0,0
  };

  const int d30_374_17[] = {
    30,
    0,0,14,25,5,23,15,16,21,3,0,2,0,24,0,26,0,27,0,7,11,17,0,0,29,1,0,20,0,0,
    0,0,0,3,8,0,23,0,4,0,28,9,7,22,0,2,30,0,19,0,13,16,0,0,1,15,29,0,0,5,
    0,0,28,11,30,29,0,26,9,2,19,0,0,0,8,0,21,15,7,25,0,12,5,0,0,14,0,17,16,0,
    6,0,30,0,0,1,25,0,0,0,11,7,0,14,0,5,2,21,0,0,3,0,27,28,0,10,8,26,18,0,
    0,4,18,0,9,0,29,20,26,19,27,12,0,0,24,17,0,28,0,0,10,0,0,0,2,0,25,13,3,8,
    0,0,5,18,4,0,11,0,0,15,1,0,28,7,16,0,0,30,12,0,26,0,0,9,13,0,27,0,23,10,
    7,30,8,0,0,0,12,25,0,26,20,0,23,17,11,0,0,0,6,16,5,24,14,0,0,0,0,18,19,13,
    29,16,0,15,26,12,3,0,24,0,21,0,19,0,0,9,0,20,0,5,8,0,18,0,0,28,0,23,14,0,
    24,6,25,0,16,0,19,0,8,0,0,0,12,27,0,0,11,0,9,14,0,29,1,21,15,3,17,0,0,0,
    8,10,0,0,15,9,0,24,0,11,0,0,0,28,19,30,1,0,0,18,0,6,0,14,3,4,22,29,0,25,
    0,15,2,12,0,0,21,22,18,10,6,25,0,0,0,20,0,5,27,0,0,14,19,0,23,30,24,0,1,0,
    0,17,0,9,20,7,0,0,29,12,3,0,18,2,15,0,5,0,25,0,0,0,21,8,0,0,10,19,22,27,
    0,27,15,0,12,30,0,0,0,0,0,14,17,19,0,1,29,7,22,0,0,4,9,10,0,21,26,28,13,0,
    16,0,17,0,22,21,0,18,2,0,0,20,11,13,3,0,24,1,0,0,29,0,0,26,25,0,12,0,10,9,
    19,0,10,8,17,0,0,13,16,7,0,0,26,0,22,0,15,12,1,0,0,0,0,24,27,0,6,0,4,30,
    1,0,0,0,23,14,17,27,13,18,9,28,15,26,21,0,0,11,8,29,0,0,25,12,0,2,0,0,0,0,
    0,26,0,14,0,18,0,10,0,0,25,11,6,0,7,12,19,0,2,22,0,9,0,0,21,0,15,4,17,16,
    0,1,0,0,10,8,4,21,0,20,29,0,0,0,14,0,17,25,26,0,0,5,0,2,0,27,9,6,0,11,
    0,0,0,21,0,13,18,29,0,5,0,17,20,0,23,15,0,4,16,6,19,0,0,7,28,0,0,30,27,0,
    21,3,9,22,0,20,14,0,0,0,26,19,2,0,0,6,0,17,15,0,12,0,13,18,10,0,5,0,0,23,
    27,0,0,0,28,4,13,14,0,22,5,18,30,0,0,24,20,0,21,9,0,0,8,11,0,25,0,2,0,0,
    4,0,0,30,3,27,7,0,11,17,0,0,0,0,0,29,26,22,23,15,0,0,0,16,0,9,0,25,6,2,
    14,0,0,0,24,26,0,0,17,23,0,27,0,10,9,0,0,6,28,12,2,7,0,13,20,19,0,0,0,21,
    0,20,24,2,0,0,0,30,28,0,12,22,0,25,29,0,0,0,0,13,1,10,17,0,4,0,19,3,9,14,
    15,23,19,26,0,0,27,0,0,0,0,3,9,21,30,0,28,0,0,17,18,1,0,5,14,22,2,0,0,0,
    3,18,0,0,0,5,0,19,0,0,23,24,13,29,10,21,8,0,0,28,20,30,2,17,0,0,0,0,0,15,
    22,21,6,7,0,0,0,5,20,16,0,0,0,0,0,19,0,29,0,3,23,11,10,0,9,0,18,8,15,17,
    11,0,20,4,0,0,0,0,5,0,10,0,0,12,27,23,25,18,0,19,0,13,22,0,26,16,0,0,28,3,
    0,5,26,0,0,0,20,1,6,0,0,21,16,18,0,0,22,0,0,0,4,23,11,0,12,8,7,10,30,28,
    13,9,21,28,6,2,0,11,3,25,0,0,27,0,0,7,12,0,24,8,17,26,29,0,0,18,0,0,0,0
  };

  const int d30_374_18[] = {
    30,
    8,29,0,16,0,21,14,7,28,0,0,0,27,13,0,12,9,0,0,22,11,0,26,0,6,0,30,0,19,5,
    13,0,7,14,9,23,0,0,20,5,17,8,24,26,0,22,0,0,27,12,21,0,1,0,0,0,19,0,10,0,
    0,0,0,3,7,29,16,11,0,0,0,0,25,14,17,30,8,20,6,15,0,26,0,2,21,0,0,0,18,10,
    0,20,14,4,0,0,25,22,9,29,30,10,19,0,0,0,0,0,16,0,3,23,2,0,8,0,1,12,0,15,
    10,7,0,24,0,14,19,29,30,0,0,0,0,28,0,0,0,0,11,6,0,0,12,25,27,18,26,16,5,0,
    0,0,19,13,17,12,1,0,27,0,9,0,0,7,0,29,0,0,23,0,16,20,3,0,18,30,0,22,4,0,
    0,11,30,0,22,16,0,1,0,8,24,0,0,0,5,0,29,2,0,25,20,19,0,26,0,4,27,6,0,9,
    0,3,24,26,12,7,4,0,0,14,0,0,21,5,8,2,0,1,0,19,0,0,22,9,15,28,0,0,13,0,
    20,17,0,0,0,30,0,16,22,3,0,4,5,1,11,23,19,0,21,0,0,0,0,0,10,0,15,2,0,13,
    12,5,4,0,6,17,26,0,0,0,11,0,0,19,0,28,3,23,0,0,1,0,0,13,16,14,0,10,15,22,
    0,6,9,25,0,27,2,20,29,4,1,19,0,0,12,3,21,0,0,16,28,0,0,0,0,10,0,0,0,17,
    30,0,8,7,2,0,22,0,17,20,0,16,26,29,24,0,5,28,12,0,0,9,0,0,0,1,0,27,23,0,
    0,0,21,30,10,0,0,0,26,0,0,17,23,12,9,0,0,13,2,29,0,6,24,15,0,7,0,0,11,4,
    15,30,0,19,0,0,0,13,5,25,3,6,29,0,0,0,0,10,18,7,22,14,16,0,28,0,0,4,27,0,
    3,0,0,21,0,0,8,0,0,0,22,12,0,0,23,9,0,24,30,1,26,11,5,10,7,29,16,0,6,0,
    5,19,0,11,28,24,0,0,0,10,4,0,8,0,7,0,16,21,0,0,15,18,29,23,30,25,0,0,0,27,
    0,28,22,0,25,20,9,0,23,0,15,29,0,8,0,0,27,12,0,0,0,0,0,11,0,5,18,14,17,6,
    14,21,23,0,0,0,0,19,7,1,28,13,0,30,18,26,0,0,0,8,12,27,11,0,2,0,25,0,0,20,
    0,0,17,0,29,0,28,12,16,0,18,11,2,0,0,10,0,6,0,21,0,24,25,19,0,3,0,0,7,30,
    0,10,0,0,26,19,0,4,25,0,0,0,3,0,27,11,0,0,13,0,17,22,7,20,0,15,9,24,14,0,
    17,9,11,0,0,3,21,10,0,22,0,26,20,27,0,0,7,0,14,0,8,0,0,18,13,0,2,0,0,1,
    0,0,0,12,0,1,13,9,0,0,2,0,0,6,16,17,20,25,0,14,24,5,28,7,3,0,29,30,0,0,
    19,4,26,2,0,0,0,18,0,28,27,0,16,0,3,7,1,15,22,17,25,0,0,0,0,0,0,11,30,12,
    4,0,6,0,27,11,0,0,0,13,0,0,14,0,21,1,22,3,0,18,23,0,0,0,0,26,17,20,25,2,
    2,0,0,0,3,0,15,8,0,16,20,1,10,9,25,27,13,17,19,0,0,0,4,21,0,0,5,0,0,0,
    0,0,2,6,0,15,0,0,1,11,0,24,18,0,29,0,25,0,0,30,7,8,14,27,26,20,28,5,0,0,
    26,1,0,27,13,5,0,3,15,30,25,22,0,18,0,6,0,14,0,0,10,2,0,8,0,19,4,0,0,0,
    6,0,20,0,30,0,5,23,0,17,0,9,0,0,22,0,14,18,28,0,0,29,27,16,0,12,0,15,0,11,
    16,24,15,0,23,0,11,17,0,0,0,7,6,0,0,0,28,0,5,3,0,4,21,30,19,0,12,13,0,0,
    9,14,0,0,0,0,0,0,8,23,5,21,0,24,0,18,0,29,10,13,0,15,0,0,12,22,0,3,28,25
  };

  const int d30_374_19[] = {
    30,
    25,0,10,2,26,22,0,11,0,6,0,0,0,3,24,5,0,0,4,23,20,0,0,1,0,19,13,15,0,0,
    0,12,28,0,0,3,19,13,0,0,0,25,27,29,1,0,0,0,16,0,10,26,0,20,0,0,11,21,2,9,
    7,2,26,25,17,0,0,0,20,13,0,0,24,0,0,11,16,27,0,10,0,5,0,8,3,0,30,12,0,0,
    0,17,0,0,30,4,26,0,13,12,23,28,7,0,0,29,0,0,15,14,0,8,11,0,19,0,0,18,21,20,
    30,0,16,0,23,0,20,0,9,3,14,0,26,24,5,0,0,21,0,0,27,22,13,18,15,10,0,0,0,0,
    9,20,0,0,14,21,0,0,0,22,0,0,1,5,30,10,0,0,26,0,25,28,29,0,8,0,27,0,12,6,
    0,0,0,1,0,0,0,24,0,0,13,0,0,21,22,8,28,18,3,19,0,7,0,23,14,25,12,11,6,5,
    21,0,0,18,27,0,25,1,30,20,0,3,9,0,28,17,0,16,0,6,22,14,12,0,26,0,0,0,13,0,
    8,3,15,21,19,0,0,0,0,0,0,24,23,26,0,0,29,14,0,27,0,4,22,12,0,5,2,25,20,0,
    0,27,24,0,0,26,0,0,15,19,30,7,0,0,14,16,2,28,0,13,18,23,5,0,0,17,9,6,0,0,
    22,0,0,12,0,0,7,9,14,24,0,0,0,23,8,0,18,25,20,0,13,2,0,27,17,29,0,0,16,11,
    3,0,0,15,0,0,27,0,0,16,11,0,0,0,0,7,0,29,24,28,23,0,19,9,18,0,20,1,5,4,
    19,15,1,23,22,0,0,25,0,2,7,17,0,11,21,13,0,0,0,5,0,0,0,0,29,14,0,0,10,12,
    0,14,0,0,9,17,0,0,3,7,20,0,5,0,19,0,30,0,25,0,0,0,0,11,22,27,8,16,18,13,
    11,0,13,28,4,1,0,0,18,23,2,0,0,6,0,25,7,0,8,0,15,9,21,0,0,20,0,0,0,27,
    16,0,0,30,5,12,0,0,0,8,29,2,14,17,0,4,22,23,7,24,21,0,0,6,0,0,0,0,19,18,
    13,26,19,24,0,0,10,0,0,30,21,29,18,0,20,0,27,0,0,0,14,0,1,28,0,16,17,0,25,23,
    14,25,8,6,1,23,0,16,0,0,0,0,0,10,2,0,15,26,19,0,0,20,9,24,0,7,0,22,0,3,
    0,0,0,0,21,29,0,23,19,0,0,11,17,18,4,2,0,0,13,20,0,6,25,30,12,15,0,0,14,0,
    10,7,5,17,0,16,8,22,29,9,0,6,0,27,0,0,0,0,0,30,4,13,0,0,20,0,23,2,26,0,
    12,23,18,13,0,14,4,30,28,0,5,0,0,0,25,0,8,22,0,17,2,0,0,10,1,0,0,19,0,16,
    0,0,0,0,0,0,0,17,5,0,27,12,19,0,9,30,0,10,11,0,1,16,18,0,7,8,15,0,22,29,
    5,0,30,8,13,19,18,2,0,0,0,9,20,0,0,28,0,1,12,0,3,0,0,4,0,21,0,10,23,26,
    0,6,0,0,0,25,5,27,2,0,0,15,0,7,26,0,14,13,9,16,0,11,30,0,4,0,22,28,0,0,
    0,29,0,0,10,2,23,26,0,15,4,14,6,30,0,21,5,19,1,0,0,0,0,0,0,0,0,20,11,24,
    0,21,22,0,0,18,29,0,4,11,19,0,2,14,0,0,1,12,0,7,28,10,16,0,0,3,26,9,0,0,
    0,10,6,0,2,13,17,28,11,27,3,5,0,15,0,0,0,20,0,9,24,0,0,26,0,18,21,0,0,30,
    23,0,4,26,18,0,3,5,12,0,10,20,29,0,0,9,6,0,0,0,0,17,28,2,21,30,0,0,8,0,
    6,28,23,5,0,0,21,15,8,0,0,1,22,0,0,14,10,2,0,11,0,0,27,0,9,0,3,29,0,7,
    0,13,20,22,0,8,1,12,21,0,24,0,25,0,16,0,9,0,10,15,19,0,3,0,0,0,28,27,0,0
  };

  const int d30_374_20[] = {
    30,
    25,14,9,0,0,0,23,20,13,7,30,0,0,6,3,4,8,0,0,29,28,19,0,22,0,0,0,24,0,21,
    9,0,0,0,19,0,6,2,0,26,0,0,28,23,0,18,0,0,30,15,10,25,22,0,20,17,0,16,7,14,
    2,0,24,30,10,0,25,4,0,21,0,0,0,18,22,1,0,9,0,28,0,26,5,17,0,0,0,6,20,7,
    30,25,1,14,9,12,2,0,26,8,0,0,19,17,27,0,0,18,15,0,21,13,0,0,0,0,0,0,24,4,
    0,17,0,19,3,0,7,0,0,2,16,4,0,14,0,25,6,8,0,0,0,0,27,0,28,0,21,26,18,5,
    0,30,0,22,27,0,3,0,2,10,12,15,0,0,0,19,0,13,1,18,16,7,0,0,25,0,20,8,0,0,
    0,15,0,7,16,0,5,0,0,0,0,28,0,0,24,0,4,23,19,21,25,0,0,0,29,12,14,10,30,27,
    0,22,0,0,26,29,0,10,1,28,11,0,7,0,18,23,0,6,13,0,19,0,21,30,0,3,0,4,0,0,
    15,26,0,0,0,7,0,29,0,30,0,0,0,10,12,24,2,0,0,4,0,27,8,11,9,25,17,28,14,0,
    8,12,0,13,5,19,29,9,10,6,22,23,21,0,0,0,0,20,0,0,17,0,11,0,7,4,0,0,27,0,
    16,0,6,3,0,28,8,24,17,22,0,27,0,2,20,0,1,0,0,13,15,0,0,12,0,10,0,19,0,9,
    17,21,7,0,6,15,10,0,18,0,0,13,25,0,30,0,0,5,0,0,0,29,2,0,24,22,3,0,12,0,
    0,2,29,0,4,14,0,16,11,3,0,5,0,0,13,0,9,17,7,0,12,28,10,0,0,24,0,0,22,25,
    23,20,27,0,18,9,1,19,0,29,26,0,5,25,6,10,0,0,8,0,2,0,13,0,0,0,15,0,0,11,
    0,19,0,0,23,26,27,15,4,0,0,10,16,11,0,6,18,0,0,2,30,0,17,1,0,0,8,0,21,3,
    28,0,0,5,17,20,19,0,15,0,3,12,26,0,0,0,0,2,24,7,0,9,0,6,13,0,0,0,11,29,
    0,0,8,0,0,0,28,0,7,20,21,0,0,0,9,16,0,0,2,0,26,24,23,13,1,15,10,14,4,0,
    0,9,22,28,11,27,0,17,0,0,10,6,8,20,0,2,7,16,0,0,14,5,0,4,0,0,26,30,0,0,
    0,16,0,27,0,13,0,0,29,12,2,0,0,0,10,20,19,26,28,3,0,0,0,15,0,21,5,0,6,1,
    0,0,20,10,14,0,0,0,6,17,24,9,0,8,0,30,5,0,18,0,4,0,0,2,3,0,22,11,25,23,
    5,0,0,12,0,6,0,28,0,0,13,0,1,0,11,27,10,7,22,30,8,0,3,21,16,0,25,0,0,0,
    6,0,4,23,30,17,14,0,19,0,0,24,29,13,0,0,3,28,20,0,18,8,0,0,0,11,0,0,0,22,
    13,24,12,0,28,16,26,0,0,27,15,0,0,0,7,17,0,0,0,1,0,0,30,14,0,23,0,29,5,20,
    0,0,19,16,20,0,0,13,0,0,0,17,22,5,0,8,23,0,0,10,0,12,24,0,15,14,29,7,9,0,
    7,4,5,0,0,0,0,25,14,0,18,20,12,29,0,0,11,24,26,0,0,21,0,0,27,2,16,22,17,0,
    0,0,11,20,0,0,9,26,5,16,4,30,15,0,0,0,0,1,0,12,0,17,19,24,8,18,0,3,0,0,
    10,29,3,4,0,30,0,0,8,0,7,18,20,12,26,0,22,0,11,24,0,0,0,5,0,13,0,21,0,2,
    22,6,13,26,0,0,15,1,0,0,0,0,11,16,2,21,24,0,0,0,0,23,0,8,5,28,30,27,0,19,
    4,0,0,0,0,0,0,5,0,0,1,21,10,22,25,0,28,30,29,9,23,15,26,27,18,0,2,0,0,0,
    0,0,26,2,0,4,0,21,28,0,27,0,14,15,0,7,0,12,5,17,0,18,29,0,22,8,1,0,0,10
  };

  const int d30_375_21[] = {
    30,
    18,13,0,20,0,29,0,0,7,16,0,8,24,0,30,28,10,0,25,21,0,19,9,0,0,14,27,0,0,0,
    0,8,16,0,0,24,0,14,0,0,2,9,0,5,0,15,26,21,29,7,17,0,13,30,22,25,0,10,0,0,
    24,4,0,3,0,2,0,0,0,15,22,28,14,8,26,30,1,0,6,0,0,17,0,27,12,5,0,0,0,25,
    11,0,26,7,0,22,0,1,27,0,0,4,19,0,6,2,12,0,30,0,15,0,0,5,10,17,21,14,0,0,
    0,0,14,0,10,0,5,0,1,23,27,12,0,20,0,21,0,16,0,0,0,7,24,25,4,3,8,0,17,11,
    0,20,0,9,21,27,0,7,6,0,26,23,0,0,3,25,0,0,0,11,16,1,0,0,28,0,15,0,24,10,
    7,3,18,27,0,21,0,0,15,10,0,0,6,11,0,12,0,24,9,0,14,5,8,0,30,0,19,0,2,0,
    6,0,0,2,29,4,15,16,17,26,21,0,7,18,20,0,8,23,24,0,12,0,0,0,0,0,1,0,28,0,
    1,12,15,0,6,0,0,8,29,28,23,0,21,3,0,0,20,7,0,4,11,0,30,26,0,0,0,0,13,22,
    0,0,0,0,0,0,10,0,25,0,0,0,0,28,2,0,4,13,14,8,20,23,19,15,17,9,3,21,27,30,
    22,0,0,30,25,0,0,5,0,0,20,7,18,26,24,0,0,19,21,15,0,10,0,14,0,0,0,23,12,6,
    20,6,0,25,0,23,14,0,26,0,24,5,0,0,4,0,21,1,0,10,2,16,7,0,0,0,9,0,8,0,
    27,0,5,0,0,0,16,15,2,12,28,0,0,9,23,0,11,0,19,20,13,0,1,0,3,0,10,17,0,14,
    8,5,0,0,0,19,4,0,23,17,0,10,30,24,28,0,27,3,18,26,22,29,0,0,2,0,0,0,0,0,
    0,14,30,10,20,8,0,6,9,0,0,0,0,0,12,18,17,25,13,0,0,4,27,0,0,16,0,0,5,7,
    28,0,6,13,7,15,0,27,3,25,0,18,10,0,17,0,19,0,0,29,0,0,16,9,21,0,4,0,0,0,
    2,0,24,6,0,0,30,3,0,11,0,29,20,0,0,23,13,5,0,0,7,12,0,8,26,0,0,25,14,0,
    0,0,28,11,30,14,2,23,0,18,7,0,0,19,0,0,0,0,22,24,0,0,10,12,29,27,0,8,6,17,
    0,21,0,23,0,0,19,0,4,30,17,25,0,0,10,3,0,0,5,16,0,0,0,22,0,6,24,20,7,26,
    30,22,27,0,0,0,1,0,0,21,0,0,25,16,13,0,0,29,0,14,5,2,0,18,8,0,12,9,3,23,
    0,26,23,0,5,0,11,20,16,2,0,0,17,0,0,10,9,0,4,13,6,0,12,0,0,7,30,0,0,15,
    16,7,0,24,0,11,21,25,30,0,18,0,0,0,0,0,15,0,0,0,28,14,3,19,1,8,0,26,23,4,
    4,2,0,0,3,25,0,0,0,0,13,19,0,0,0,14,23,0,0,28,0,20,0,7,5,24,18,29,22,16,
    21,25,7,0,1,0,24,0,0,22,15,14,0,12,0,5,30,0,28,0,0,0,18,2,0,11,20,13,0,0,
    0,17,0,0,13,0,18,11,0,14,8,1,16,0,0,0,0,2,0,19,24,21,6,29,7,23,0,3,0,12,
    15,0,3,0,23,0,0,24,0,19,0,13,29,14,25,22,0,4,11,0,27,6,0,0,0,1,2,0,21,28,
    0,0,0,29,15,17,23,10,11,0,30,0,28,0,14,8,0,0,0,5,0,0,21,20,0,0,26,7,9,18,
    3,0,11,0,4,28,26,0,0,0,6,16,12,7,0,0,0,22,1,0,23,0,29,13,9,30,0,19,0,0,
    0,0,17,15,14,9,25,0,10,13,0,0,22,21,27,11,2,20,0,0,3,0,5,0,0,19,28,18,0,0,
    0,19,21,28,9,20,27,17,0,0,16,0,13,15,0,1,0,8,0,0,0,18,0,0,23,26,0,22,30,0
  };

  const int d30_375_22[] = {
    30,
    12,30,9,17,0,6,2,27,0,0,24,0,3,0,13,23,0,0,0,0,0,14,25,19,5,29,0,11,0,0,
    0,0,0,15,24,29,30,20,25,18,7,14,0,0,0,10,22,0,8,0,0,2,3,4,0,6,0,28,5,0,
    28,19,0,0,0,17,18,0,29,22,3,0,0,0,25,20,0,6,0,23,16,0,12,0,0,26,21,0,1,30,
    0,0,11,25,0,20,6,0,17,0,18,12,24,28,0,0,1,27,7,15,0,3,22,0,30,0,0,0,10,23,
    26,0,0,0,28,30,0,4,0,3,29,21,20,7,0,0,15,0,0,2,0,6,0,22,13,0,27,17,18,0,
    30,15,0,0,0,27,0,0,5,28,4,20,6,0,22,7,17,2,14,13,0,8,0,0,23,0,10,0,9,0,
    22,10,29,13,3,5,7,0,0,8,0,1,0,0,0,0,0,23,15,0,0,0,9,12,0,4,25,24,11,27,
    29,0,0,28,17,7,4,8,22,0,26,0,1,0,0,18,10,0,11,21,0,13,0,9,14,30,6,0,0,0,
    0,0,0,0,19,0,0,11,24,16,0,0,15,6,8,0,7,13,12,9,0,0,1,2,21,0,5,18,22,28,
    0,0,5,27,6,0,0,19,0,0,10,30,0,0,4,0,24,20,1,0,21,0,13,29,0,12,0,14,15,8,
    25,0,0,0,9,0,10,14,1,11,0,6,30,21,0,12,0,0,0,0,29,18,0,27,0,22,15,4,17,0,
    0,6,0,26,7,0,19,0,20,0,23,13,0,15,18,0,0,25,9,11,8,0,21,0,2,5,0,0,12,0,
    0,0,20,21,0,3,27,0,14,7,0,0,5,29,0,0,0,30,0,6,2,19,18,23,11,13,1,0,0,0,
    9,18,3,0,30,0,0,0,0,0,20,5,27,0,15,16,0,12,0,14,0,0,2,0,1,17,7,25,0,11,
    0,0,7,12,0,26,5,3,0,1,27,0,19,8,14,13,4,24,16,0,0,20,0,18,0,25,22,0,0,0,
    21,7,0,0,0,18,13,29,28,17,0,8,14,2,0,0,0,16,26,1,23,10,24,30,9,0,0,0,0,0,
    17,0,0,5,1,13,0,30,6,0,0,29,0,0,20,15,14,0,22,0,19,21,0,0,7,28,4,10,0,0,
    0,21,10,0,20,0,0,0,0,5,0,28,22,14,26,9,0,29,0,25,6,23,17,0,18,7,0,0,0,2,
    3,0,24,18,15,0,0,10,9,26,0,0,2,0,0,27,6,0,0,28,25,0,29,7,0,20,8,0,0,19,
    16,29,13,7,27,23,24,0,8,0,0,22,0,18,12,6,0,19,20,0,0,0,0,0,0,9,0,0,21,1,
    6,1,0,0,0,16,14,24,21,0,0,0,0,10,0,0,18,0,28,27,12,0,4,26,15,0,20,7,29,22,
    18,24,23,4,0,14,0,21,0,0,11,17,8,0,0,30,20,9,0,0,27,5,0,16,28,15,19,0,0,0,
    23,27,4,30,0,0,15,0,0,9,0,18,12,11,3,0,0,7,0,29,0,26,19,1,17,0,0,5,0,20,
    0,0,1,0,26,9,0,0,0,0,30,2,18,0,0,25,0,0,21,8,20,22,14,6,16,0,11,23,28,29,
    0,0,19,24,0,21,25,9,0,30,22,0,23,4,16,0,2,0,0,0,14,0,20,0,26,3,0,12,27,0,
    0,22,28,8,10,0,0,0,0,13,0,0,21,25,7,0,11,4,24,17,0,9,26,0,0,0,0,2,30,15,
    27,17,2,16,8,15,0,1,19,0,12,0,0,0,23,24,26,0,0,4,13,28,0,0,0,0,30,0,25,21,
    20,12,0,2,0,0,8,6,7,0,19,0,0,13,27,22,29,17,18,0,15,24,0,0,0,0,0,21,0,5,
    0,3,8,0,2,0,0,17,12,27,0,23,0,24,1,0,0,21,25,26,10,0,0,11,0,0,28,13,6,14,
    11,9,30,0,0,0,3,0,15,23,5,10,0,16,2,4,25,0,6,0,22,0,0,0,0,19,0,20,13,12
  };

  const int d30_375_23[] = {
    30,
    5,23,0,9,0,0,0,0,29,0,0,0,25,6,7,21,0,0,16,0,18,1,0,19,22,28,13,20,30,17,
    10,0,21,0,3,0,30,8,22,0,0,28,2,26,0,0,0,0,0,11,25,5,0,0,18,24,27,0,19,7,
    27,12,2,0,4,9,3,0,0,0,7,13,18,17,0,0,29,26,5,0,0,0,22,24,0,21,15,0,0,8,
    19,10,0,6,0,30,0,11,15,0,26,0,0,13,12,4,0,21,2,23,0,0,20,27,24,0,0,5,1,0,
    30,2,0,1,0,0,24,0,0,10,13,25,0,5,0,0,6,7,15,21,17,0,0,0,11,29,0,9,14,0,
    0,0,28,0,0,0,9,0,8,17,15,12,1,29,26,30,0,0,10,13,22,23,0,20,0,3,0,0,18,5,
    0,24,19,21,23,28,29,2,12,0,0,4,0,0,1,0,0,17,0,14,26,30,3,0,0,8,0,0,7,0,
    24,0,0,0,0,16,27,0,25,0,29,0,11,9,4,23,17,0,0,12,5,21,15,0,10,26,0,13,0,2,
    1,11,0,24,26,14,6,7,28,8,23,0,0,30,0,0,0,2,3,29,0,0,0,4,0,16,0,0,12,0,
    9,0,13,0,11,12,0,17,27,25,10,0,0,0,0,15,14,8,23,22,3,7,16,0,2,0,0,0,0,19,
    13,1,0,0,0,29,18,0,0,0,0,9,4,24,0,2,7,15,26,0,27,19,0,22,3,0,21,11,0,6,
    0,0,26,0,1,6,0,0,0,27,25,24,22,0,0,7,28,5,21,0,2,0,13,16,19,11,0,18,0,0,
    0,25,0,0,9,0,16,0,0,24,6,0,5,0,0,0,15,19,0,0,0,12,27,14,8,13,2,3,29,23,
    21,0,16,2,18,0,0,26,0,0,28,14,0,0,24,29,22,0,13,7,23,0,6,9,4,0,10,1,0,0,
    0,16,25,0,30,7,0,20,0,26,0,0,0,0,5,24,19,3,4,0,9,2,0,15,29,0,23,0,0,18,
    18,27,0,10,21,3,0,0,0,28,0,2,0,0,16,20,11,0,0,0,0,17,7,30,1,0,5,8,0,22,
    0,0,0,19,22,24,25,18,0,0,8,3,0,10,13,5,21,0,0,15,20,0,4,2,0,12,30,0,27,0,
    0,0,7,30,0,0,0,14,6,0,21,0,0,16,17,0,26,0,22,9,29,13,2,0,15,4,11,0,5,1,
    0,5,27,11,0,0,26,19,20,21,0,0,15,0,25,10,16,12,0,18,0,0,8,0,0,0,14,30,9,4,
    6,0,0,18,27,0,0,5,21,0,0,20,14,0,2,0,0,10,0,26,12,29,17,13,9,0,0,19,24,25,
    4,20,29,3,0,23,0,0,1,11,0,26,19,15,0,0,0,0,9,0,0,24,0,28,0,6,8,10,0,12,
    20,8,10,12,6,0,7,23,0,0,4,0,27,2,19,0,0,0,0,0,30,16,0,1,13,0,0,26,25,0,
    3,0,0,13,14,20,28,6,19,0,1,10,0,12,30,27,0,22,8,0,0,0,0,21,0,23,25,24,0,0,
    0,15,5,0,19,4,0,25,10,3,24,0,0,0,6,9,0,1,0,0,13,20,21,0,0,0,18,27,28,0,
    0,4,18,23,0,0,0,0,0,14,0,6,9,0,15,12,13,11,0,0,0,28,25,0,17,1,7,0,20,26,
    22,0,0,8,0,5,0,9,0,29,2,0,13,28,20,0,12,0,0,24,14,4,26,7,0,0,17,16,23,0,
    0,17,15,29,8,18,23,0,16,13,0,22,0,0,0,1,27,20,24,10,6,0,0,0,0,0,19,0,26,28,
    15,0,0,0,13,0,11,30,7,18,0,21,29,22,0,3,0,24,0,25,0,9,0,0,0,10,1,28,6,0,
    0,28,1,0,0,27,17,12,26,9,5,8,21,20,22,0,30,23,6,0,0,0,0,0,0,18,0,0,0,14,
    7,0,11,0,20,0,1,3,2,30,0,17,8,14,0,16,0,0,25,27,0,0,23,26,28,0,0,0,0,29
  };

  const int d30_375_24[] = {
    30,
    19,22,0,1,30,0,0,0,23,17,0,28,20,0,14,0,0,2,0,9,13,4,0,5,0,26,18,25,0,7,
    30,2,0,0,0,28,0,6,0,0,0,1,4,0,0,21,14,23,0,7,19,0,0,29,18,25,20,0,12,5,
    1,0,27,6,18,23,3,0,12,20,0,0,0,29,0,24,9,0,7,0,2,8,4,0,13,0,26,0,0,11,
    28,0,15,0,13,10,1,20,22,8,11,0,0,0,0,17,30,0,5,0,0,9,0,18,26,6,0,24,3,0,
    15,0,14,7,16,0,28,0,25,0,0,18,0,2,1,11,0,29,0,0,0,0,24,27,17,12,10,4,0,8,
    0,14,0,0,21,27,11,8,0,0,25,0,26,23,16,6,0,0,0,29,0,0,3,7,0,28,5,10,30,17,
    20,0,0,26,0,0,6,0,14,2,0,3,13,0,7,25,24,30,1,0,15,10,0,12,0,0,11,17,0,0,
    26,23,20,0,29,0,0,17,2,0,0,0,3,13,0,15,0,27,25,16,1,5,12,0,0,0,6,0,0,24,
    27,0,29,12,0,15,25,0,0,0,0,2,30,8,0,0,11,21,0,0,14,0,6,9,0,0,24,1,22,13,
    0,0,0,0,20,6,14,0,1,27,26,5,0,16,17,13,29,0,19,3,22,0,8,0,10,30,0,0,18,0,
    0,0,0,0,12,25,2,24,0,28,0,27,7,26,4,0,3,17,30,5,0,14,0,0,16,0,0,21,9,0,
    4,0,5,15,24,26,27,0,10,1,0,7,23,0,12,0,0,0,16,0,0,19,28,0,6,29,0,18,11,0,
    0,11,0,9,2,20,13,12,0,0,23,19,27,0,29,0,0,15,6,10,0,0,21,0,24,3,0,8,16,0,
    22,12,0,19,8,0,0,16,0,18,15,0,11,9,26,0,0,20,0,30,0,28,29,25,27,5,0,0,0,3,
    0,3,8,22,0,0,21,15,0,5,4,23,6,1,0,29,0,10,0,13,30,0,0,14,0,0,0,16,0,12,
    6,13,24,0,0,14,0,0,27,10,12,26,0,19,0,0,25,3,20,0,0,0,23,0,5,11,8,0,0,22,
    0,0,0,10,0,21,0,0,9,0,18,16,0,0,25,2,23,0,11,0,12,24,14,0,15,20,19,0,27,30,
    17,28,0,0,0,0,0,22,26,0,30,0,0,27,13,19,6,18,29,0,0,20,16,24,0,0,3,9,15,4,
    0,19,10,0,11,0,20,26,29,21,0,0,2,0,8,12,7,0,0,24,3,0,0,13,0,14,22,0,4,0,
    0,6,11,30,0,9,0,23,0,0,24,0,14,22,19,3,0,7,0,0,0,0,0,28,1,0,4,26,10,21,
    5,15,9,0,0,0,0,0,16,0,3,11,10,0,18,0,0,19,13,0,0,12,26,0,0,22,7,14,6,27,
    23,0,0,0,14,17,0,1,30,3,16,24,0,6,0,0,28,0,0,4,7,0,18,19,29,0,13,0,0,20,
    0,0,18,3,0,0,0,30,0,9,19,21,0,14,0,0,5,8,28,26,10,17,0,16,0,27,29,0,25,0,
    13,24,0,5,3,1,0,18,0,7,0,15,0,30,27,0,0,22,9,0,4,29,0,0,0,0,0,11,20,26,
    9,17,0,18,22,16,24,25,28,0,29,12,21,15,0,10,0,0,0,27,0,2,20,0,30,0,0,0,8,0,
    12,9,21,0,7,0,26,0,15,0,0,0,16,5,24,0,1,0,0,0,8,11,0,20,14,10,17,3,2,0,
    0,0,16,20,26,0,8,0,0,25,21,0,9,0,0,0,12,24,4,1,5,0,0,30,0,7,28,6,13,29,
    3,29,12,14,6,0,22,11,7,23,0,0,0,0,2,9,0,0,0,21,17,26,15,4,19,0,0,30,0,0,
    0,20,22,23,0,24,19,7,0,14,28,0,0,25,0,4,27,0,10,6,0,13,5,8,0,2,0,0,0,0,
    0,21,19,0,0,5,18,14,24,26,27,4,12,0,0,22,10,0,23,28,9,0,30,0,25,0,0,0,0,2
  };

  const int d30_375_25[] = {
    30,
    0,18,27,5,0,0,0,20,0,12,6,7,2,0,9,0,1,14,10,0,26,4,0,8,0,3,0,15,19,0,
    17,12,16,0,3,8,11,0,28,6,13,22,1,0,0,0,0,21,0,5,0,9,29,30,0,0,0,0,4,14,
    0,0,23,0,0,15,19,5,0,14,22,6,9,25,0,0,13,12,0,0,11,27,0,2,0,30,1,0,0,3,
    0,0,13,23,22,1,0,0,24,0,0,16,0,9,20,0,15,0,0,8,30,0,3,18,11,28,25,21,0,17,
    0,7,0,22,18,3,24,29,0,1,30,0,0,0,0,28,0,20,0,0,0,0,0,5,17,23,21,10,12,15,
    2,8,4,0,19,18,0,0,9,0,28,0,0,11,7,0,14,0,15,29,13,0,10,6,24,5,0,0,27,0,
    0,0,2,0,0,9,30,19,0,4,0,12,23,0,0,16,28,0,6,3,5,0,1,25,0,21,20,0,22,26,
    9,0,24,15,27,0,0,14,0,10,20,11,0,26,0,19,0,0,22,0,0,0,0,13,21,2,18,5,0,7,
    10,11,12,0,0,0,6,0,0,0,7,27,4,21,0,0,2,0,24,14,19,3,0,20,0,9,0,30,26,5,
    30,0,18,0,23,0,9,0,0,2,0,5,12,6,28,21,0,0,17,13,0,19,15,0,0,22,0,4,10,0,
    0,20,0,24,0,14,26,15,3,0,9,30,0,0,0,29,0,17,13,23,0,22,8,0,0,0,5,12,25,27,
    0,19,29,8,21,0,0,0,13,0,5,28,15,0,2,0,22,9,14,0,25,0,0,0,7,26,16,0,0,10,
    25,0,0,0,28,0,0,1,0,8,14,0,13,24,6,3,19,2,0,22,27,29,12,0,0,0,10,0,7,0,
    13,14,0,6,0,0,5,21,0,0,0,19,0,12,0,30,16,26,0,2,9,0,23,10,0,4,7,8,20,0,
    0,30,0,27,0,0,25,7,12,0,2,15,8,20,21,0,0,0,0,18,23,13,16,26,4,0,11,0,0,0,
    16,27,6,7,17,0,10,3,18,0,0,0,0,0,30,1,0,4,9,0,0,0,0,0,25,14,26,28,0,8,
    3,26,15,14,1,0,0,0,8,18,4,0,0,16,11,20,27,6,0,25,10,21,0,29,0,0,19,0,0,0,
    28,21,0,9,8,17,15,0,6,0,26,25,0,0,24,0,4,10,29,7,0,0,0,22,16,0,0,23,0,18,
    27,1,21,0,29,20,0,10,0,22,23,0,26,0,13,4,25,0,3,0,7,11,6,0,12,0,0,0,24,0,
    26,0,0,20,0,0,7,24,29,11,0,0,14,0,17,12,10,5,19,16,0,30,21,0,0,8,0,18,0,2,
    0,4,0,0,30,0,16,9,27,19,0,18,3,23,8,0,0,0,28,0,24,25,0,0,13,0,12,17,0,29,
    0,22,20,0,13,5,0,8,0,7,0,29,11,10,0,0,0,16,12,26,14,2,0,0,28,17,0,6,0,0,
    1,0,25,12,0,0,0,0,17,0,21,0,20,7,5,10,3,0,0,0,15,26,2,4,19,0,30,0,11,0,
    0,24,8,30,0,13,4,0,0,3,0,0,0,0,22,25,0,0,0,12,0,23,5,14,26,20,0,9,15,21,
    11,0,0,18,24,7,0,0,2,25,0,0,0,8,0,22,21,23,0,0,0,28,27,12,29,0,0,14,5,1,
    0,0,28,13,20,16,3,0,0,26,8,14,0,22,18,15,12,0,5,27,0,17,0,0,10,6,0,0,29,0,
    21,0,5,0,0,6,2,4,23,0,0,1,0,14,27,0,0,28,7,0,12,0,22,0,20,0,3,19,17,13,
    18,0,0,2,11,30,0,25,5,29,0,0,19,3,0,14,9,0,23,0,4,6,0,24,0,15,0,0,0,16,
    23,5,0,17,0,2,13,0,16,0,0,21,18,0,26,11,29,30,0,10,20,0,4,0,6,0,14,0,28,0,
    5,0,0,0,2,25,0,22,1,20,3,0,16,30,0,0,0,13,0,4,0,0,7,15,0,10,8,26,23,0
  };

  const int d30_375_26[] = {
    30,
    24,0,10,0,0,1,19,0,5,0,6,0,12,4,25,21,0,29,26,0,0,0,13,18,23,27,3,22,0,0,
    1,0,0,21,15,7,0,0,17,23,25,14,0,19,0,0,8,0,0,0,27,0,9,12,10,11,28,20,0,0,
    20,14,7,23,0,24,11,8,0,4,0,17,0,0,0,25,0,0,13,10,0,6,3,27,0,0,5,0,1,12,
    7,20,8,27,0,25,0,24,0,0,0,18,0,3,19,0,0,0,5,30,21,23,14,28,9,0,12,0,29,0,
    25,29,4,0,0,0,27,20,1,6,0,0,13,11,28,0,12,0,0,9,0,0,0,26,19,0,7,0,5,2,
    11,0,2,0,13,0,9,30,12,16,23,0,19,0,5,0,4,0,0,0,26,18,28,0,0,3,0,14,0,15,
    14,7,22,12,0,0,26,0,4,0,28,0,15,8,0,13,0,16,10,3,0,19,17,0,11,0,30,18,0,0,
    0,23,0,29,1,20,5,27,0,28,17,0,0,0,6,0,9,2,3,0,14,16,0,24,0,13,0,7,0,0,
    18,6,0,0,0,22,0,0,0,21,14,19,28,0,16,0,23,17,0,1,8,0,15,13,24,0,0,29,30,7,
    13,0,0,28,25,0,0,9,23,0,0,6,0,0,4,0,2,0,14,0,5,17,20,16,3,1,0,10,0,26,
    3,17,28,0,16,5,0,22,13,15,30,9,8,0,0,0,7,0,2,0,0,0,0,25,29,0,0,0,18,20,
    10,0,5,22,14,21,0,1,0,0,4,7,16,30,0,3,0,6,0,0,0,2,0,15,0,26,25,28,13,0,
    0,25,0,18,0,6,0,0,30,0,19,0,0,0,14,2,11,27,0,0,0,12,5,7,21,22,9,8,20,4,
    0,0,0,0,5,2,24,16,0,0,20,25,4,17,0,0,0,12,0,21,22,13,19,0,1,7,11,0,14,0,
    0,0,0,3,8,0,14,0,28,0,16,26,0,29,30,11,0,21,23,6,0,10,0,1,25,0,0,27,12,0,
    0,4,12,11,0,30,21,19,0,25,29,0,7,0,13,18,0,0,0,0,0,27,26,3,0,8,17,1,6,0,
    22,0,0,0,24,0,28,0,15,13,3,11,0,1,2,9,10,0,18,0,12,0,0,0,0,4,26,16,0,27,
    6,2,20,10,0,8,17,0,0,29,13,24,0,0,21,0,16,3,0,12,30,0,0,0,0,14,22,0,4,0,
    0,0,0,26,9,0,8,6,19,0,1,5,3,28,7,27,0,20,0,0,0,0,12,29,15,0,0,24,2,18,
    0,10,13,0,12,15,23,0,0,0,2,4,21,0,0,24,18,28,20,29,3,25,7,17,0,30,0,0,0,0,
    0,30,27,14,3,0,0,18,16,26,0,20,29,7,12,0,0,1,0,2,0,8,6,0,0,0,10,0,0,19,
    19,3,0,2,28,0,22,0,0,0,11,10,0,21,0,16,0,14,0,0,7,20,0,0,4,23,6,15,24,13,
    0,8,25,0,0,18,16,17,24,0,0,15,11,10,0,20,28,0,6,19,4,29,30,0,5,0,13,0,0,0,
    0,13,0,15,21,9,4,26,0,11,0,0,18,0,0,6,19,24,27,7,16,0,0,14,0,20,0,0,0,29,
    15,0,17,0,0,27,0,0,10,2,26,0,14,0,0,23,25,18,0,22,13,11,0,5,0,0,29,30,0,24,
    16,28,18,0,0,10,13,0,11,0,0,30,26,25,24,0,29,0,19,14,23,0,0,0,17,21,0,0,7,0,
    0,27,30,6,10,0,0,4,9,7,0,0,2,20,0,15,21,0,17,28,0,0,16,0,22,25,0,0,11,23,
    0,0,0,5,2,12,30,0,7,27,0,0,17,0,1,4,0,0,21,26,6,0,8,0,0,19,15,13,22,10,
    5,1,0,0,29,0,0,21,0,10,0,12,0,24,8,14,22,11,15,4,20,0,0,0,28,0,0,17,25,6,
    28,0,6,0,11,0,0,5,3,19,0,0,0,23,15,30,0,7,22,0,18,26,0,9,0,17,20,0,0,1
  };

  const int d30_375_27[] = {
    30,
    22,15,19,0,0,0,21,28,0,29,0,20,8,0,11,0,2,10,0,9,18,0,0,0,16,6,30,27,0,4,
    4,0,12,5,24,18,0,9,17,20,21,30,0,0,0,25,0,0,26,0,29,0,15,7,0,13,0,3,0,16,
    0,0,0,15,12,0,14,0,23,27,0,6,2,10,26,0,0,0,16,22,21,0,3,18,9,0,11,30,0,0,
    12,9,13,0,0,6,8,27,0,0,24,0,0,0,28,18,0,4,11,23,26,0,2,20,0,5,0,0,0,21,
    0,27,0,14,11,15,30,0,0,0,5,2,0,0,17,10,21,25,0,0,28,9,0,1,3,19,0,12,0,0,
    0,0,0,10,28,21,2,7,20,0,0,5,0,1,12,6,23,0,14,16,0,29,0,24,13,0,0,0,26,0,
    0,21,0,8,6,0,9,0,24,0,0,25,0,16,29,27,0,15,0,0,14,5,10,0,1,30,17,23,3,0,
    0,0,7,0,19,0,4,25,3,2,0,0,0,0,6,5,20,0,13,24,0,18,0,28,0,21,8,0,11,1,
    5,0,0,27,0,0,16,0,1,0,19,17,21,29,0,24,25,26,0,8,0,28,30,14,7,2,0,0,0,22,
    17,0,9,4,0,22,0,13,27,19,18,3,1,0,21,23,26,0,0,0,2,0,0,0,0,16,5,10,0,14,
    0,29,6,28,25,0,0,0,0,0,22,15,3,0,10,0,14,0,12,0,17,4,0,0,26,7,13,8,5,2,
    0,4,0,0,26,0,18,19,0,14,3,13,0,20,0,12,29,17,7,21,15,8,11,0,22,0,0,2,0,0,
    6,0,0,9,7,8,0,0,0,0,17,12,18,22,27,0,10,0,0,15,0,14,4,0,0,28,29,11,20,23,
    0,0,10,0,8,11,25,17,4,13,0,0,29,2,16,15,19,20,0,0,23,0,0,0,30,0,3,0,1,0,
    9,25,0,0,0,29,0,22,0,7,16,0,27,12,15,0,0,8,20,0,0,24,21,11,5,0,0,0,23,26,
    15,10,20,22,0,1,5,14,29,0,0,16,26,17,0,11,0,0,24,30,12,0,19,0,0,0,27,0,0,0,
    0,0,24,0,2,19,0,0,28,16,15,23,0,0,3,22,4,13,0,0,8,0,14,21,0,0,1,29,10,0,
    21,19,22,3,0,30,1,5,0,26,0,0,0,0,4,0,18,12,0,0,7,0,0,13,6,27,16,0,24,8,
    13,18,8,26,0,16,0,0,0,23,27,0,15,7,0,0,0,6,4,0,0,20,0,0,2,0,25,28,9,24,
    2,0,17,0,0,0,26,0,16,22,7,0,5,9,14,3,0,0,29,0,0,0,23,8,21,0,24,6,0,30,
    29,24,0,18,5,0,0,0,0,0,25,0,0,0,0,0,7,22,23,27,0,26,16,0,8,1,2,17,30,28,
    0,8,16,0,0,7,0,23,11,15,0,0,14,0,0,0,0,0,5,4,0,1,6,2,17,9,26,18,28,0,
    30,3,0,17,14,0,0,0,22,0,0,1,24,0,19,8,12,21,0,25,0,0,13,23,27,29,0,0,16,20,
    19,0,26,0,22,0,7,20,14,0,9,0,13,11,0,1,24,0,0,0,0,23,27,6,0,8,0,4,12,15,
    7,22,0,0,0,0,27,15,2,0,10,9,17,18,0,16,28,29,30,6,24,0,0,0,25,0,0,5,13,0,
    0,2,0,11,20,13,0,0,25,10,30,0,0,19,0,14,0,23,0,26,6,17,0,3,24,0,28,7,0,27,
    25,0,4,6,9,26,0,10,8,11,0,0,0,28,2,0,0,0,1,3,16,0,17,19,0,0,18,0,29,0,
    0,12,2,0,0,4,0,29,0,1,0,19,23,6,9,0,0,24,27,11,30,10,18,0,0,14,0,0,25,13,
    0,30,25,0,0,20,3,0,5,24,4,8,28,14,0,29,11,9,19,0,0,7,0,22,0,17,0,0,0,0,
    23,0,18,29,1,0,28,12,0,5,14,0,11,0,0,0,8,16,2,7,0,6,26,0,0,22,19,0,0,10
  };

  const int d30_375_28[] = {
    30,
    0,22,21,0,18,17,5,10,1,0,0,0,11,0,0,19,0,24,8,0,27,15,14,0,0,2,29,6,0,0,
    3,14,16,28,23,15,29,0,0,10,0,0,0,2,12,6,0,0,4,13,0,19,5,0,8,0,24,0,0,0,
    27,6,0,0,0,18,24,0,14,0,0,0,0,0,1,0,25,16,13,15,2,12,0,11,20,0,10,8,28,30,
    0,0,9,10,5,26,0,18,28,1,14,2,4,22,6,29,8,0,0,25,0,0,0,0,0,0,23,0,19,27,
    0,10,4,0,0,28,23,26,0,7,0,20,27,0,25,8,22,6,0,0,0,0,0,0,30,5,15,9,2,29,
    0,23,12,24,0,25,13,0,18,0,0,6,14,20,0,15,3,17,0,29,0,16,4,1,0,0,26,0,22,0,
    22,0,0,25,14,20,0,12,19,0,0,16,0,24,0,23,29,0,0,5,9,0,30,3,0,26,0,10,0,1,
    0,0,10,0,0,3,8,17,0,24,12,27,22,23,0,21,13,0,0,0,11,29,6,16,4,0,19,15,0,0,
    9,3,6,15,8,0,16,0,27,19,4,25,0,0,0,0,0,10,0,18,0,26,0,24,0,21,28,30,20,0,
    0,5,8,0,24,0,28,19,20,0,25,0,0,16,7,0,30,0,0,17,4,11,22,6,0,0,12,0,1,2,
    18,0,0,27,0,6,7,13,0,0,8,22,0,0,0,25,9,23,16,0,0,0,0,12,2,20,0,19,14,21,
    25,0,11,0,30,23,20,0,17,0,0,0,0,0,19,7,0,21,5,24,22,13,0,0,0,16,9,14,0,12,
    0,24,0,4,0,0,0,5,2,21,16,7,13,26,0,0,28,0,0,0,25,30,0,20,14,0,11,29,0,8,
    1,30,0,0,19,0,10,0,5,23,0,0,0,8,11,0,0,13,28,0,0,27,0,9,29,18,0,12,7,6,
    0,11,0,5,0,0,25,0,8,14,9,4,18,21,0,17,0,30,20,7,0,0,10,0,6,28,27,0,3,0,
    17,0,23,0,12,0,0,28,13,22,15,14,16,0,20,2,26,25,0,11,1,0,29,0,0,0,0,0,21,0,
    24,29,15,3,0,0,0,11,21,0,0,0,20,6,18,0,0,4,0,16,0,28,25,23,12,17,0,26,30,0,
    10,0,18,7,16,8,0,0,0,26,0,19,0,13,24,0,20,0,30,27,23,0,15,0,11,0,0,25,17,28,
    11,0,0,23,4,30,0,3,0,0,0,28,10,29,21,0,12,26,0,2,7,18,13,0,0,0,25,0,0,9,
    0,2,0,0,6,10,9,14,0,0,21,0,0,25,5,0,15,20,27,0,0,3,24,0,0,7,17,11,0,18,
    0,20,30,0,0,4,17,0,0,0,28,0,15,0,10,0,0,7,22,0,5,1,11,21,13,25,16,3,8,0,
    0,4,0,19,29,12,30,0,0,2,0,26,23,0,8,1,0,14,17,3,24,9,0,0,0,13,0,0,25,20,
    28,0,20,1,2,9,0,27,0,0,19,8,5,11,0,18,24,0,0,0,6,0,0,0,3,0,22,13,15,0,
    0,16,14,0,0,0,0,0,11,18,10,21,30,0,26,12,0,27,0,0,13,0,7,29,15,4,0,22,0,17,
    7,25,0,20,22,0,0,0,4,0,26,17,2,0,0,9,0,11,21,28,19,0,8,5,18,15,0,0,10,0,
    19,0,1,29,27,0,0,23,22,20,5,18,0,0,3,26,16,0,14,30,0,0,21,4,7,11,0,0,0,0,
    0,7,0,0,0,0,18,25,16,6,2,0,9,12,0,0,0,29,3,10,0,22,0,30,1,0,0,17,5,4,
    4,12,22,14,1,0,0,16,0,15,6,0,7,3,0,0,0,0,18,23,26,0,9,28,0,0,2,0,0,25,
    26,0,0,8,0,27,6,0,0,13,24,0,25,0,16,22,18,0,7,0,0,14,0,0,19,30,21,0,29,10,
    12,0,0,9,0,16,22,20,0,28,18,29,0,15,0,0,14,0,10,0,30,7,19,2,26,27,0,0,0,5
  };

  const int d30_375_29[] = {
    30,
    27,0,3,26,14,0,0,17,0,0,0,0,1,2,10,20,28,11,0,0,0,30,23,0,0,13,16,15,4,12,
    0,0,30,19,0,11,0,0,0,13,0,0,0,23,26,0,20,14,17,0,16,29,22,9,3,5,0,21,0,7,
    0,12,0,0,29,5,11,20,0,17,0,1,21,9,15,2,0,22,0,0,0,28,27,14,0,19,3,13,0,0,
    1,20,4,18,16,13,0,0,0,23,9,14,26,0,0,0,0,0,0,10,6,0,28,0,0,0,30,7,22,11,
    20,0,7,0,3,24,17,14,0,11,1,28,4,8,0,0,0,0,5,18,15,0,0,19,30,0,2,23,0,0,
    8,10,5,13,6,0,0,0,25,16,0,0,0,18,28,1,19,17,21,11,0,0,0,0,26,0,0,14,9,23,
    0,8,11,0,18,0,0,0,0,12,30,0,0,14,13,9,22,0,0,19,0,24,4,2,28,17,10,16,15,0,
    3,26,6,2,0,0,0,27,15,21,7,0,0,0,20,0,1,16,0,23,28,22,18,0,0,12,9,25,0,0,
    18,1,0,0,24,0,0,0,0,29,19,25,0,10,0,0,0,0,2,0,3,4,8,13,0,28,11,26,16,22,
    10,19,1,0,20,6,0,24,2,0,18,0,25,0,0,7,0,0,0,5,0,0,3,0,23,9,26,0,27,13,
    0,0,17,6,0,18,8,7,23,0,3,0,2,1,11,4,30,0,0,0,13,0,0,28,22,0,25,0,0,14,
    4,22,0,0,12,0,29,10,0,1,28,0,0,16,0,5,0,21,23,9,24,15,0,26,0,2,6,0,0,8,
    5,2,0,0,28,0,13,19,0,0,0,29,27,11,0,15,26,0,24,17,21,0,0,0,7,22,0,10,6,9,
    0,0,21,25,0,15,20,0,5,0,0,0,7,0,9,0,4,26,0,27,19,11,0,16,14,29,0,30,28,0,
    0,29,28,0,10,1,25,0,0,30,20,24,19,0,0,3,21,23,8,0,14,0,0,0,15,0,0,22,0,26,
    6,0,0,0,11,0,15,0,3,2,5,23,16,0,12,27,0,10,19,0,0,0,0,0,29,14,0,17,13,20,
    15,30,29,23,0,0,26,0,17,0,0,21,0,6,27,0,8,18,3,13,12,0,9,5,0,0,0,0,7,16,
    19,16,0,0,0,0,0,26,28,0,0,9,0,0,21,23,3,7,0,22,30,14,11,27,25,0,0,1,5,24,
    11,13,0,29,25,12,0,28,0,26,0,0,9,22,16,18,0,0,7,3,2,0,0,15,0,24,0,0,30,4,
    0,0,0,22,7,4,5,6,0,0,21,0,0,13,0,19,17,27,15,0,0,23,25,0,24,16,8,0,29,18,
    0,4,0,21,0,2,22,13,24,0,14,11,6,0,25,0,0,3,0,7,0,0,0,0,18,1,0,27,23,29,
    16,0,10,20,0,22,18,30,21,0,0,17,0,0,0,0,5,1,27,14,4,0,2,0,8,0,15,24,0,0,
    0,0,13,0,30,21,2,0,18,0,26,0,0,28,7,14,25,0,12,15,0,27,24,3,17,0,5,0,10,0,
    0,0,15,24,0,9,27,1,10,22,17,6,23,0,5,0,0,4,0,12,0,3,0,20,2,0,18,0,25,0,
    0,0,0,30,0,0,24,15,6,14,27,20,3,0,0,13,18,0,26,0,29,9,0,7,16,23,21,0,0,0,
    29,6,0,5,0,26,1,8,7,0,25,10,22,0,3,0,0,0,11,0,0,21,20,23,0,0,28,18,0,0,
    0,11,16,0,15,25,0,4,30,10,13,7,14,27,0,0,0,0,20,29,17,0,0,0,9,18,0,3,12,0,
    14,28,24,0,0,0,7,0,26,27,0,5,30,12,0,6,2,0,0,20,0,18,1,10,0,11,0,0,0,15,
    0,0,18,8,9,27,0,0,4,6,0,0,15,29,0,21,0,24,28,0,25,19,12,30,0,0,13,0,14,0,
    7,18,0,1,2,0,0,25,0,0,22,12,0,24,14,26,9,28,29,0,0,5,10,21,0,0,17,0,0,0
  };

  const int d30_375_30[] = {
    30,
    0,25,6,26,2,0,0,23,7,21,0,11,0,17,0,0,0,0,3,18,0,15,0,0,13,9,22,0,14,1,
    19,0,12,17,26,0,18,0,10,0,0,28,3,0,4,21,0,20,0,6,2,0,24,27,14,0,25,0,9,0,
    0,20,9,23,16,0,0,0,19,8,10,0,6,0,18,22,0,0,4,0,0,11,29,12,1,30,0,15,0,14,
    0,0,0,3,12,0,7,18,0,0,16,0,2,9,28,1,0,11,0,29,23,0,0,0,27,22,10,19,8,30,
    15,0,0,22,27,24,0,28,8,1,0,21,11,0,0,0,18,0,7,0,0,6,23,0,30,13,9,0,16,0,
    5,7,0,0,0,0,20,6,0,15,0,13,0,23,0,30,8,10,0,11,0,9,27,29,0,1,2,18,0,0,
    13,14,0,0,0,0,28,4,21,0,0,29,26,0,1,0,30,25,2,8,0,3,0,0,12,0,5,24,6,10,
    8,21,15,1,0,30,5,0,0,2,29,0,0,26,0,0,0,0,27,19,17,24,25,0,0,3,13,0,22,11,
    6,29,0,7,30,15,14,0,0,0,24,0,8,21,17,23,5,16,20,0,26,0,0,3,0,0,0,22,0,27,
    0,3,27,0,17,0,0,0,16,0,7,2,0,15,26,25,14,0,0,10,22,29,0,0,0,0,20,1,13,4,
    28,26,20,0,0,3,30,0,0,0,17,4,0,22,14,0,6,24,0,9,1,0,18,25,0,7,0,21,0,0,
    30,0,0,10,1,8,0,7,13,28,0,0,0,27,20,16,0,0,15,0,0,22,6,14,24,23,0,0,25,5,
    0,6,29,0,0,26,23,0,0,7,3,22,0,30,27,18,0,8,16,17,0,19,10,0,0,28,0,5,2,0,
    24,13,19,11,0,12,27,0,28,0,23,0,0,8,0,0,17,0,21,7,0,2,5,26,22,0,0,0,0,18,
    0,0,0,0,8,20,10,22,12,18,0,0,24,0,0,2,27,15,29,0,16,21,3,0,9,11,0,23,0,0,
    22,0,0,16,0,0,0,27,14,12,0,0,5,2,0,9,26,7,17,1,0,20,21,11,0,0,3,0,24,19,
    1,4,30,0,18,19,0,2,0,16,0,8,12,13,23,0,7,6,0,0,10,0,17,0,0,26,28,0,0,9,
    18,24,23,0,0,0,0,0,0,26,20,6,0,19,2,8,0,0,14,28,13,0,7,22,5,0,11,29,27,0,
    2,22,26,8,0,0,0,0,23,0,28,1,0,0,30,7,21,13,18,0,0,0,0,10,4,0,24,16,0,29,
    29,0,0,0,0,14,4,9,0,0,13,0,23,3,22,20,0,0,0,0,15,18,2,24,10,16,12,30,0,0,
    0,0,22,24,15,1,25,0,0,0,30,0,0,20,0,29,0,18,8,12,0,27,0,21,0,17,0,4,19,26,
    0,0,3,0,6,2,0,12,29,9,22,7,30,1,21,0,0,26,0,15,14,0,8,17,0,0,0,0,5,25,
    10,30,0,12,25,0,21,0,4,29,11,16,19,0,0,0,0,1,0,0,8,23,0,9,6,0,7,20,0,0,
    0,0,5,25,20,0,0,17,0,4,0,15,10,0,29,0,24,9,0,22,0,0,13,0,8,18,21,0,11,3,
    20,0,0,29,10,7,0,0,15,11,0,0,17,0,3,27,1,0,6,25,28,4,0,0,16,19,18,0,0,0,
    25,12,0,2,14,0,6,29,0,13,9,3,22,0,0,24,0,0,0,0,5,0,0,4,18,0,0,27,20,21,
    0,28,25,0,7,5,8,3,24,23,18,0,0,0,0,15,22,0,0,16,9,0,11,0,20,0,6,0,1,13,
    0,1,2,19,0,4,24,10,20,30,0,0,15,28,0,5,9,23,13,21,0,26,0,0,0,0,0,14,29,0,
    12,0,28,14,0,29,0,24,27,0,26,5,18,0,6,0,16,2,11,0,25,0,30,19,0,10,23,0,0,0,
    0,0,0,0,9,21,26,20,6,0,0,18,14,0,19,0,3,4,0,0,11,16,0,5,29,27,0,2,0,24
  };

  const int d30_375_31[] = {
    30,
    14,0,4,13,17,0,8,21,0,23,0,0,1,24,26,0,20,7,9,16,0,6,11,0,0,0,5,0,0,12,
    0,17,11,22,29,30,0,23,0,3,0,0,0,21,14,0,19,0,0,0,10,27,28,0,2,0,24,5,12,18,
    0,0,0,7,25,5,0,0,16,26,15,23,30,0,0,12,27,13,0,0,0,0,0,17,18,3,14,28,29,0,
    9,12,24,0,0,29,22,19,0,0,25,0,21,27,1,15,30,0,28,0,0,14,8,0,5,16,0,10,0,0,
    20,14,0,6,19,8,30,0,21,0,16,0,4,22,25,7,0,0,23,0,0,10,0,24,0,26,0,0,13,29,
    16,6,19,0,21,0,0,7,4,29,0,30,0,5,22,0,8,3,10,2,18,0,15,0,0,0,25,0,0,0,
    0,26,5,27,0,10,0,0,0,0,1,0,0,16,8,9,0,25,0,0,13,20,18,21,15,17,28,30,0,0,
    8,0,0,0,0,14,19,0,0,0,17,24,7,0,0,29,26,22,13,0,3,1,6,0,28,0,23,25,9,0,
    7,0,14,0,20,27,0,0,22,0,12,3,0,0,23,25,17,26,0,15,0,0,24,6,4,29,0,0,0,9,
    2,15,0,0,0,21,9,0,23,20,0,16,29,14,18,0,0,0,3,0,27,4,0,22,0,0,19,0,8,6,
    0,0,7,0,0,0,21,8,17,16,0,13,22,26,0,2,18,20,0,0,28,0,0,0,29,11,0,14,24,19,
    6,0,0,0,0,0,29,0,30,18,20,0,27,0,2,11,0,21,17,23,0,16,0,0,22,19,9,7,0,3,
    12,0,0,25,14,7,0,30,29,22,10,0,3,15,0,0,1,19,0,4,0,0,16,11,24,0,0,0,28,2,
    19,0,30,1,11,0,17,16,0,14,2,0,0,3,15,0,0,0,7,29,9,0,4,0,25,0,12,0,22,24,
    13,24,0,0,6,0,28,5,15,0,0,18,0,9,0,0,0,0,16,17,1,11,12,3,30,0,2,26,4,0,
    0,21,13,18,9,0,23,10,0,7,0,0,25,6,27,0,0,0,11,8,12,3,0,30,17,24,4,0,0,0,
    0,0,0,20,18,11,0,0,8,19,30,10,2,7,6,0,0,24,0,0,0,0,14,27,0,0,3,29,17,23,
    21,22,0,0,30,0,16,27,7,0,26,0,0,0,10,0,24,0,0,25,20,28,5,14,12,6,0,17,1,0,
    0,27,0,15,13,18,14,29,0,0,0,11,28,2,24,26,0,0,22,0,30,0,17,12,0,0,16,8,0,0,
    0,13,23,17,0,3,18,0,0,28,6,5,0,0,0,16,0,30,0,20,0,0,0,9,1,15,0,12,2,22,
    0,20,6,3,0,0,15,9,0,27,11,0,10,29,0,4,16,18,1,7,0,0,21,2,13,0,0,0,25,0,
    0,3,1,2,0,6,4,0,0,8,28,0,26,13,20,0,0,9,0,0,0,5,22,18,27,23,0,0,11,15,
    0,0,27,5,0,2,0,12,9,0,13,19,11,0,0,6,10,17,29,3,26,0,0,8,0,20,30,0,0,28,
    23,11,25,10,3,0,2,0,0,0,0,20,5,0,0,13,22,8,15,18,6,19,0,0,0,0,21,9,27,0,
    15,19,20,0,1,0,27,3,12,0,0,26,13,0,9,22,0,0,0,6,5,25,23,0,0,7,0,11,0,0,
    1,0,0,0,28,9,0,22,3,24,0,14,0,0,0,30,13,23,0,0,16,0,19,25,6,18,27,0,0,4,
    10,7,0,0,24,0,20,0,1,21,9,0,0,11,0,0,25,0,2,22,17,29,27,0,0,28,0,15,0,8,
    18,29,15,26,0,19,0,6,24,0,3,12,0,0,0,0,28,0,4,1,11,0,0,5,0,25,0,23,30,14,
    11,0,8,0,0,0,0,0,0,6,22,7,15,0,0,23,2,27,0,12,29,9,0,28,14,0,0,19,16,21,
    0,0,9,8,0,16,0,28,20,17,0,15,0,23,21,18,3,14,25,19,0,13,0,0,0,5,29,0,7,0
  };

  const int d30_375_32[] = {
    30,
    23,3,1,17,0,19,0,13,11,12,0,8,16,0,28,0,7,0,5,0,0,25,30,0,0,26,21,0,0,24,
    0,16,3,0,26,0,10,2,28,1,0,0,12,0,29,0,27,4,0,9,0,0,15,0,0,20,6,11,7,0,
    21,8,0,0,4,15,14,12,0,0,16,17,9,19,10,0,0,7,0,0,18,0,0,11,0,27,20,26,6,0,
    0,7,0,0,0,11,19,0,0,0,21,6,8,17,0,0,30,28,15,0,0,9,18,5,20,1,0,24,23,0,
    22,19,23,6,8,28,0,3,10,15,0,0,2,30,26,0,11,0,0,0,27,24,0,0,0,0,0,0,4,18,
    13,0,15,20,0,2,0,9,0,0,22,0,0,0,19,11,8,24,21,14,0,0,17,26,23,0,0,12,27,0,
    0,17,16,0,0,0,0,19,13,5,3,22,7,24,0,0,0,26,25,8,0,28,0,10,0,30,23,29,0,0,
    0,24,0,0,16,0,6,22,17,0,11,3,0,0,7,0,28,2,4,0,30,27,25,0,14,0,0,9,21,19,
    0,6,11,0,12,9,1,16,0,0,0,0,0,0,0,26,29,20,0,4,24,17,0,30,22,0,3,8,14,21,
    26,0,20,0,0,0,0,0,21,22,10,27,17,0,2,23,3,5,0,25,0,29,0,8,9,18,30,19,0,0,
    3,0,18,1,13,0,26,0,24,23,0,0,28,10,0,20,21,0,11,17,0,2,0,0,4,12,29,0,0,8,
    0,12,0,25,3,24,11,0,0,17,0,23,0,6,5,19,0,9,1,28,0,0,7,18,0,8,0,0,0,10,
    29,0,0,30,14,0,23,0,3,9,2,25,0,16,0,27,1,0,19,0,13,18,4,0,21,0,0,10,24,0,
    0,15,28,7,0,0,16,20,5,6,0,0,11,0,23,0,17,0,8,0,22,0,0,3,0,4,18,14,0,13,
    8,23,0,14,7,13,0,0,0,2,26,0,0,4,0,25,24,17,0,0,12,6,1,0,19,0,27,0,0,11,
    4,0,0,9,30,10,18,0,23,21,8,20,0,0,0,0,0,11,27,13,6,0,0,17,16,0,5,0,0,29,
    0,0,4,10,0,17,5,0,0,0,0,0,0,8,30,21,12,18,14,0,2,13,6,22,0,29,1,27,0,15,
    27,1,0,0,0,12,3,17,18,29,13,21,0,0,22,5,0,0,28,0,7,0,0,23,24,16,0,0,25,4,
    6,0,5,0,20,0,8,26,4,0,17,1,13,22,16,24,18,0,0,30,0,0,12,21,25,0,0,0,0,0,
    0,4,0,29,0,14,0,24,26,0,0,5,0,9,20,0,0,10,6,0,21,30,8,12,15,3,11,0,0,25,
    11,0,21,22,0,27,20,0,9,0,0,19,23,15,18,10,16,0,0,0,0,0,3,0,26,0,2,30,0,17,
    17,0,8,27,0,0,0,1,0,0,0,4,10,5,0,18,0,12,0,22,0,14,0,15,6,9,0,23,3,16,
    0,20,26,11,22,0,21,25,0,0,24,28,0,29,0,8,14,23,30,0,4,0,5,0,2,0,0,0,9,7,
    0,26,27,0,5,0,0,30,22,24,28,0,21,0,0,12,0,0,23,7,19,0,10,25,0,2,14,16,8,0,
    14,11,30,0,0,8,0,29,25,7,5,0,1,0,0,28,6,0,0,26,0,3,16,9,0,15,0,13,20,0,
    0,0,2,12,27,0,17,28,0,16,18,7,15,0,1,4,0,0,0,11,25,0,29,14,3,0,0,0,5,20,
    12,13,0,15,28,7,4,8,0,0,9,2,26,20,0,0,0,0,3,5,0,16,14,0,1,23,0,0,0,0,
    1,0,0,18,11,23,24,0,6,4,0,14,3,13,25,0,0,27,0,20,8,5,0,0,0,0,9,17,15,0,
    0,0,7,0,2,21,0,0,0,30,19,0,25,0,24,9,0,0,0,27,14,0,0,16,13,5,8,6,29,26,
    30,5,0,21,0,22,0,0,0,26,0,0,0,11,3,15,20,19,2,29,28,8,0,0,0,0,10,0,13,14
  };

  const int d30_375_33[] = {
    30,
    7,0,0,17,0,28,9,0,30,0,3,0,27,11,0,0,8,0,0,18,4,12,14,0,1,2,0,10,29,22,
    4,6,18,12,22,7,10,0,0,0,13,0,0,8,0,0,0,0,20,0,19,0,16,1,0,27,5,3,30,2,
    0,20,25,0,14,0,22,16,4,23,17,0,12,0,5,18,29,0,0,0,0,0,0,19,0,0,6,24,2,7,
    0,0,0,7,26,20,11,0,0,12,0,30,0,25,0,0,10,24,14,13,8,0,0,4,16,23,29,0,0,28,
    3,9,0,0,0,0,29,0,13,28,0,0,15,21,0,20,0,27,0,1,23,5,0,0,22,0,11,30,19,26,
    29,23,7,30,11,3,4,18,27,0,0,0,8,0,0,5,0,9,2,0,0,0,0,24,0,12,16,0,0,1,
    22,1,28,0,0,8,16,0,7,0,30,3,13,0,0,10,2,6,0,29,0,14,9,26,18,0,0,0,0,0,
    0,0,0,5,1,0,13,29,28,0,2,14,0,0,27,17,0,15,6,30,0,20,26,0,0,25,3,0,9,12,
    0,29,17,15,10,0,12,0,6,2,0,21,14,7,23,8,0,0,0,0,26,0,22,3,27,20,0,0,5,0,
    1,0,26,0,0,19,0,0,0,22,24,10,0,0,30,12,0,21,11,5,20,25,13,17,0,7,8,0,0,14,
    16,13,27,11,19,0,0,21,24,29,18,22,0,9,0,0,0,20,0,6,3,15,23,0,0,0,0,0,28,25,
    0,0,23,0,0,25,0,27,0,9,5,8,30,3,0,6,19,0,0,22,0,0,0,13,2,21,24,26,15,0,
    8,14,9,16,23,29,0,17,0,4,15,0,0,0,20,0,5,0,0,0,18,13,0,6,0,10,19,1,0,0,
    25,16,0,1,0,26,0,28,8,0,0,0,0,0,11,19,7,18,27,20,0,2,0,15,0,30,14,0,6,13,
    17,27,0,6,0,21,0,10,12,0,7,26,18,19,0,0,0,0,3,24,0,16,0,0,13,0,1,25,23,0,
    0,0,0,0,20,18,0,5,0,21,11,19,28,0,15,0,1,16,26,10,0,0,3,22,6,29,0,17,0,0,
    0,21,19,0,0,0,1,0,0,8,9,13,16,20,17,0,4,2,0,0,0,30,0,0,12,18,15,29,7,23,
    0,0,3,0,0,17,0,0,2,24,19,25,0,0,10,15,0,4,7,0,13,0,1,0,8,16,26,5,21,18,
    0,0,0,25,6,13,7,0,22,0,12,28,4,23,9,0,27,30,5,0,29,18,8,0,11,0,0,0,0,16,
    0,15,0,23,9,2,26,1,0,0,0,17,0,4,0,21,0,5,19,28,0,0,24,0,0,14,0,7,16,10,
    11,5,12,8,0,24,0,9,10,16,0,0,21,2,14,25,3,26,29,15,0,0,0,0,30,0,0,18,0,0,
    10,0,20,0,0,0,27,15,0,0,16,0,1,24,26,0,14,28,0,3,9,17,0,8,5,11,0,6,25,0,
    6,19,0,0,0,0,0,22,16,5,0,0,24,30,0,4,0,0,10,2,28,27,17,20,26,0,0,21,14,0,
    0,12,15,28,24,0,0,0,9,20,14,0,0,29,0,13,6,10,0,0,25,11,2,0,21,4,0,27,0,3,
    28,0,6,24,4,0,30,0,21,15,0,20,0,0,12,0,11,0,17,19,5,22,25,0,29,0,2,16,0,0,
    0,0,0,0,0,0,15,0,5,0,0,16,3,6,1,24,20,14,12,23,30,0,21,11,0,0,7,0,13,19,
    26,18,14,0,8,23,0,3,0,10,25,6,0,0,0,29,0,0,0,0,12,28,0,2,0,0,4,11,22,17,
    0,28,0,19,3,0,24,7,0,11,0,5,25,10,16,0,15,0,13,0,14,4,6,9,20,0,23,0,0,0,
    5,0,24,0,17,16,14,6,1,0,4,0,0,12,19,27,25,23,8,0,0,3,10,0,0,26,0,0,0,0,
    14,25,29,27,28,12,6,11,0,0,0,0,2,0,3,30,16,0,0,17,0,0,0,10,4,0,21,9,0,0
  };

  const int d30_375_34[] = {
    30,
    8,30,0,0,0,11,0,0,19,0,28,13,0,9,27,22,20,0,26,7,0,0,12,25,10,4,1,0,0,3,
    0,21,2,15,25,8,18,24,22,11,0,16,0,0,12,0,29,0,0,26,0,6,0,7,0,0,9,0,19,0,
    9,0,23,26,0,0,0,3,0,6,30,0,22,0,13,24,2,16,19,0,4,28,0,0,18,0,12,5,11,0,
    26,19,8,27,22,29,0,0,0,1,0,28,13,6,0,25,0,21,7,0,0,0,11,0,12,0,0,18,0,9,
    0,28,15,20,12,5,0,21,0,0,0,0,29,26,6,0,11,1,25,4,0,2,24,30,0,0,13,0,0,0,
    0,0,0,2,19,0,11,15,16,0,3,0,5,4,0,8,0,23,28,25,0,0,0,24,1,6,7,0,22,0,
    7,1,0,0,15,14,0,0,26,29,10,0,0,12,22,0,0,11,0,0,2,23,13,0,16,17,5,0,20,8,
    0,0,6,13,18,26,12,28,15,10,0,0,0,0,0,14,19,8,0,0,0,7,5,29,17,16,0,3,24,0,
    29,20,27,24,0,0,25,7,11,23,8,0,21,10,0,0,0,19,0,0,12,9,1,15,0,18,0,16,0,0,
    5,11,1,8,4,0,0,14,20,0,0,24,3,2,0,0,0,7,15,6,19,0,17,0,0,28,16,23,0,0,
    0,8,0,9,0,28,0,22,0,3,23,0,0,0,14,0,0,0,27,13,7,4,0,17,25,2,0,11,6,26,
    0,26,0,0,10,2,0,0,7,0,21,0,23,13,15,9,27,29,0,8,0,14,6,18,0,24,0,0,0,4,
    30,0,12,4,0,23,22,8,0,0,2,1,0,0,0,27,13,0,0,20,29,3,0,0,0,5,28,0,10,14,
    0,6,0,0,17,0,19,0,27,24,18,15,30,0,8,13,22,26,0,0,0,21,20,0,14,10,0,0,1,7,
    13,0,22,0,0,25,0,1,0,0,0,3,28,0,17,30,0,2,24,16,0,15,14,0,21,0,8,26,0,20,
    20,15,0,17,14,0,13,19,0,4,0,0,0,28,0,21,8,9,10,27,25,1,23,0,0,0,6,0,0,16,
    12,0,13,0,0,17,0,5,1,18,25,10,9,0,0,0,6,0,0,11,16,0,0,3,4,0,30,15,28,0,
    19,27,10,30,2,0,20,12,0,0,0,0,0,0,7,5,4,15,0,24,0,0,0,0,0,13,14,25,21,17,
    0,0,0,0,0,30,15,0,13,0,5,8,18,23,1,28,9,14,21,29,20,0,0,6,0,11,2,0,0,0,
    27,10,0,18,0,13,17,0,0,19,0,11,7,21,0,1,0,25,12,28,3,16,29,0,0,0,15,0,0,30,
    0,0,19,0,13,3,26,20,0,15,6,25,4,16,0,0,21,10,0,9,22,0,0,0,0,0,0,17,18,1,
    28,0,0,0,24,21,10,9,0,27,0,0,14,7,11,0,3,22,29,0,6,0,0,0,30,1,25,0,23,18,
    1,17,30,0,0,0,29,0,0,12,0,7,10,3,21,2,0,27,5,0,13,0,4,0,0,22,0,14,9,0,
    14,0,7,0,0,0,0,18,24,0,22,27,19,0,25,0,26,0,1,0,9,17,30,4,3,0,0,29,8,6,
    0,24,14,29,1,6,4,0,0,0,0,12,0,8,19,0,0,0,0,15,18,0,0,20,13,21,26,10,7,25,
    6,0,28,0,21,0,0,26,4,5,27,20,0,19,0,0,0,0,3,14,0,30,0,9,7,0,11,8,29,0,
    0,0,16,6,0,24,7,0,10,17,9,18,0,0,4,12,0,0,23,0,5,19,25,28,20,0,27,21,0,0,
    0,18,3,1,20,16,5,4,14,25,0,2,0,22,0,6,23,0,0,0,0,0,7,10,11,0,0,24,0,21,
    25,0,0,23,16,0,9,0,3,0,14,29,0,0,20,11,0,0,8,18,15,13,0,19,5,7,0,0,0,2,
    3,22,0,19,0,0,0,0,29,16,26,0,17,0,0,23,12,0,13,0,10,27,8,14,0,15,0,2,4,24
  };

  const int d30_375_35[] = {
    30,
    0,17,23,0,20,26,0,19,24,12,0,5,0,0,27,3,25,0,8,14,7,29,2,0,0,0,0,28,15,0,
    0,19,0,3,27,0,25,5,28,29,23,0,6,0,21,26,0,2,4,0,17,1,7,0,0,0,16,10,0,0,
    0,0,0,0,25,17,1,0,23,0,8,0,10,28,0,0,21,24,12,0,0,3,26,13,20,30,15,9,0,19,
    20,0,27,0,3,0,4,24,0,7,0,1,0,12,5,29,17,23,0,15,0,8,0,11,26,14,19,0,0,0,
    0,14,0,0,1,0,0,0,4,0,19,24,22,0,0,25,0,0,0,0,11,6,17,29,23,5,26,2,7,8,
    11,7,0,0,0,0,6,30,0,24,4,25,0,26,1,0,13,28,16,2,0,0,0,15,21,10,0,0,20,23,
    16,0,0,26,22,23,0,7,8,0,11,0,0,9,30,0,18,3,13,0,0,0,27,4,28,0,12,15,21,0,
    28,10,17,0,0,0,0,23,19,27,0,9,0,6,0,0,11,29,0,0,26,5,20,18,0,7,30,12,0,4,
    15,9,4,8,0,27,3,29,0,18,0,20,0,0,0,23,12,0,0,0,0,0,19,26,16,0,14,11,28,13,
    0,0,5,2,15,9,0,21,7,16,0,0,8,0,18,1,6,20,25,0,13,0,0,0,11,0,10,23,0,29,
    0,0,18,28,13,0,19,0,29,23,16,0,0,24,0,0,20,30,0,6,0,0,3,0,10,26,17,7,25,0,
    0,0,0,24,0,4,0,14,0,17,27,3,18,0,28,20,0,21,22,29,0,26,0,12,6,9,2,0,0,0,
    22,0,28,17,6,3,15,0,0,0,0,0,0,2,0,11,7,14,0,26,4,0,12,27,0,0,0,21,16,24,
    0,27,22,13,17,6,11,0,0,21,0,0,5,0,0,15,26,16,0,10,12,30,29,0,19,0,24,0,0,14,
    5,0,19,0,21,20,2,0,26,10,0,13,0,7,0,0,1,0,0,28,0,0,16,0,14,18,4,0,17,12,
    18,26,0,5,0,0,20,0,14,0,24,0,23,1,16,0,0,7,10,0,21,15,25,0,9,0,0,17,0,30,
    0,2,26,0,0,0,21,4,0,14,0,22,17,3,9,0,0,0,29,19,0,27,10,0,15,0,8,5,6,20,
    23,24,0,0,12,0,0,0,20,11,0,19,30,13,6,14,0,0,9,0,18,0,1,7,0,29,0,0,27,16,
    0,22,0,0,0,0,23,12,17,0,1,0,2,0,0,30,0,19,3,16,5,28,0,20,8,11,0,13,18,0,
    0,8,2,6,18,15,0,17,9,0,0,27,21,29,4,0,3,0,1,12,30,0,0,0,0,24,0,0,0,7,
    21,0,0,18,7,24,0,22,0,0,10,15,0,17,20,0,19,0,0,4,0,25,14,5,0,27,0,0,29,9,
    0,0,0,1,0,19,5,0,6,15,28,0,0,10,24,22,9,8,7,20,0,14,0,25,17,0,0,0,0,26,
    8,0,0,16,11,2,14,0,0,20,17,30,1,0,0,0,0,22,15,0,9,19,0,3,0,13,5,4,10,0,
    29,30,16,21,0,0,13,11,10,3,6,8,7,0,0,18,0,0,0,0,14,0,0,24,0,20,1,0,22,0,
    2,6,0,29,4,18,7,9,0,28,0,0,0,22,25,16,0,0,0,5,0,24,8,0,30,15,3,0,0,1,
    13,4,20,0,0,0,0,6,25,0,15,12,26,0,29,19,0,10,0,0,27,0,28,17,18,23,22,0,11,0,
    0,21,11,22,24,7,0,0,0,6,29,0,19,0,0,0,8,18,27,3,28,0,15,0,0,0,0,20,13,25,
    3,28,13,0,16,0,0,20,27,0,14,21,0,25,7,2,10,0,19,17,8,11,0,30,0,0,0,18,0,0,
    14,3,24,15,0,13,10,0,0,0,21,17,28,16,26,9,0,0,0,0,25,12,0,0,1,0,0,30,8,0,
    27,0,1,11,0,21,0,0,0,0,5,0,25,0,22,6,0,17,2,8,15,0,0,23,0,12,13,26,4,0
  };

  const int d30_375_36[] = {
    30,
    8,0,11,0,0,0,0,23,0,5,21,9,0,0,28,22,0,0,14,0,2,7,13,29,0,26,6,1,18,24,
    0,11,0,23,28,0,0,30,21,0,3,7,16,4,0,14,0,5,15,29,0,0,1,13,9,18,0,12,0,0,
    12,0,1,14,7,10,9,8,29,16,0,0,0,24,0,0,0,2,0,5,22,13,0,0,0,11,0,0,17,23,
    23,17,5,0,8,6,1,0,0,15,0,30,9,11,0,0,28,0,27,0,0,0,0,14,4,0,13,2,0,20,
    0,25,0,8,26,0,12,18,0,0,13,0,15,1,4,30,16,20,0,27,0,0,0,7,0,10,0,9,21,0,
    2,15,18,22,0,9,0,0,0,7,0,14,0,23,0,0,0,21,3,0,0,17,16,20,5,0,4,28,0,1,
    0,9,0,0,16,0,0,0,26,0,22,0,28,0,13,10,15,4,0,19,3,30,0,17,0,12,27,0,29,2,
    29,12,10,0,3,19,2,0,0,0,4,6,27,25,0,0,1,0,13,7,0,21,5,0,20,24,0,0,0,9,
    6,24,27,15,0,0,0,20,13,0,17,0,0,9,0,11,0,10,28,23,0,25,4,0,18,0,0,16,3,19,
    0,30,16,28,1,11,27,0,0,0,20,0,12,8,15,26,0,19,5,0,0,0,0,23,13,3,22,0,25,0,
    14,0,26,0,9,20,3,22,0,24,15,0,0,0,7,5,13,0,0,0,6,0,11,0,17,0,0,25,8,12,
    0,0,0,16,18,0,0,19,0,13,14,2,0,5,12,28,3,26,10,0,0,8,30,27,22,0,21,0,0,0,
    0,16,14,24,0,0,21,10,0,3,0,12,23,29,25,17,27,28,0,6,0,0,26,1,0,22,8,0,0,0,
    28,1,21,30,10,26,0,0,24,0,0,18,0,0,29,2,6,11,0,13,0,27,0,3,0,23,12,0,0,14,
    0,0,0,0,30,0,13,3,5,0,24,15,0,28,1,12,0,14,2,25,26,0,0,0,27,0,10,7,23,0,
    1,0,20,17,0,13,22,0,18,4,0,0,0,27,16,0,19,0,25,26,30,28,6,0,0,0,2,0,9,0,
    15,0,0,0,19,16,0,0,22,21,0,13,20,26,0,24,0,30,18,0,0,4,29,0,1,0,28,0,2,25,
    18,4,23,0,20,22,0,0,0,12,16,29,26,0,10,13,0,0,0,1,14,0,0,24,25,27,0,15,0,6,
    27,23,0,0,0,4,0,14,0,0,30,24,0,7,0,0,5,6,21,16,1,2,19,0,10,0,29,26,13,0,
    0,10,0,0,0,0,19,7,3,0,28,23,24,13,30,27,17,0,4,11,5,0,0,12,0,1,18,20,0,0,
    0,21,15,12,0,29,0,0,1,27,0,0,25,0,3,0,0,0,11,0,17,22,14,2,8,16,23,0,5,0,
    4,19,0,2,0,0,24,25,0,14,0,0,0,0,9,8,23,12,17,10,0,1,21,0,0,29,0,30,6,15,
    0,0,7,1,0,15,26,4,10,2,0,0,13,0,0,0,12,18,19,0,0,24,23,25,11,0,0,3,0,21,
    0,0,3,7,13,0,16,0,2,23,26,1,6,0,0,0,9,8,0,0,10,0,20,0,14,19,0,24,15,18,
    26,29,2,5,0,0,0,27,4,9,11,21,0,30,18,6,0,22,0,8,13,0,28,16,0,0,0,23,0,0,
    11,0,9,25,0,0,15,13,0,22,27,0,19,0,0,23,0,0,12,14,16,18,0,0,0,28,26,0,7,29,
    30,0,0,0,27,12,29,0,28,0,0,25,11,0,0,15,20,0,0,9,4,0,2,10,16,14,24,5,22,0,
    20,0,0,19,14,25,5,26,23,0,12,0,4,0,0,0,11,15,0,17,27,9,0,6,7,0,1,18,0,0,
    0,13,0,0,0,21,23,0,8,26,29,28,14,22,5,0,2,0,6,0,15,16,0,0,0,17,0,0,11,10,
    22,0,28,26,17,24,0,15,27,0,0,20,30,14,2,0,21,0,0,0,11,23,9,0,0,0,0,0,1,3
  };

  const int d30_375_37[] = {
    30,
    24,0,0,0,0,30,29,28,4,7,0,0,14,26,0,0,25,2,0,13,5,21,27,16,11,20,0,6,0,0,
    28,7,0,24,0,10,0,0,0,0,0,18,25,3,19,12,8,0,1,5,0,20,0,4,2,14,0,0,0,23,
    27,14,15,3,0,17,0,0,2,5,21,0,0,23,8,0,0,0,13,28,9,10,26,0,7,0,0,29,0,1,
    18,0,17,0,0,25,3,0,16,30,10,15,0,0,14,0,0,23,0,12,1,29,0,0,0,2,24,26,0,22,
    20,0,30,23,0,9,8,26,0,0,7,0,0,0,0,1,18,22,25,2,15,0,10,19,27,0,0,0,14,0,
    0,5,0,0,0,21,0,0,14,9,27,17,0,30,13,0,0,12,28,0,0,6,29,7,16,0,4,19,23,20,
    1,0,24,0,6,27,0,21,22,0,0,0,12,18,17,0,15,0,2,0,0,13,0,0,19,7,11,0,28,5,
    6,0,10,0,23,0,27,13,0,0,2,25,0,4,12,3,0,5,0,9,0,11,21,18,22,26,0,0,29,0,
    0,1,21,0,17,0,24,10,9,27,8,19,4,12,0,18,0,0,0,0,0,22,0,28,23,25,7,0,11,0,
    0,0,0,0,0,15,0,22,0,8,4,13,24,0,16,20,0,14,0,23,28,1,25,0,0,17,5,30,0,26,
    0,15,13,25,27,0,0,0,29,6,11,24,7,0,0,30,0,0,19,17,23,16,12,0,0,9,0,1,22,0,
    13,0,0,0,10,22,18,5,15,4,16,2,27,21,0,11,0,26,0,0,0,0,0,17,12,6,20,0,0,19,
    0,19,6,2,0,0,17,11,18,10,0,0,28,29,0,8,0,30,0,20,0,25,0,24,0,4,15,5,7,0,
    0,11,3,26,0,16,0,0,21,12,19,6,10,0,0,22,7,0,0,14,29,0,2,0,17,0,0,18,0,28,
    3,0,0,28,12,0,15,24,0,11,26,0,29,19,2,0,0,0,5,10,0,0,0,20,6,18,16,0,0,30,
    0,18,8,7,0,0,28,6,0,0,12,20,22,0,0,4,0,21,14,0,2,0,0,5,30,10,25,0,1,0,
    29,0,7,22,0,0,25,20,5,18,17,14,26,24,23,0,21,0,4,1,0,19,11,0,0,0,0,0,9,0,
    19,4,0,0,22,0,0,0,30,29,28,0,0,0,20,6,1,11,17,0,14,0,24,25,0,0,18,21,5,0,
    22,0,0,17,18,20,19,0,13,26,24,0,30,0,11,29,23,8,0,15,7,0,0,0,0,0,9,14,12,0,
    0,6,28,20,3,1,23,7,10,0,0,5,0,0,0,24,16,0,8,18,0,0,22,0,0,0,12,15,30,0,
    4,23,14,19,1,0,2,0,0,0,3,28,0,16,0,15,12,0,18,0,22,26,6,30,0,29,0,0,0,9,
    15,3,0,13,14,0,12,0,0,16,0,26,0,0,18,0,9,24,0,0,30,0,0,27,29,0,28,4,25,17,
    0,0,1,14,20,26,30,0,27,0,0,0,5,11,0,0,22,0,0,0,0,4,3,29,21,16,17,23,0,8,
    5,13,9,0,8,23,1,12,0,0,0,29,16,7,30,0,6,10,0,0,19,0,0,0,0,11,0,27,15,25,
    26,20,0,18,9,28,16,0,0,0,0,10,0,5,21,7,0,25,30,0,3,8,0,1,13,0,0,0,0,4,
    0,0,0,15,0,0,26,1,0,25,0,0,0,14,28,13,29,6,23,8,0,2,9,0,18,21,30,0,10,24,
    0,0,19,12,15,0,0,8,7,0,0,11,13,6,0,0,30,0,21,16,4,0,0,2,20,0,26,25,18,0,
    0,12,0,27,21,18,0,9,8,2,0,0,11,0,0,0,14,13,0,6,24,0,30,0,0,19,0,22,17,3,
    30,16,20,0,24,0,0,0,0,0,14,4,17,22,27,25,26,19,9,0,8,18,5,0,0,0,0,0,6,12,
    23,28,0,0,0,6,0,16,20,1,0,0,0,0,5,0,2,29,27,0,10,15,18,8,24,30,13,11,0,0
  };

  const int d30_375_38[] = {
    30,
    0,0,0,0,1,18,8,0,17,4,5,16,6,20,30,0,0,0,26,0,23,0,22,0,15,0,0,3,9,14,
    9,0,0,8,0,0,12,0,21,16,0,25,0,0,18,14,0,4,24,0,2,15,5,19,0,20,0,29,10,0,
    12,15,29,25,27,28,0,4,13,1,21,22,30,0,0,0,5,0,11,0,0,0,19,2,23,0,0,0,0,0,
    11,22,14,1,16,23,28,0,0,0,27,21,0,2,13,0,0,0,7,0,0,0,0,0,30,12,15,24,26,4,
    20,28,16,27,0,25,21,0,11,15,0,0,0,22,0,18,19,29,30,0,0,9,0,3,0,0,8,12,14,0,
    1,30,13,0,0,0,18,7,0,23,0,0,20,0,26,28,0,0,16,4,14,0,0,12,8,0,0,6,25,11,
    8,0,0,6,0,0,2,3,15,0,14,13,23,0,5,0,27,30,0,0,9,17,0,4,0,11,12,0,20,0,
    0,3,0,0,12,0,19,15,0,21,7,0,4,28,2,16,24,11,5,17,0,0,9,0,6,18,0,0,0,10,
    27,0,0,0,11,0,0,0,0,18,15,30,25,17,8,0,0,0,29,22,5,21,0,10,14,1,3,0,0,16,
    0,21,18,7,17,0,0,29,0,6,23,10,0,24,0,0,28,0,0,5,8,11,20,22,16,0,9,27,0,0,
    0,0,0,0,0,6,0,10,27,22,0,8,29,14,19,0,20,17,25,2,16,13,21,18,0,0,26,0,0,0,
    0,25,8,3,28,0,0,0,26,0,4,18,24,0,11,6,10,0,23,7,0,20,0,0,9,19,0,22,0,2,
    19,0,20,5,0,29,15,24,0,17,0,23,0,0,16,8,0,0,0,0,26,1,10,11,0,27,4,7,6,0,
    0,0,0,11,7,0,6,25,10,0,0,0,22,0,28,2,9,3,21,16,27,12,14,0,20,0,0,0,0,17,
    29,16,22,14,8,0,0,0,9,11,0,0,7,23,0,5,1,25,0,15,0,18,30,17,0,6,28,0,0,0,
    0,29,3,12,0,21,0,0,6,7,8,2,13,0,0,0,11,27,0,30,4,10,24,0,0,0,14,16,22,0,
    2,8,1,16,23,24,0,19,4,5,29,26,0,12,0,0,25,0,0,0,0,0,0,7,0,0,21,15,27,0,
    0,0,9,22,0,14,0,28,0,0,20,0,8,16,6,11,2,13,0,18,19,0,1,26,0,24,0,23,0,15,
    15,0,0,0,9,16,20,0,1,28,0,0,0,30,0,4,0,0,27,10,0,23,18,0,21,0,24,19,29,8,
    28,7,0,0,14,3,17,2,30,0,10,4,0,0,0,0,16,22,0,27,24,26,13,0,29,0,0,0,23,6,
    13,14,0,0,0,26,0,6,3,0,22,28,0,18,0,25,23,21,0,12,0,29,0,24,0,30,7,0,0,20,
    0,0,26,29,0,0,30,17,0,0,0,0,0,11,24,13,0,28,20,19,25,0,0,23,1,2,10,0,3,18,
    17,11,27,18,0,19,4,9,16,0,2,0,0,0,0,10,0,0,22,0,0,14,6,8,25,0,13,26,0,12,
    0,0,24,0,0,0,14,0,23,0,25,27,28,26,0,7,0,20,1,8,21,0,12,0,2,5,0,17,18,22,
    5,24,30,0,0,27,0,0,0,25,17,0,26,13,1,0,4,0,0,0,11,16,3,0,10,9,0,0,7,19,
    0,27,0,10,4,20,5,0,0,0,0,9,15,29,21,26,0,14,6,0,28,24,0,0,19,8,16,30,0,0,
    10,0,2,21,20,1,11,16,7,0,0,0,27,4,0,23,14,6,0,0,15,0,0,5,0,22,0,0,28,9,
    0,1,17,0,18,0,9,14,0,8,0,0,0,0,7,15,22,19,28,24,0,6,29,0,0,23,2,0,11,0,
    18,0,0,0,6,11,27,21,0,24,30,0,9,0,20,0,0,12,14,3,0,0,0,0,7,4,0,2,8,26,
    7,12,4,20,24,2,0,0,0,27,0,5,0,10,17,0,26,9,0,0,30,0,28,21,0,16,22,8,0,0
  };

  const int d30_375_39[] = {
    30,
    19,14,11,16,1,0,0,0,0,10,0,18,5,13,27,0,2,0,17,26,12,0,0,0,15,30,0,0,0,3,
    0,0,23,7,0,4,3,0,0,6,14,0,19,12,0,11,27,17,24,13,2,0,0,9,0,8,0,5,26,0,
    11,3,10,0,30,0,0,15,6,22,27,0,0,2,0,0,7,9,12,0,0,26,20,4,23,0,28,0,0,1,
    0,0,13,21,0,0,4,19,0,0,15,24,22,6,16,2,0,0,7,30,0,0,5,0,12,0,9,28,0,26,
    1,26,0,23,0,29,14,0,19,0,0,22,7,0,28,10,30,15,0,11,0,27,0,0,16,0,6,9,3,0,
    15,0,29,0,0,0,0,3,18,12,9,14,20,0,22,0,0,11,25,0,4,5,7,0,0,16,0,26,8,23,
    0,0,0,0,0,5,17,0,10,1,26,21,9,0,25,20,4,0,0,6,0,0,3,14,30,0,19,13,28,15,
    18,27,25,4,17,8,0,2,0,0,23,0,21,26,20,0,0,0,30,19,15,10,29,0,0,0,0,7,0,0,
    0,24,0,19,0,9,7,12,23,17,0,0,0,0,2,0,0,27,4,10,0,13,28,11,25,29,0,8,0,20,
    0,1,0,0,0,0,15,0,0,29,17,5,0,8,12,0,0,26,27,24,21,30,18,0,0,0,4,2,20,10,
    12,0,0,1,27,0,13,24,0,26,0,28,0,4,0,0,17,30,20,25,0,3,0,6,0,9,23,18,0,22,
    26,0,0,0,16,0,6,27,29,20,3,15,0,10,30,5,0,12,0,0,0,0,24,17,8,28,0,0,21,0,
    5,13,28,27,0,22,25,0,17,0,0,0,0,16,15,12,21,0,0,0,18,0,23,10,4,19,0,0,0,9,
    7,16,26,28,25,21,18,0,0,15,4,0,0,3,0,0,29,0,8,0,9,22,0,20,11,1,0,0,10,0,
    0,5,9,25,28,0,0,20,0,0,2,3,12,0,23,0,14,4,0,17,0,0,11,0,0,15,1,6,18,30,
    30,8,0,9,0,6,0,0,0,24,12,0,0,28,29,14,22,2,0,20,26,0,0,0,3,23,0,19,17,0,
    8,0,2,15,21,0,30,28,20,18,0,0,6,0,0,23,0,0,0,0,19,17,27,12,0,14,22,0,24,0,
    25,23,14,0,12,0,22,7,0,0,29,11,0,21,24,9,0,6,5,0,3,0,8,15,18,0,0,0,0,19,
    0,18,0,5,0,0,0,21,13,0,0,25,0,27,10,0,23,7,0,0,8,11,15,2,1,0,14,22,19,0,
    0,0,0,14,18,23,0,30,15,0,8,19,28,7,0,26,0,5,29,0,0,0,0,16,20,0,27,24,11,6,
    16,0,22,20,14,24,19,0,0,0,0,0,29,0,13,8,11,0,26,0,0,0,17,0,0,10,25,27,30,21,
    0,2,0,0,8,17,0,11,0,21,0,12,1,0,0,18,6,23,0,3,22,25,0,19,26,20,15,0,13,0,
    2,11,30,0,29,10,0,18,16,4,0,6,26,5,0,15,0,0,21,0,7,8,0,0,0,0,3,0,9,0,
    13,0,0,0,6,19,0,0,9,8,16,20,0,0,14,24,0,29,15,2,0,23,30,0,10,7,26,0,0,18,
    0,25,24,30,3,15,9,0,5,0,0,0,23,0,0,0,18,0,13,0,11,12,10,22,0,26,20,0,14,0,
    0,21,0,29,0,2,0,13,0,23,30,16,11,24,0,0,3,0,0,7,0,9,4,28,0,27,0,15,0,25,
    0,30,20,0,0,3,0,0,27,0,0,0,4,0,7,22,12,21,0,18,13,28,25,8,6,11,16,0,0,5,
    0,0,27,0,0,11,24,0,4,13,28,10,0,0,0,0,9,0,18,16,25,1,0,5,22,0,29,20,12,0,
    20,17,16,0,15,0,28,6,1,0,19,0,14,29,5,21,0,13,0,0,0,0,0,7,0,0,30,11,0,24,
    10,0,15,8,4,0,26,1,21,9,0,0,3,0,0,27,24,0,0,29,14,0,0,0,17,18,0,25,2,16
  };

  const int d30_375_40[] = {
    30,
    0,18,10,0,7,0,9,1,17,14,24,25,13,0,0,19,0,0,28,0,5,8,0,0,23,0,0,15,26,12,
    0,0,23,3,14,7,0,11,0,9,10,0,0,15,25,1,0,0,26,0,8,0,27,30,12,0,22,6,21,0,
    0,0,25,19,2,0,0,0,21,0,30,28,15,0,1,3,0,12,8,6,14,0,0,0,20,0,27,24,17,10,
    15,17,3,0,16,0,22,14,0,1,0,0,26,30,0,0,21,0,6,28,0,13,0,0,25,18,29,23,2,0,
    5,0,30,14,15,0,21,23,24,7,6,0,8,0,28,0,3,19,0,0,0,11,0,29,0,0,0,22,16,18,
    0,20,27,22,0,0,6,0,16,30,29,0,2,11,0,5,0,26,1,0,12,0,17,4,13,0,0,0,0,15,
    12,23,4,28,0,24,0,9,0,0,13,14,0,0,17,27,0,25,0,1,0,26,0,8,0,20,10,0,11,0,
    26,0,22,0,13,23,15,0,7,21,0,0,19,10,0,17,0,0,30,12,4,18,20,0,1,8,0,0,0,3,
    0,0,0,0,0,6,5,12,0,0,0,16,0,9,10,14,20,17,23,4,0,30,7,0,11,19,3,0,25,2,
    11,12,0,0,0,15,0,0,4,20,0,13,5,0,0,0,1,0,17,18,2,0,0,25,26,14,24,16,6,8,
    24,0,17,0,25,22,23,0,0,0,1,0,0,13,0,7,6,0,21,0,16,20,3,19,0,2,28,27,0,5,
    7,5,26,0,0,0,19,0,0,25,28,0,24,0,15,0,0,8,20,2,6,10,18,12,4,0,0,0,23,0,
    0,8,16,0,0,1,0,22,18,4,0,0,0,0,29,12,2,7,0,24,0,0,10,26,0,25,13,0,28,14,
    0,0,14,26,0,29,16,0,12,0,0,2,0,0,30,28,27,9,15,7,24,0,23,0,0,4,8,0,0,20,
    0,30,24,0,21,18,14,5,22,0,17,0,28,12,19,0,7,0,0,10,0,0,9,0,27,16,0,11,0,0,
    28,29,0,0,5,0,0,2,30,0,0,12,25,14,0,4,16,20,24,27,0,0,21,0,0,13,19,0,0,23,
    0,26,0,30,11,0,0,0,20,8,0,1,0,29,2,0,10,24,0,9,28,19,0,18,21,3,0,0,13,4,
    6,0,9,16,4,30,13,15,0,18,11,27,23,0,26,24,0,0,0,0,0,21,0,5,0,0,20,0,3,22,
    30,14,0,10,22,0,0,24,13,0,0,0,29,18,4,0,9,5,0,0,17,0,0,23,0,0,15,8,27,7,
    3,13,20,21,0,0,30,8,15,22,7,0,18,5,0,0,17,10,16,0,0,14,11,0,0,28,0,0,24,0,
    21,0,6,0,12,10,0,20,0,2,8,11,0,0,22,0,0,0,0,3,1,9,26,27,7,0,0,0,29,19,
    0,0,0,1,30,19,10,17,27,24,3,0,0,26,0,0,25,23,9,14,0,15,2,0,0,0,16,28,0,0,
    13,0,0,0,28,4,0,0,0,3,25,8,1,0,9,0,0,21,0,23,27,16,15,20,29,12,30,0,0,6,
    0,11,0,5,8,14,0,3,28,0,19,7,0,24,27,18,13,0,0,0,26,29,22,21,0,0,23,25,0,0,
    27,0,5,4,0,8,12,0,10,0,0,30,0,19,20,9,0,22,7,11,0,0,0,2,0,21,1,17,18,0,
    0,2,0,11,0,5,0,0,0,26,0,10,0,0,18,6,19,15,4,8,0,28,0,3,16,9,0,29,20,0,
    0,28,0,13,9,0,3,27,26,15,0,23,0,25,0,0,0,4,11,0,19,5,0,6,0,30,0,10,0,21,
    8,0,0,0,18,28,0,6,0,0,4,0,16,23,0,22,14,0,12,0,10,0,13,9,30,7,5,19,0,0,
    18,4,8,27,0,13,17,0,0,19,0,26,21,28,6,0,24,0,0,0,20,25,0,0,22,10,0,12,0,0
  };

  /*
   * Instances taken from examples that ship with the generator
   * "lsencode" by Carla Gomes <gomes@cs.cornell.edu>.
   */

  const int d33_381[] = {
    // Size: 33 x 33
    33,
    // Pre-assigned fields
    0,0,2,3,28,19,14,33,21,17,0,15,0,0,0,26,4,8,32,16,20,6,29,31,0,0,25,11,0,0,0,0,10,
    19,0,0,24,26,9,21,30,0,12,20,0,0,10,33,14,32,0,6,0,11,0,0,28,0,0,23,7,4,0,18,16,22,
    0,1,28,0,18,0,19,20,0,5,8,17,0,12,27,0,24,23,0,9,0,32,0,22,10,7,0,29,0,31,3,26,16,
    7,16,9,14,27,0,32,0,0,0,31,0,0,5,0,6,25,33,0,0,10,2,0,30,19,0,0,26,12,13,1,22,18,
    11,10,4,31,25,0,8,24,0,30,27,0,0,0,19,0,29,32,17,0,18,28,22,6,7,9,0,20,15,0,0,0,0,
    24,0,3,9,33,0,5,21,16,31,0,6,25,22,29,0,30,0,10,0,0,11,18,0,1,26,28,0,0,4,0,15,19,
    31,32,20,5,0,29,23,0,33,3,17,4,26,19,16,0,22,0,15,0,28,0,0,12,24,6,7,30,13,0,0,0,0,
    5,0,6,0,0,2,26,13,10,0,1,24,11,30,4,31,33,29,0,0,19,25,0,16,32,14,0,21,0,0,22,3,0,
    0,0,0,0,10,17,12,23,32,18,14,33,19,8,0,3,0,26,0,22,21,1,11,0,0,0,20,27,31,0,9,13,6,
    32,33,0,8,5,0,0,4,28,22,24,0,15,0,0,10,31,21,0,26,0,0,7,0,6,1,13,0,30,19,17,29,9,
    0,29,0,19,2,31,3,6,12,9,18,0,4,0,21,17,7,0,26,33,32,0,0,8,0,0,22,23,0,20,0,10,0,
    1,15,0,0,0,33,24,18,11,19,2,22,29,0,31,32,16,5,27,0,23,7,12,0,25,0,0,9,0,8,0,0,14,
    25,24,12,0,15,13,0,0,18,20,0,0,21,33,32,2,14,0,28,0,9,29,6,19,0,0,0,0,0,22,16,8,31,
    30,11,18,33,29,0,0,0,0,21,28,14,10,2,13,0,0,0,5,31,7,0,19,26,16,22,27,4,0,1,6,0,0,
    0,22,26,27,0,4,0,0,29,0,30,0,5,11,0,0,0,24,16,6,12,0,25,18,31,8,0,28,19,3,13,2,0,
    14,23,0,0,0,10,0,12,0,2,26,0,3,0,7,18,0,31,0,17,29,27,30,0,20,32,15,19,6,28,0,9,0,
    4,27,19,0,31,0,6,28,3,0,0,16,17,0,22,0,8,13,0,32,0,18,2,0,0,0,0,24,21,12,15,33,25,
    15,0,0,12,7,25,20,8,22,23,0,0,0,21,10,0,26,2,0,29,5,0,24,17,0,0,6,13,14,11,0,0,27,
    8,4,0,21,12,11,31,0,15,0,0,30,0,9,20,0,0,1,23,0,13,10,3,14,26,0,24,0,22,27,2,5,0,
    0,0,0,29,0,28,18,1,9,25,11,19,6,15,0,33,12,10,0,0,0,5,26,32,0,2,30,0,17,21,31,0,20,
    0,7,0,0,4,18,0,0,30,0,25,32,20,29,24,0,17,0,14,2,27,15,0,21,0,12,9,0,8,0,10,19,3,
    20,25,31,11,0,0,0,26,14,0,7,0,1,13,12,0,21,0,8,3,0,0,0,5,29,0,4,15,2,10,24,32,17,
    0,0,10,0,0,6,11,22,0,13,0,21,16,20,23,4,0,3,2,14,0,0,33,0,15,29,32,25,18,17,0,31,0,
    0,0,16,0,20,1,0,32,24,0,0,27,22,0,0,21,23,4,33,28,31,0,14,11,17,15,3,10,29,0,8,0,0,
    27,0,14,0,6,15,25,10,0,32,22,3,13,17,0,20,1,0,18,0,0,8,9,0,23,31,5,0,16,26,0,0,0,
    18,30,0,7,22,26,4,19,27,0,0,25,8,0,0,5,0,17,21,12,0,16,28,0,0,13,2,33,0,0,20,14,0,
    0,12,0,26,13,16,30,0,1,0,0,0,0,0,8,7,0,0,3,15,22,33,32,20,9,5,0,0,11,29,19,27,21,
    13,0,8,30,0,14,0,0,31,16,21,18,0,32,9,1,0,7,0,20,0,3,0,27,12,10,0,0,33,23,25,0,4,
    0,0,27,16,11,0,0,15,0,33,0,5,0,25,0,24,0,22,1,30,3,9,17,13,0,4,12,8,10,0,26,18,29,
    16,20,22,32,0,0,10,2,0,15,6,7,0,27,0,23,0,0,11,18,25,13,0,0,33,21,1,0,3,30,0,0,26,
    3,26,21,15,1,5,0,0,2,11,9,29,18,28,0,30,0,12,31,0,0,19,0,23,14,0,0,32,0,0,4,7,8,
    10,2,17,0,9,0,29,0,0,1,3,0,0,0,5,12,27,0,0,11,15,20,4,33,0,16,31,0,28,25,30,21,13,
    17,31,5,10,0,0,0,27,23,0,15,12,9,0,6,29,11,25,13,19,24,0,8,0,2,33,18,0,0,0,0,20,30
  };

  const int d35_405[] = {
    // Size: 35 x 35
    35,
    // Pre-assigned fields
    1,2,3,0,5,6,0,8,9,10,11,12,13,14,0,16,17,0,0,20,0,22,23,0,25,0,27,28,29,30,31,32,33,0,0,
    2,3,4,5,0,0,8,9,10,11,12,0,14,0,16,0,18,19,20,21,22,23,24,25,0,27,28,29,0,31,32,33,34,0,1,
    0,4,5,0,7,8,0,10,11,0,13,14,15,16,0,0,0,0,21,0,23,24,25,26,27,0,29,0,31,32,33,34,35,1,2,
    0,5,0,7,0,9,0,11,0,13,0,0,16,0,18,0,20,0,22,23,0,0,0,27,0,29,30,31,32,33,34,0,0,2,3,
    5,6,7,8,0,10,11,12,13,0,15,16,17,0,19,20,21,22,0,24,25,26,27,28,29,30,0,32,0,34,35,1,2,0,4,
    6,7,8,9,10,11,12,13,14,15,16,17,18,19,0,21,0,23,24,0,26,27,28,29,0,0,32,33,34,35,1,0,0,4,5,
    7,0,0,0,0,12,13,14,0,16,0,0,19,20,21,22,0,24,25,0,27,0,29,30,31,0,0,0,35,1,2,0,0,5,6,
    0,0,0,0,12,13,14,0,0,0,18,0,20,21,22,23,24,0,0,27,28,29,30,31,32,33,34,0,1,0,3,4,0,6,7,
    9,0,11,12,13,0,0,16,0,18,19,20,21,0,23,24,25,26,27,28,29,30,31,32,33,0,35,0,2,3,4,0,0,7,8,
    10,11,12,0,14,0,16,17,0,0,20,0,0,23,0,25,26,27,28,0,30,31,32,33,34,0,1,0,3,0,5,0,7,8,9,
    11,0,13,0,15,0,17,0,19,0,0,22,0,24,0,26,27,0,29,30,31,32,33,34,35,1,2,3,0,5,0,7,8,9,10,
    12,13,14,0,16,17,18,0,20,21,22,0,0,25,26,27,28,29,30,31,32,33,34,35,1,2,3,0,5,6,7,0,9,0,0,
    13,14,15,16,17,0,19,20,21,0,0,0,25,26,0,28,29,30,31,0,0,34,0,1,2,3,0,0,6,0,8,9,10,11,0,
    14,0,16,17,18,19,20,21,22,23,0,0,26,27,0,29,0,0,32,0,34,35,1,2,3,4,5,6,7,8,9,10,11,0,0,
    0,16,17,0,19,0,21,22,23,0,0,26,27,28,29,30,31,32,0,34,35,1,0,3,4,5,0,7,8,9,10,0,12,0,14,
    16,17,18,19,20,21,22,0,0,25,0,27,0,29,0,31,0,0,34,0,0,0,0,4,0,0,7,8,9,10,11,12,0,14,15,
    0,18,19,20,0,22,0,24,0,0,27,0,29,30,31,32,33,34,0,0,0,0,0,5,0,7,8,9,10,11,0,0,14,15,16,
    18,19,20,21,22,0,24,25,26,0,28,0,0,31,32,33,0,35,1,0,0,0,0,0,7,8,9,10,11,12,13,14,15,16,0,
    19,0,0,22,23,0,25,26,27,28,29,30,0,32,33,34,35,1,2,3,4,0,6,0,8,0,10,11,12,0,0,0,0,0,18,
    0,0,22,23,24,0,0,27,28,29,0,31,32,0,34,35,0,0,0,4,0,0,7,8,9,0,11,0,0,14,15,16,0,0,19,
    21,22,23,0,25,0,27,28,29,0,0,0,33,0,35,1,2,0,4,0,6,7,0,9,0,0,0,0,0,0,0,0,0,19,0,
    0,0,24,0,0,27,28,29,30,0,32,0,0,35,1,2,3,4,0,6,7,0,9,10,11,12,13,14,0,16,17,18,0,0,21,
    0,24,0,26,27,0,29,30,0,32,33,34,0,1,2,3,4,5,0,0,0,9,0,11,12,13,14,15,0,0,0,0,20,0,22,
    24,25,0,0,28,0,30,0,32,0,34,35,1,2,3,4,0,6,7,0,9,0,11,0,13,0,0,16,0,18,19,0,21,22,23,
    25,0,27,28,0,30,0,32,33,0,35,0,0,3,0,5,0,0,0,9,10,11,12,0,14,15,16,17,18,19,20,0,22,23,24,
    26,27,28,29,0,0,0,33,0,35,1,2,3,0,0,0,0,8,0,0,11,12,0,0,15,16,17,18,19,20,21,22,23,24,25,
    27,0,0,30,0,32,33,34,35,1,2,3,4,5,6,7,0,9,0,11,12,0,14,0,16,17,18,0,0,21,22,23,24,25,26,
    0,29,0,0,32,0,0,35,1,2,0,4,5,6,0,0,9,0,11,12,0,14,15,16,17,18,0,0,0,0,23,0,25,26,27,
    29,30,31,32,33,0,0,1,2,3,0,5,6,7,8,0,10,0,0,13,0,15,0,17,18,0,20,21,22,0,24,25,26,0,0,
    30,31,0,33,34,35,0,0,3,0,0,6,7,0,9,10,0,12,13,0,0,16,0,0,19,0,21,22,23,0,25,0,27,28,29,
    0,32,33,0,35,1,2,3,4,5,0,7,8,9,10,0,0,13,14,15,16,17,18,0,0,21,22,23,24,0,26,0,0,29,30,
    0,33,34,35,1,0,3,0,5,6,7,8,9,0,11,12,13,14,15,16,17,0,19,20,21,22,0,0,25,26,27,28,0,0,0,
    33,0,0,1,0,0,4,5,6,7,0,9,10,11,12,0,14,15,16,17,18,19,20,21,22,23,24,25,26,27,0,0,30,31,32,
    0,35,0,2,0,0,5,6,7,8,9,10,11,12,0,0,15,0,0,18,19,20,21,0,23,24,25,0,27,28,0,0,31,32,0,
    35,1,2,3,4,5,6,0,0,0,10,11,0,13,14,15,16,0,18,19,20,21,22,23,24,0,0,27,28,29,30,31,32,33,34
  };

  const int d40_528[] = {
    // Size: 40 x 40
    40,
    // Pre-assigned fields
    1,2,3,4,5,0,7,8,9,10,0,0,0,0,15,0,0,0,19,0,21,22,0,0,0,26,27,28,0,0,31,32,0,0,35,0,0,0,39,0,
    2,3,0,5,6,7,0,9,0,11,12,13,14,15,16,17,18,19,20,21,22,23,24,0,26,0,0,29,30,0,32,33,34,0,0,0,38,39,0,1,
    3,4,0,6,7,8,9,0,0,0,0,0,0,16,0,0,0,0,21,22,23,24,25,26,27,28,29,30,0,0,0,34,35,0,37,38,39,40,1,2,
    0,5,6,0,8,9,10,0,12,0,14,0,16,17,18,19,0,0,0,0,24,25,0,27,28,29,0,31,32,33,0,35,36,0,38,0,0,1,2,0,
    5,6,7,0,9,0,11,12,13,14,15,16,17,0,19,0,21,0,23,24,25,26,27,28,0,30,0,32,33,0,35,0,37,38,0,40,1,0,0,4,
    6,7,8,9,0,11,12,13,14,15,0,17,18,0,20,21,22,0,24,25,0,27,0,29,30,31,0,33,34,35,0,37,38,39,0,1,2,0,0,5,
    7,8,0,10,0,0,0,14,15,16,17,18,19,20,21,0,23,24,25,0,27,28,29,30,31,32,33,34,35,36,37,38,39,0,1,2,3,4,5,6,
    0,9,10,0,12,13,14,15,16,17,0,19,20,0,0,23,24,25,26,27,28,29,0,31,32,33,0,0,36,37,0,39,40,1,0,0,0,5,0,0,
    9,10,11,12,13,14,0,0,17,18,19,0,21,22,23,24,25,0,0,28,29,0,0,0,33,34,0,0,37,38,39,0,1,2,0,0,5,0,7,8,
    10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,0,0,29,30,31,0,0,0,35,36,37,38,39,40,1,0,3,4,5,6,0,8,9,
    11,12,0,14,15,16,17,18,19,0,0,0,23,24,25,26,27,28,0,0,31,0,33,0,35,0,0,38,39,40,0,2,0,4,5,6,7,0,9,10,
    0,13,14,15,0,17,0,0,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,0,36,37,0,39,40,0,0,3,4,5,6,0,0,0,10,0,
    0,0,15,16,0,18,19,20,0,0,23,24,25,26,0,28,29,0,31,0,0,34,35,36,37,38,39,0,0,2,3,0,0,6,7,8,9,10,11,0,
    14,15,16,0,18,19,0,21,22,23,24,25,26,27,28,0,30,31,32,33,34,0,36,37,38,39,0,1,0,3,4,0,6,0,0,9,10,11,12,13,
    15,16,17,0,0,20,21,22,23,0,25,26,27,28,29,30,31,32,33,0,35,36,37,0,0,40,0,2,3,4,0,6,7,8,0,10,11,12,13,0,
    0,0,0,19,20,21,0,0,24,25,26,0,28,29,30,31,32,0,34,35,36,0,0,39,40,1,2,0,0,5,0,7,0,9,10,11,0,13,0,15,
    17,18,19,0,21,22,23,0,0,26,27,0,29,30,31,32,33,34,0,36,0,38,0,40,1,2,3,0,5,6,0,8,0,0,11,12,0,14,15,16,
    18,0,0,21,22,23,24,0,26,0,28,0,0,31,0,0,34,0,0,37,38,39,40,1,2,3,4,0,6,7,8,9,0,11,12,0,14,0,16,0,
    0,20,21,22,0,24,0,0,0,28,29,30,31,32,33,34,0,36,37,38,0,0,1,2,3,0,0,6,7,8,0,10,0,12,13,0,0,0,17,0,
    0,0,22,23,0,25,0,27,28,29,0,31,0,33,34,35,36,37,0,39,40,0,2,3,4,5,6,7,8,9,10,0,0,13,0,0,16,17,0,0,
    21,22,23,24,25,26,27,0,29,30,31,0,33,34,35,36,37,0,39,40,1,2,3,4,0,6,7,8,0,0,0,12,13,14,0,16,0,18,19,20,
    0,23,0,0,0,0,28,29,30,31,32,33,34,35,36,37,38,0,0,0,2,0,0,5,6,0,0,0,10,11,12,13,14,15,0,17,18,0,20,21,
    23,0,25,26,27,28,29,30,31,32,33,34,35,0,0,0,39,40,0,2,3,4,5,6,0,0,0,10,0,0,13,14,0,16,17,18,0,20,21,22,
    24,0,0,27,28,0,0,0,0,33,0,35,0,0,0,39,40,1,2,3,4,0,6,7,8,9,10,11,12,13,14,15,0,0,0,19,20,21,0,23,
    0,26,27,28,0,30,0,32,33,34,0,0,37,38,0,40,1,2,3,4,5,6,7,8,0,0,0,12,13,14,0,0,0,0,19,20,21,0,23,24,
    26,27,28,0,0,0,32,0,34,35,0,37,0,39,0,1,2,0,4,0,0,7,0,9,0,0,0,13,14,0,16,0,0,0,20,0,22,23,0,25,
    0,28,0,30,31,0,33,34,35,0,37,0,0,40,1,0,3,4,0,6,7,0,9,10,11,0,0,0,15,16,17,18,19,20,21,0,0,24,25,0,
    28,29,0,31,32,0,34,0,0,0,38,39,0,0,2,3,4,0,6,7,8,9,10,11,12,13,14,0,16,17,18,19,20,21,0,23,24,25,26,27,
    0,0,0,32,33,34,0,0,37,38,0,40,1,0,0,0,5,6,7,8,9,0,0,12,13,14,0,16,0,0,0,20,21,22,23,24,25,26,27,0,
    30,0,32,33,0,35,36,37,38,39,0,1,2,0,0,0,0,0,0,0,0,11,12,0,14,15,16,0,18,0,0,0,22,0,24,0,26,0,28,29,
    0,0,0,0,35,36,37,0,0,40,1,2,3,4,0,6,7,8,0,10,11,0,0,0,0,0,17,0,0,20,21,0,23,24,0,26,27,28,0,30,
    32,0,34,0,36,0,0,39,40,1,2,3,4,5,6,0,8,9,10,11,0,13,14,0,0,17,18,0,20,0,0,23,0,25,0,27,28,0,30,31,
    0,0,35,36,37,38,39,40,0,0,0,4,5,6,7,8,9,10,0,12,13,14,15,16,17,18,19,0,0,22,23,24,25,26,27,28,0,30,31,0,
    0,35,36,0,38,39,40,1,2,0,4,0,0,7,8,0,10,11,12,0,14,15,16,17,18,0,20,0,0,23,24,25,26,27,28,29,30,31,32,33,
    35,36,37,38,0,40,1,2,3,0,5,6,7,8,9,10,11,12,0,14,15,16,17,18,19,20,21,22,23,24,0,0,0,0,29,30,31,32,0,34,
    0,37,0,39,40,0,0,3,0,0,6,0,8,9,0,11,0,13,0,15,0,0,18,19,0,21,22,23,0,25,26,27,28,29,30,31,32,33,0,35,
    0,38,39,40,1,2,0,4,5,0,0,0,0,10,0,0,13,14,15,16,17,18,19,0,21,22,0,0,0,0,0,28,0,30,31,32,33,34,35,36,
    38,39,40,1,2,3,4,5,6,0,8,0,0,11,12,0,14,0,16,0,18,0,0,0,22,23,0,0,26,27,28,0,30,31,32,33,34,0,36,37,
    0,40,1,2,3,4,0,6,7,8,0,10,11,12,13,14,15,16,17,18,19,20,21,22,0,0,0,26,0,28,0,30,31,0,33,0,35,36,37,38,
    40,1,2,3,0,5,0,7,8,9,10,0,12,13,14,15,0,17,18,19,0,0,0,23,0,25,0,0,28,29,30,31,0,33,34,35,0,37,0,39
  };

  const int d40_544[] = {
    // Size: 40 x 40
    40,
    // Pre-assigned fields
    15,0,2,0,34,30,26,29,18,0,35,37,4,13,6,0,14,25,21,20,0,9,0,24,38,3,16,0,40,0,28,23,33,11,10,17,39,22,31,5,
    27,22,7,26,17,40,0,1,13,28,12,0,31,0,30,8,20,5,33,38,10,0,29,23,0,0,2,0,37,18,24,4,14,32,0,9,25,21,0,0,
    0,10,0,4,0,32,14,26,0,0,18,0,36,9,0,0,27,11,1,35,0,16,33,37,0,31,8,0,0,40,0,0,6,5,29,34,3,7,24,30,
    1,0,34,10,39,0,0,0,6,0,23,35,25,0,27,0,9,3,5,24,29,37,0,0,0,0,30,22,0,2,13,31,26,0,32,38,11,15,0,0,
    34,0,5,22,1,19,38,0,0,18,0,0,32,2,36,11,0,16,0,12,33,30,0,29,21,23,15,0,0,27,10,0,24,0,0,7,31,4,28,13,
    0,29,27,13,0,24,22,0,0,0,16,3,10,34,19,4,33,30,15,9,20,0,0,6,7,36,40,0,39,17,0,0,21,14,31,18,26,2,11,23,
    0,32,4,0,0,22,25,34,0,26,0,27,0,0,40,2,11,29,0,0,16,3,8,1,19,18,20,14,0,21,5,0,0,9,0,0,35,6,0,0,
    11,0,38,17,2,31,0,36,0,5,0,25,40,4,16,10,19,0,37,18,0,0,0,32,9,7,33,23,0,0,0,0,39,12,0,0,24,0,29,21,
    35,0,30,0,0,39,0,38,28,6,0,23,0,0,31,33,36,0,27,29,3,12,20,0,18,0,37,21,15,24,16,0,0,8,2,0,1,17,25,0,
    25,0,15,6,9,29,0,18,21,23,26,28,16,30,24,37,0,17,14,22,0,5,11,0,10,0,34,0,0,0,31,2,36,4,1,32,0,19,0,0,
    0,0,0,0,38,0,0,0,39,16,22,24,13,6,0,0,12,18,2,11,28,0,1,0,31,0,0,0,0,30,32,0,34,36,26,0,4,5,15,19,
    7,1,10,21,40,35,9,0,0,0,3,34,12,36,0,13,8,0,0,27,32,17,25,0,39,0,23,26,6,0,0,24,0,0,0,20,15,37,0,0,
    6,20,39,0,0,26,4,30,0,40,0,0,29,14,0,1,0,7,0,8,0,38,2,28,5,0,32,0,0,11,25,37,0,0,22,0,10,12,27,36,
    0,27,8,0,0,2,0,25,32,29,36,15,0,11,21,17,1,9,19,4,18,34,5,0,23,0,12,0,22,0,38,35,7,28,14,0,37,24,0,26,
    0,0,0,29,0,11,0,20,14,25,0,0,26,40,38,28,6,0,0,13,34,24,16,22,0,0,0,0,35,33,9,0,0,31,8,1,0,18,0,39,
    0,0,0,16,11,21,7,39,29,0,34,4,1,0,25,24,32,0,3,0,6,33,0,5,36,0,22,35,18,19,0,26,38,0,13,0,9,30,0,0,
    3,38,24,0,14,36,0,2,12,10,11,19,0,8,0,39,4,6,40,16,15,1,27,0,30,37,0,17,0,0,0,22,29,0,0,26,0,23,0,35,
    21,23,18,19,29,25,10,6,24,0,1,14,15,5,35,0,0,0,13,28,0,26,0,34,0,0,4,38,0,16,12,30,37,17,0,40,20,36,0,22,
    40,0,0,0,22,0,32,8,35,0,25,20,0,0,7,18,37,38,29,30,0,14,0,0,4,0,0,33,2,39,0,0,16,10,0,23,6,0,26,17,
    16,15,14,18,0,0,12,0,0,0,0,32,0,0,26,40,34,28,0,37,35,8,30,39,33,5,6,19,0,0,0,0,10,3,38,0,29,20,17,25,
    0,12,29,15,32,34,39,0,0,0,30,0,0,26,33,0,31,40,35,3,17,0,21,27,14,2,0,25,24,4,0,1,0,18,36,28,38,16,9,37,
    9,17,20,0,24,1,0,19,0,35,32,40,5,37,0,0,0,0,36,21,31,10,4,26,0,33,29,3,16,25,0,38,22,2,18,13,0,0,0,6,
    10,13,26,12,0,0,40,0,0,4,0,0,21,0,28,0,3,0,17,0,30,0,37,14,2,22,0,0,23,32,39,27,18,34,6,29,16,8,1,0,
    20,0,16,34,28,0,21,27,40,0,2,36,0,0,15,29,0,37,10,31,13,0,3,35,0,6,0,7,0,14,1,0,17,23,5,19,30,39,0,32,
    33,37,19,23,25,17,0,35,31,30,0,5,24,39,0,0,22,32,0,40,0,21,7,0,27,12,38,36,29,0,15,0,0,0,20,16,28,26,0,11,
    14,34,11,33,0,12,3,37,19,20,9,0,6,32,17,30,0,23,31,1,0,4,0,0,0,13,27,2,0,35,0,29,15,22,39,0,0,38,5,16,
    32,18,17,36,0,4,0,0,33,2,40,26,0,0,8,0,21,14,11,7,37,28,6,0,0,15,39,9,34,13,0,0,0,20,12,22,5,10,38,1,
    39,0,33,2,0,5,0,28,34,0,38,0,11,25,0,6,0,0,7,0,36,0,0,18,0,24,0,8,20,0,17,19,23,0,27,0,40,35,30,3,
    17,24,23,9,0,14,5,31,25,0,28,11,0,38,0,0,16,35,32,19,0,7,0,12,0,0,18,0,0,37,3,0,13,0,0,0,22,0,0,29,
    2,0,0,37,18,10,1,0,0,9,0,31,3,27,5,21,29,0,6,39,0,0,22,0,11,28,35,13,0,15,19,0,25,38,0,8,7,0,34,20,
    30,11,21,35,0,33,24,32,10,31,5,0,8,0,0,0,25,12,0,14,0,18,39,38,0,0,0,27,19,0,29,0,0,7,0,0,0,13,16,40,
    4,19,40,14,26,0,0,22,0,0,0,21,35,0,0,9,5,27,0,15,0,6,13,2,3,25,0,30,12,0,34,0,31,33,17,0,32,29,18,24,
    13,40,37,0,27,0,0,9,30,0,33,0,14,0,0,38,0,0,28,6,11,0,24,0,1,0,26,0,0,23,7,16,0,0,21,3,12,34,19,31,
    23,25,3,0,8,28,0,7,16,27,0,17,20,15,13,22,2,33,12,0,19,31,36,0,6,10,1,0,26,38,14,34,40,0,0,24,18,9,35,0,
    22,31,0,7,37,6,0,15,27,0,0,0,18,0,1,5,38,0,0,0,24,11,0,8,0,30,13,4,32,34,33,12,20,0,0,10,14,28,0,0,
    29,5,12,40,0,37,33,16,0,36,8,39,27,0,3,26,18,0,38,0,1,0,14,0,0,4,19,32,13,0,35,0,9,24,0,25,23,31,21,28,
    36,0,6,0,15,16,2,12,3,13,0,0,0,21,0,0,30,0,22,0,0,19,0,9,37,14,0,24,33,29,0,0,1,35,4,27,0,25,0,18,
    0,0,22,0,0,0,11,13,38,34,0,0,0,0,0,35,39,0,0,2,4,0,40,31,17,16,0,29,10,12,0,32,8,26,25,5,0,1,20,27,
    26,30,0,0,20,0,0,17,7,19,0,6,0,0,0,16,0,0,8,33,0,39,38,0,12,35,36,0,0,3,4,0,32,0,37,2,0,27,14,0,
    0,2,0,8,0,0,0,3,36,22,0,33,34,0,23,14,15,13,25,5,38,0,0,20,0,0,28,0,9,31,26,21,0,16,35,4,19,0,0,12
  };

  const int d40_560[] = {
    // Size: 40 x 40
    40,
    // Pre-assigned fields
    15,36,2,27,0,0,26,0,18,32,35,37,4,13,0,12,0,0,21,20,8,0,19,0,38,3,16,0,40,7,0,0,33,11,0,17,39,0,31,5,
    27,0,7,26,0,0,36,0,13,28,12,16,31,3,30,8,20,5,33,0,0,0,29,23,35,0,0,0,37,18,24,4,14,32,0,0,25,0,6,34,
    0,0,0,4,19,32,14,26,17,12,0,0,36,9,20,0,27,11,0,35,0,0,33,37,0,31,0,0,21,40,0,0,6,0,29,34,0,7,24,30,
    1,33,34,10,39,0,20,4,6,0,23,35,0,12,0,19,9,3,5,24,29,37,0,0,40,17,30,0,14,2,0,0,26,21,32,38,11,0,36,8,
    0,39,0,22,1,19,38,0,26,0,0,0,0,2,36,11,35,16,0,12,33,30,0,29,0,23,0,0,17,27,0,14,24,25,0,7,31,0,28,13,
    38,0,0,13,12,24,22,5,8,1,16,0,10,34,0,4,33,30,15,9,20,0,35,6,7,0,40,28,0,0,37,0,0,14,0,0,26,0,11,23,
    0,0,4,38,33,22,25,34,0,26,0,0,17,0,40,2,11,0,0,36,0,0,0,0,19,18,20,14,28,0,5,15,0,9,23,31,35,6,13,0,
    0,8,38,17,2,31,13,36,20,0,27,0,40,0,16,10,0,22,37,18,0,0,34,32,0,7,0,23,1,28,30,6,39,12,15,14,0,3,0,21,
    35,9,0,32,0,39,34,38,28,0,13,23,22,0,31,33,36,0,27,29,3,12,0,7,18,0,0,21,15,0,16,40,0,8,0,0,0,17,25,14,
    0,0,15,6,9,29,8,18,0,23,0,28,0,30,0,37,40,17,14,22,0,5,0,3,10,38,34,12,27,20,0,2,0,4,1,32,13,0,0,33,
    37,7,0,0,0,3,23,21,39,0,22,24,13,6,0,0,0,18,2,11,28,40,1,33,0,0,0,10,0,30,32,17,34,0,26,35,4,5,15,19,
    0,1,10,21,40,0,0,0,0,33,3,0,12,0,0,13,0,0,16,27,32,17,25,0,39,19,23,26,0,5,18,24,28,0,30,20,0,0,22,38,
    6,0,39,0,0,0,4,0,9,0,15,0,0,0,0,1,0,7,0,0,23,38,0,28,0,0,32,0,0,0,0,37,0,0,22,0,10,12,0,0,
    0,27,8,3,13,2,6,0,0,29,36,15,30,0,21,17,1,0,19,4,0,34,5,40,23,20,12,16,22,10,38,0,7,28,14,39,37,0,33,26,
    0,21,0,29,7,11,0,20,14,0,0,0,26,0,0,28,6,4,23,13,0,0,16,0,0,32,3,37,35,33,9,0,0,31,8,1,0,0,0,39,
    8,14,0,16,11,21,0,39,0,37,34,4,0,31,0,0,32,0,3,17,6,33,0,5,0,0,0,0,0,19,2,26,38,0,13,0,9,30,10,0,
    0,38,0,0,14,36,0,0,12,0,11,19,7,8,18,39,0,6,0,16,0,0,27,25,0,0,21,17,5,0,20,22,0,0,0,26,33,23,32,35,
    21,23,18,19,29,25,10,6,0,0,1,14,15,5,0,0,7,2,0,0,9,26,32,34,0,0,4,0,11,16,12,30,0,17,33,0,20,36,3,22,
    40,3,31,0,22,0,0,8,35,15,25,0,0,0,0,0,0,38,0,30,0,14,12,19,4,34,0,33,0,0,36,13,16,0,24,23,6,0,0,17,
    16,0,0,18,21,0,12,0,1,24,0,0,2,23,26,0,0,28,9,37,35,0,30,39,33,5,0,19,31,22,27,7,10,3,38,0,29,0,0,25,
    0,0,29,15,32,34,39,10,11,8,0,22,23,26,0,20,31,0,35,3,0,0,0,27,14,2,7,0,0,4,0,0,0,0,36,0,38,16,0,37,
    9,0,0,30,24,0,27,19,0,0,0,0,5,0,12,0,0,0,36,21,31,10,0,26,28,0,29,3,16,25,11,0,22,0,0,13,34,14,39,6,
    10,13,26,12,35,0,40,33,5,4,19,0,0,0,28,36,0,15,17,0,0,20,0,14,2,22,31,11,0,32,0,27,18,0,6,0,16,8,0,0,
    20,4,0,34,28,9,21,27,40,11,2,36,0,0,15,0,26,0,10,0,13,0,3,35,0,0,24,0,38,14,0,0,0,23,5,19,30,0,0,32,
    33,37,19,23,25,0,18,35,0,30,6,5,0,0,0,0,0,0,34,40,0,21,0,13,27,12,38,36,29,0,15,9,0,1,20,16,0,26,4,0,
    14,34,0,33,0,12,3,0,19,0,9,18,6,0,0,0,28,23,31,1,25,0,26,36,24,13,0,2,0,0,0,0,0,22,0,21,0,38,5,16,
    32,18,0,0,31,0,29,0,33,0,40,26,19,35,8,25,0,14,11,7,0,0,6,30,16,0,39,0,34,0,0,0,0,20,12,0,5,10,0,1,
    39,16,33,0,4,5,31,28,34,0,0,12,11,25,29,6,0,21,7,26,0,0,10,18,32,0,9,8,20,1,17,0,23,37,27,0,0,35,30,3,
    0,24,23,0,36,14,5,0,0,21,0,0,0,38,2,34,0,35,32,19,27,7,15,12,26,0,18,20,4,37,3,10,0,6,40,30,22,33,0,0,
    0,26,0,37,18,10,1,23,4,0,0,31,3,0,5,0,0,24,6,39,0,36,22,17,11,28,35,13,0,15,0,33,0,38,16,0,7,40,0,20,
    0,11,0,35,23,0,0,0,0,0,5,0,8,20,0,15,0,12,4,14,22,0,39,38,34,9,0,27,0,26,29,36,3,7,28,6,2,13,0,40,
    4,19,40,0,0,0,16,22,23,0,20,21,35,0,11,0,5,0,39,15,0,6,13,2,0,25,0,0,12,36,0,0,31,0,17,37,0,0,18,24,
    13,0,37,25,27,0,0,9,30,17,33,2,14,22,32,38,10,0,28,6,11,29,24,4,1,8,26,18,0,23,0,16,0,15,21,3,12,0,19,31,
    23,25,3,39,8,28,37,7,16,0,0,17,0,15,13,22,2,33,12,32,0,0,36,21,6,10,0,0,26,38,0,34,0,30,11,24,18,9,0,4,
    0,0,0,7,37,6,0,15,27,3,21,29,18,16,1,0,38,36,26,23,0,0,17,8,25,0,13,4,0,0,33,0,0,39,9,0,14,28,40,2,
    29,5,12,0,30,37,33,16,22,36,8,39,27,17,3,0,0,34,0,10,1,2,14,0,20,4,0,0,13,0,0,11,0,24,0,25,0,0,21,28,
    36,28,0,0,15,0,2,12,3,13,0,10,0,21,39,0,30,0,0,0,40,19,0,0,0,14,11,0,0,29,8,20,0,35,0,27,0,25,23,0,
    0,0,0,28,0,0,0,0,38,0,24,7,37,33,0,35,39,19,30,0,0,23,0,31,0,16,14,29,10,0,0,32,8,26,0,0,36,1,0,0,
    0,30,13,11,20,23,15,17,7,19,31,6,28,29,0,16,24,1,8,0,5,0,38,10,0,35,0,34,25,3,0,18,0,40,37,0,0,0,0,0,
    24,2,1,8,6,7,0,3,0,22,17,33,34,0,0,0,15,13,25,0,0,27,0,20,0,39,28,40,9,0,0,21,11,16,0,4,19,32,0,12
  };

  const int d50_750_bal[] = {
    // Size: 50 x 50
    50,
    // Pre-assigned fields
    43,28,49,45,17,31,9,0,19,26,0,29,18,41,15,14,32,0,0,0,3,13,50,8,0,10,2,1,0,25,27,11,0,39,47,0,20,0,36,12,0,5,0,44,37,0,0,40,4,0,
    26,18,14,0,43,24,6,17,9,38,0,2,0,12,0,46,47,0,44,15,0,0,0,0,42,8,48,41,30,32,4,25,37,33,19,0,0,34,11,35,40,0,10,23,0,39,50,0,0,5,
    5,38,10,32,18,23,21,31,41,16,17,0,36,0,50,0,33,37,0,40,48,25,2,0,0,49,0,0,3,7,14,29,30,44,0,46,43,0,0,0,24,0,15,0,0,22,1,8,27,12,
    18,0,2,36,26,48,0,27,22,0,0,20,41,0,30,25,11,7,9,28,0,39,21,37,0,15,24,40,0,3,35,0,14,10,0,0,4,0,8,34,0,23,43,29,5,32,0,0,31,6,
    11,36,0,14,37,0,7,10,0,30,38,0,4,1,0,18,22,6,32,2,20,50,0,29,39,31,3,12,0,0,42,0,0,34,26,41,25,8,19,44,13,0,35,0,0,46,5,0,49,0,
    40,0,33,29,16,0,0,1,0,36,0,0,0,0,31,0,0,0,5,27,0,6,8,49,2,0,4,34,35,20,24,32,46,47,25,22,38,0,50,23,19,13,30,45,15,0,14,11,7,3,
    0,0,25,0,30,27,50,45,0,0,31,42,13,4,10,35,0,40,28,19,15,11,49,36,0,0,17,8,16,34,21,6,7,0,0,0,47,0,0,29,0,0,5,3,38,44,37,9,14,32,
    45,0,0,15,6,44,0,37,43,50,13,8,0,33,19,36,7,23,20,5,30,38,1,40,29,0,0,9,32,0,0,0,0,46,0,26,17,0,35,4,21,0,48,16,0,0,3,18,42,47,
    0,30,11,16,14,33,42,15,0,10,44,0,17,7,0,49,0,4,0,22,37,26,0,45,34,47,5,50,13,24,0,48,8,23,20,27,9,0,0,0,0,25,0,28,39,6,19,0,0,2,
    0,12,24,19,42,6,49,0,0,31,43,27,0,37,45,17,0,0,3,8,21,36,34,15,0,44,0,10,5,41,33,2,50,22,0,48,29,14,28,0,0,9,16,0,0,0,0,46,18,1,
    19,48,4,13,0,1,40,11,49,23,0,0,35,34,20,0,9,17,26,43,0,44,25,18,46,0,21,0,0,5,0,12,16,29,28,0,15,47,0,27,0,24,0,37,0,10,8,42,0,14,
    33,5,39,50,24,12,20,6,10,0,30,18,0,16,26,0,27,0,0,4,0,40,41,31,36,48,25,0,0,21,0,7,0,14,0,2,44,32,37,45,0,3,29,38,1,0,0,22,34,0,
    7,3,0,47,0,42,0,21,37,0,15,24,34,31,36,1,0,33,0,11,10,12,0,32,25,6,0,49,48,14,8,0,0,27,17,43,0,0,23,16,26,0,0,46,40,41,18,38,0,50,
    34,7,19,33,0,0,30,49,47,11,16,36,9,14,4,41,0,0,0,26,6,0,31,10,40,18,45,32,0,37,0,23,0,25,50,0,13,39,44,0,0,0,0,2,27,0,12,5,46,17,
    3,32,36,2,23,0,16,0,21,22,35,33,7,0,0,0,37,0,34,31,50,0,0,12,0,1,0,44,4,0,17,0,49,45,0,0,10,24,9,6,5,30,28,0,43,19,13,20,40,26,
    46,39,0,0,12,0,8,0,0,49,20,1,24,0,41,16,40,0,0,0,26,0,47,44,11,2,36,28,23,0,45,0,21,19,35,3,22,6,32,18,9,50,0,0,7,17,0,48,15,25,
    48,9,46,34,49,0,33,28,44,42,19,0,32,0,27,10,0,0,0,3,0,16,11,23,0,13,29,14,20,15,7,17,25,0,45,36,40,18,31,38,1,22,0,0,47,0,24,0,0,0,
    38,37,1,27,20,0,0,35,12,0,4,13,6,8,0,0,23,9,0,0,44,17,14,0,0,33,0,43,46,16,0,45,36,21,0,32,34,48,0,22,7,18,0,26,29,5,41,24,0,10,
    37,47,0,30,4,25,0,0,36,3,46,0,40,20,0,6,34,49,15,10,0,31,28,35,5,7,16,39,1,0,9,0,44,0,23,0,33,29,41,0,0,0,12,11,24,0,42,45,26,0,
    36,0,32,31,0,15,12,8,0,0,0,23,39,42,25,0,44,35,11,0,0,0,0,30,45,41,49,0,27,40,0,38,6,16,9,13,21,2,34,0,37,33,14,18,26,3,0,19,0,43,
    32,0,22,48,0,0,0,14,0,39,24,31,0,40,35,13,0,36,27,44,23,34,37,47,38,0,7,0,42,49,19,18,0,0,16,29,0,26,5,0,3,4,8,9,17,43,0,12,6,0,
    25,19,15,0,28,0,0,39,20,5,0,0,21,0,0,50,42,45,7,35,16,32,13,17,33,0,0,4,22,46,18,36,40,24,49,47,0,44,0,2,0,38,11,30,0,12,0,0,9,34,
    0,33,30,39,32,11,0,23,28,8,21,49,0,38,0,0,31,15,18,0,25,37,19,24,44,22,34,42,6,0,36,43,12,17,0,9,0,0,3,7,4,0,0,0,0,0,29,35,48,45,
    0,40,0,9,45,29,28,48,4,0,5,0,0,3,0,15,0,0,19,39,17,0,23,22,14,0,0,47,25,42,41,35,2,1,18,6,32,36,0,8,0,46,37,24,16,30,11,0,0,20,
    12,0,47,7,0,18,25,36,0,0,40,38,5,49,1,0,35,27,50,33,13,41,0,0,37,0,42,11,39,28,29,16,0,0,0,10,23,0,0,0,2,14,9,21,8,31,0,17,20,46,
    0,14,0,37,7,0,4,29,24,0,26,0,0,36,0,43,0,25,42,0,8,35,3,6,15,11,13,33,0,0,16,44,39,28,0,21,50,38,27,0,0,40,22,31,32,45,49,30,23,0,
    44,8,0,0,15,38,0,30,32,9,6,0,0,47,34,0,45,20,21,12,0,0,10,39,31,16,23,18,24,4,0,0,17,35,41,0,0,0,42,14,28,36,46,49,33,0,27,0,25,22,
    9,50,28,23,25,14,43,0,0,34,48,21,49,0,0,33,46,0,0,17,0,0,15,0,0,27,47,19,29,0,0,41,35,0,38,44,16,30,45,42,36,20,7,0,0,1,32,31,37,40,
    0,46,38,25,29,22,36,0,40,13,0,10,0,0,44,7,39,0,24,14,42,2,0,0,0,0,35,20,37,17,30,5,23,0,4,12,45,41,1,43,48,11,18,6,0,0,9,0,33,0,
    50,29,0,0,0,13,35,16,8,25,12,32,44,46,49,0,0,14,4,0,28,7,5,2,20,36,0,38,17,43,39,10,0,3,21,19,30,40,0,0,15,0,0,0,45,11,23,0,0,37,
    39,6,0,3,0,0,47,9,45,18,0,34,0,0,13,37,50,16,23,7,0,5,0,42,48,17,31,0,12,0,22,4,0,36,15,25,41,11,0,21,14,19,49,20,10,29,0,26,0,0,
    27,0,42,21,3,34,0,0,33,29,32,46,30,0,0,0,36,0,10,0,0,1,4,7,16,23,37,17,26,13,12,0,15,2,31,14,0,43,0,24,44,49,6,19,0,9,20,0,8,0,
    0,0,20,42,2,0,0,0,29,27,50,0,31,21,0,23,8,18,30,32,33,46,40,0,0,12,43,0,0,0,25,9,0,15,34,17,0,37,14,10,41,47,26,0,13,24,28,44,5,39,
    0,0,45,0,5,16,34,13,46,0,18,41,38,26,0,9,0,28,0,0,47,0,20,0,49,30,33,0,0,8,32,27,31,42,10,1,37,3,22,36,11,21,19,25,44,35,17,0,0,0,
    17,34,0,0,8,20,32,0,0,0,2,22,1,23,6,21,18,0,45,36,29,0,39,46,10,0,0,35,0,27,0,24,0,41,5,33,0,15,38,30,31,37,42,12,19,13,0,47,28,0,
    23,10,5,0,0,46,44,0,27,40,0,16,15,0,17,30,0,34,41,0,11,21,35,0,9,38,22,37,19,0,0,0,0,31,7,45,49,0,18,32,47,2,39,36,0,26,43,14,3,0,
    0,0,0,38,0,0,29,32,42,7,0,19,14,15,47,0,30,41,0,1,0,24,48,0,22,45,46,0,21,44,0,28,11,0,8,4,3,12,40,39,20,0,2,13,6,18,31,37,0,16,
    42,27,0,28,36,32,13,7,18,4,45,39,2,5,43,48,24,50,0,0,49,15,0,0,0,0,30,0,11,0,44,0,22,0,0,0,0,25,20,9,34,41,47,33,14,23,40,10,0,35,
    4,0,8,0,0,2,0,0,6,32,0,15,43,9,40,28,19,39,25,23,34,18,0,16,0,0,38,48,33,30,3,22,41,0,29,5,0,27,0,13,0,35,17,0,12,50,0,1,47,36,
    24,22,0,0,1,0,26,0,7,0,0,4,37,0,18,0,15,44,14,21,36,0,16,13,0,28,50,23,10,47,34,3,20,48,43,35,0,46,6,0,38,0,40,0,11,33,39,0,45,31,
    15,35,26,22,44,50,37,18,0,12,36,48,0,27,3,8,0,32,43,0,24,0,0,14,17,0,19,0,0,45,0,0,38,20,39,11,31,10,49,5,0,6,0,40,9,0,0,41,1,7,
    0,4,0,0,0,35,46,26,0,1,0,14,33,28,24,44,49,42,40,30,41,0,36,0,32,5,0,45,15,9,38,19,13,0,0,0,6,16,0,25,12,31,23,0,20,8,0,43,39,21,
    0,45,17,0,40,26,23,42,3,24,11,25,22,10,14,5,0,29,46,0,38,49,27,0,43,9,32,36,0,0,50,20,47,0,0,0,8,21,0,31,39,15,0,4,0,0,44,7,19,0,
    0,0,37,46,13,0,18,34,5,14,39,17,0,0,28,38,3,11,35,9,22,0,42,50,41,43,0,0,7,1,0,15,0,0,33,30,36,0,2,40,16,0,0,8,31,4,0,21,29,19,
    0,0,31,0,0,41,5,2,0,21,1,37,27,32,11,19,10,48,6,29,39,33,45,20,13,0,0,0,8,38,23,0,34,40,14,0,24,4,12,0,50,0,3,0,42,0,46,0,44,18,
    41,0,9,20,0,7,38,4,13,0,14,0,46,0,39,0,16,19,36,0,43,30,12,5,24,3,1,0,34,31,10,0,29,26,27,0,0,28,17,0,0,8,0,0,35,42,47,32,50,15,
    0,43,35,0,19,36,27,0,23,0,34,0,16,22,46,24,26,38,0,18,14,0,0,25,47,32,41,3,0,0,37,31,0,0,48,42,0,0,15,33,17,1,21,50,49,20,10,13,0,28,
    28,20,6,8,21,5,0,0,0,0,10,0,29,11,33,26,2,13,1,50,0,23,0,0,0,4,0,31,14,39,0,37,45,0,0,24,46,0,30,17,27,0,32,35,18,49,15,36,16,42,
    10,13,23,18,0,30,0,20,50,17,0,5,3,45,37,39,12,0,0,42,0,19,44,0,7,34,0,29,0,6,31,14,33,4,46,0,2,9,24,0,8,32,38,0,22,16,36,0,0,0,
    1,15,3,41,48,4,24,22,14,37,9,35,0,43,0,32,17,5,33,0,19,42,0,27,6,0,8,0,0,0,49,0,18,38,36,34,0,20,0,0,25,44,0,10,0,0,16,23,21,13
  };

  const int d50_825_bal[] = {
    // Size: 50 x 50
    50,
    // Pre-assigned fields
    0,2,3,4,0,0,7,0,9,10,11,0,13,14,0,16,17,18,19,20,0,22,23,24,25,0,0,0,29,30,0,32,33,0,35,36,0,0,0,40,41,42,43,44,0,0,47,48,49,50,
    0,3,4,5,6,0,8,9,10,11,0,13,14,0,16,0,0,19,20,0,22,23,24,0,0,27,28,29,30,0,32,0,34,35,36,0,38,0,40,41,0,0,0,45,46,47,48,49,50,1,
    3,0,0,6,7,8,9,0,11,12,13,14,15,16,0,0,0,20,21,0,23,0,25,26,27,0,29,0,31,32,0,34,0,36,37,0,39,0,41,42,0,44,45,0,47,48,49,50,1,0,
    4,5,6,7,8,9,10,0,12,0,14,0,16,0,18,19,20,0,22,23,0,0,26,0,0,29,30,31,0,33,0,0,36,37,38,39,0,41,42,43,44,45,46,0,0,0,0,1,2,3,
    0,6,7,8,9,10,0,12,0,14,15,16,17,18,19,20,0,0,23,24,25,0,27,28,29,30,31,32,33,0,0,36,37,0,0,40,0,0,43,44,45,46,47,0,49,0,0,0,3,4,
    0,7,8,9,10,0,12,13,0,15,16,17,0,0,0,21,22,23,24,0,26,0,28,0,30,31,32,33,34,35,36,37,0,0,0,41,0,43,44,45,46,47,48,0,50,0,2,3,0,5,
    7,0,9,10,0,12,13,14,15,16,0,0,19,20,0,0,23,24,25,26,27,0,0,30,0,32,33,34,35,0,37,38,39,40,41,0,0,44,45,0,47,0,0,0,1,2,0,4,5,6,
    8,9,10,0,12,0,0,15,16,17,0,0,0,0,22,23,24,25,26,0,0,29,30,31,32,0,34,35,36,37,0,39,40,0,42,0,44,45,46,47,48,0,50,0,2,3,4,5,6,0,
    9,0,0,0,0,14,15,0,17,18,19,20,0,0,23,24,25,0,27,0,29,30,31,32,33,34,35,0,37,0,0,40,41,0,43,44,0,46,47,48,49,50,1,0,0,4,0,6,7,8,
    10,0,12,13,14,0,16,0,0,19,20,21,22,23,24,25,26,27,0,0,30,31,32,0,0,0,0,0,38,39,0,41,42,43,44,45,46,0,48,0,50,0,2,3,4,5,6,7,0,9,
    0,12,0,0,15,16,17,18,0,0,0,22,23,24,25,26,27,0,0,30,31,32,33,34,0,36,0,38,39,40,0,42,43,44,45,46,0,48,49,0,1,2,0,4,0,6,7,8,0,10,
    12,13,14,15,16,17,0,19,0,21,22,0,24,0,26,27,28,0,0,0,0,33,0,35,36,37,38,39,40,41,42,43,44,0,0,47,48,0,0,1,0,3,4,5,6,7,0,0,0,11,
    0,14,15,16,0,0,0,20,0,22,23,24,0,26,0,0,29,30,31,0,33,0,35,36,0,38,0,0,0,42,43,44,45,46,47,48,49,50,1,2,3,4,0,6,7,0,9,10,0,12,
    0,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,0,34,35,36,0,0,0,40,41,42,0,44,0,0,0,48,0,0,1,0,3,4,0,0,7,8,0,10,11,12,0,
    15,16,0,18,0,20,0,22,23,24,0,0,27,28,29,0,31,32,0,34,35,0,0,38,39,40,0,42,0,0,45,46,47,0,49,0,0,2,3,4,5,6,7,8,9,0,0,12,13,14,
    0,0,18,19,20,21,0,23,24,25,26,0,0,29,30,31,32,0,0,35,0,37,0,0,40,41,0,43,44,45,46,0,48,49,50,1,0,3,0,5,0,7,8,0,0,11,12,13,14,15,
    17,18,19,20,0,22,23,24,25,26,0,28,29,30,31,0,0,34,35,0,0,0,39,40,41,42,0,0,0,0,47,48,49,50,1,2,3,4,0,0,7,8,9,0,0,0,13,14,0,16,
    18,19,20,0,22,0,0,25,0,0,28,29,30,31,32,33,34,35,0,37,38,39,40,41,42,43,44,0,0,47,48,49,50,1,0,0,4,5,0,7,0,9,0,0,0,13,0,0,16,17,
    0,20,21,22,23,24,25,26,0,0,29,30,31,0,0,34,0,0,0,38,39,40,41,42,0,44,0,46,0,48,49,50,1,0,0,4,5,6,0,8,9,10,11,12,13,0,15,16,0,18,
    20,21,22,23,0,0,0,27,0,29,0,0,32,33,0,0,36,0,38,39,40,41,42,43,0,45,46,47,48,49,0,1,2,3,0,5,6,7,8,9,10,11,12,0,14,15,0,17,0,0,
    21,22,23,0,25,26,0,0,29,30,31,0,33,0,35,36,0,38,39,40,41,42,43,44,45,0,0,48,0,0,1,0,0,4,5,6,7,8,9,0,11,0,13,0,0,16,17,18,19,20,
    22,23,0,25,26,27,0,0,30,31,32,33,0,35,36,0,0,0,40,41,42,43,0,45,0,47,48,49,50,1,0,0,0,0,6,0,8,9,10,11,0,13,14,15,0,17,18,0,20,21,
    23,0,25,0,27,28,29,0,31,0,33,34,0,36,37,0,0,40,41,42,0,0,0,0,47,48,49,0,0,2,3,4,5,6,7,8,9,10,11,12,0,14,15,0,17,18,0,0,21,22,
    24,25,26,27,28,29,30,31,0,33,34,35,0,0,0,39,0,41,42,43,0,45,46,47,0,49,50,0,2,3,4,5,0,0,8,0,10,11,0,13,0,15,16,17,0,0,20,0,22,0,
    25,26,27,28,0,30,31,32,33,0,35,36,37,38,39,0,0,42,43,44,0,46,0,0,49,50,1,2,3,4,0,0,0,8,0,0,0,12,13,14,15,16,17,18,19,20,21,0,0,0,
    0,27,0,29,0,0,32,33,34,0,36,37,0,39,40,0,0,43,44,45,46,47,48,49,0,1,2,3,4,5,6,7,0,9,10,11,12,0,0,0,0,17,18,0,20,21,22,23,24,0,
    27,0,29,0,31,0,33,0,35,0,37,0,39,0,41,42,43,44,45,46,0,48,0,50,1,2,3,4,0,6,7,8,0,10,0,12,13,14,0,0,17,0,19,20,0,22,23,24,25,26,
    28,29,30,31,0,33,34,35,36,37,0,39,40,41,0,0,44,0,46,47,48,0,50,0,2,3,4,5,0,7,8,0,10,11,12,0,14,0,16,17,0,19,20,0,0,0,24,25,0,0,
    29,30,0,32,0,34,35,0,37,0,39,40,41,42,43,0,45,0,0,0,0,0,0,2,3,4,5,6,7,8,9,10,0,12,13,14,0,0,17,18,19,20,21,0,23,24,25,0,27,28,
    30,0,32,33,34,0,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,1,0,3,4,0,0,0,8,0,0,11,12,13,0,15,16,0,18,0,20,21,22,23,24,0,0,0,0,0,
    31,32,33,0,35,36,0,38,39,40,0,42,0,0,45,46,47,48,49,0,0,2,3,4,0,6,7,0,0,10,11,12,13,14,15,16,17,0,0,0,21,22,0,24,25,0,27,28,29,0,
    32,33,34,35,0,0,38,39,40,0,0,0,44,45,0,47,48,49,0,0,0,0,0,5,6,0,8,9,10,0,12,13,14,15,16,17,18,19,0,21,22,23,24,25,26,0,28,29,30,0,
    33,34,35,0,37,38,39,40,41,42,43,0,45,46,47,0,0,50,0,2,3,4,5,0,7,0,9,0,11,12,13,14,15,0,17,0,0,0,21,22,23,0,0,26,0,28,29,0,31,32,
    34,35,36,37,0,0,40,0,42,43,0,45,46,47,48,49,0,0,0,3,4,0,0,7,8,9,0,0,12,0,14,0,16,17,18,19,20,0,0,23,24,0,26,27,28,0,30,31,32,33,
    35,0,37,38,39,0,0,0,43,0,0,46,47,48,0,50,1,2,3,4,0,0,7,8,9,10,11,12,13,14,15,0,17,0,19,0,0,0,23,0,25,0,27,28,29,30,31,32,33,34,
    0,37,0,0,0,41,42,43,0,45,0,47,48,49,50,1,2,0,4,5,6,7,8,9,10,0,12,13,0,0,16,0,18,19,20,21,22,0,24,25,26,0,0,29,30,31,0,0,34,35,
    0,0,39,40,41,0,43,44,0,46,47,0,0,50,1,2,3,4,0,0,7,8,0,0,11,0,0,14,15,16,17,18,0,20,21,22,23,24,0,26,0,0,29,30,31,32,0,34,35,36,
    38,0,0,41,0,43,44,45,46,0,48,49,50,1,0,3,4,0,0,7,8,9,0,11,12,13,14,0,16,17,0,19,20,21,22,23,0,25,0,27,28,0,0,31,0,33,0,35,36,0,
    0,0,41,42,0,44,0,46,0,0,49,50,0,0,0,4,5,6,0,0,9,10,11,12,13,0,15,0,17,18,19,20,0,0,23,24,25,26,27,28,0,30,31,32,33,34,35,0,37,38,
    40,0,42,0,44,45,0,47,0,49,50,1,2,3,4,5,0,7,8,9,10,11,0,0,0,15,0,17,0,19,20,0,22,23,0,25,26,27,28,0,0,0,32,33,34,35,0,37,38,39,
    41,42,0,0,45,46,47,48,0,0,1,2,3,0,5,6,0,8,9,10,11,12,0,0,15,0,17,18,19,20,0,0,0,24,25,0,27,28,29,0,31,32,0,34,35,36,0,38,0,40,
    0,43,0,0,46,0,48,49,50,1,0,3,4,5,6,7,8,9,0,11,12,13,14,15,0,0,18,0,0,21,22,23,0,25,26,0,28,0,30,0,32,33,0,35,36,37,38,39,0,41,
    0,44,45,46,47,48,49,0,1,0,3,4,0,6,0,0,0,0,11,12,13,0,15,16,17,18,0,20,21,0,23,0,25,0,0,28,29,30,31,32,33,34,35,0,37,38,39,40,41,0,
    44,45,0,0,48,49,0,0,0,0,4,5,0,7,0,9,10,11,12,0,14,15,16,0,0,0,20,21,0,23,24,25,26,27,28,29,30,31,32,33,34,35,0,37,38,39,40,0,0,0,
    45,46,0,0,49,50,1,0,3,4,5,6,0,0,9,10,11,0,13,14,0,16,17,18,19,20,0,22,23,24,0,26,0,0,0,0,31,32,33,0,0,36,0,38,39,40,41,42,43,44,
    46,0,48,0,0,0,0,3,4,5,6,7,0,0,10,11,12,13,14,15,16,17,18,0,20,21,22,23,0,25,26,0,28,29,0,31,0,33,34,0,0,0,38,39,0,41,42,43,0,45,
    47,48,0,50,1,2,0,4,5,6,0,0,9,10,11,0,13,14,0,16,0,18,19,20,21,22,23,24,25,0,27,28,29,30,0,32,33,34,35,36,0,38,0,40,0,0,0,0,0,0,
    48,0,0,1,2,3,4,5,6,0,0,0,0,11,0,13,14,15,16,17,18,0,20,21,22,23,24,0,26,27,0,0,0,0,32,33,34,35,36,37,38,39,0,41,42,0,44,0,46,47,
    49,50,1,2,3,4,5,0,7,8,0,10,11,0,13,14,15,0,0,0,19,0,0,22,23,0,25,26,27,0,29,30,31,32,0,34,0,36,0,0,0,0,41,42,43,44,45,46,47,0,
    0,0,0,3,4,5,6,7,0,9,10,0,12,13,14,15,0,17,18,19,0,21,22,0,0,0,0,27,28,0,0,31,32,33,0,35,36,37,38,0,40,41,42,43,44,45,0,0,48,49
  };

  const int d60_1080_bal[] = {
    // Size: 60 x 60
    60,
    // Pre-assigned fields
    8,0,44,0,0,25,0,26,48,6,57,0,0,7,29,40,0,0,31,36,50,21,32,52,38,49,13,56,37,51,0,0,45,59,1,27,15,9,0,0,47,0,16,33,18,0,39,46,0,0,42,17,54,20,11,0,0,23,53,60,
    34,31,17,0,39,0,12,0,3,52,25,55,0,0,16,43,0,0,57,0,30,0,48,13,15,7,49,10,45,60,0,22,11,51,5,29,6,53,19,0,36,0,0,0,58,1,35,33,37,0,47,41,0,0,44,28,0,42,56,24,
    0,45,18,44,58,11,53,41,0,35,26,23,0,0,0,49,17,57,1,0,7,19,31,43,21,0,9,54,38,36,4,51,0,39,29,22,60,25,24,0,50,0,0,42,47,0,32,14,0,0,12,0,0,30,13,55,34,20,0,0,
    1,0,33,46,20,17,41,49,34,26,35,21,3,13,44,0,0,47,0,0,40,36,38,0,31,27,0,42,30,32,48,2,4,43,8,0,0,24,0,0,58,0,9,7,29,50,51,18,45,0,0,0,39,0,6,0,5,60,57,0,
    35,4,10,59,0,57,56,9,60,34,0,0,5,25,38,46,33,0,21,29,53,37,11,42,36,32,0,58,0,0,0,50,43,0,3,13,0,6,0,28,40,18,0,1,19,8,23,16,14,26,0,39,0,0,7,0,49,0,52,0,
    45,10,49,0,16,0,14,48,0,0,37,17,57,52,0,3,51,0,0,58,4,0,0,7,8,0,33,29,42,5,28,54,23,21,20,0,0,0,43,0,30,0,18,15,0,46,0,0,25,22,32,60,13,40,2,39,44,9,35,34,
    27,21,16,0,23,37,0,0,35,45,0,30,51,9,50,20,36,54,0,34,0,25,17,0,22,0,58,13,59,0,0,0,52,4,0,0,0,15,46,0,31,49,44,0,10,33,29,47,0,53,7,18,55,0,5,41,39,48,28,56,
    32,0,1,41,18,0,49,23,7,13,15,22,43,0,0,12,0,16,30,45,21,44,8,50,19,0,0,26,0,2,59,11,6,0,47,35,28,55,5,25,34,29,0,46,0,0,0,42,27,3,0,0,0,17,58,0,4,0,39,40,
    0,60,9,3,29,53,0,33,50,54,22,52,37,0,0,6,0,40,55,23,8,26,58,36,56,1,0,14,57,10,30,47,28,20,0,0,11,48,31,4,59,19,0,0,0,42,0,0,44,0,0,46,41,39,25,0,0,15,0,17,
    60,56,37,0,55,5,10,59,40,8,48,12,21,0,43,22,15,25,45,42,0,7,0,0,0,0,57,0,27,0,23,44,31,52,4,49,9,51,41,0,0,1,24,0,26,16,0,0,58,30,20,11,0,0,14,19,50,54,0,0,
    31,55,0,0,47,29,15,1,0,57,0,0,0,35,10,8,3,33,14,0,32,0,20,30,40,59,0,16,0,9,11,52,7,54,0,0,12,49,0,27,4,5,0,22,0,58,19,37,34,44,36,0,56,51,28,50,0,0,2,23,
    12,58,36,50,0,13,0,35,51,0,39,4,0,27,0,0,6,0,0,0,0,5,42,0,23,28,43,0,21,20,45,0,46,11,55,44,33,38,25,15,56,48,19,9,2,0,0,7,0,47,0,59,49,54,0,14,30,18,10,1,
    54,0,28,31,4,42,58,0,0,16,47,0,48,0,0,7,0,9,0,0,14,32,0,53,0,46,26,35,0,13,10,49,0,0,33,43,3,0,40,55,22,56,51,19,44,5,37,24,17,39,6,25,60,15,0,45,57,0,0,52,
    38,0,6,26,60,0,0,30,8,46,7,9,13,56,35,58,0,48,11,1,0,0,0,23,32,34,0,43,54,29,17,0,21,10,37,0,40,3,52,36,0,0,15,18,51,0,0,20,0,0,4,0,25,41,0,24,42,2,19,59,
    49,52,0,45,0,0,35,51,10,9,4,0,23,0,12,33,44,27,28,17,0,40,56,1,37,22,38,0,7,57,0,0,25,0,0,0,30,32,11,39,19,3,0,13,42,0,60,0,0,58,55,16,34,0,26,2,20,0,48,14,
    26,0,0,0,0,10,51,52,17,0,9,8,36,55,32,18,50,0,39,37,29,0,53,24,34,6,31,0,0,0,7,59,0,45,21,0,0,0,54,12,35,23,13,0,56,0,0,5,15,57,16,42,0,19,49,33,46,27,60,30,
    19,0,55,0,48,0,60,46,0,17,0,0,29,14,0,0,0,0,24,18,35,0,0,11,52,45,25,38,32,28,36,15,30,42,23,50,49,0,9,5,0,12,39,37,41,44,40,51,7,27,0,4,22,57,3,6,47,0,0,0,
    36,6,34,52,0,0,0,17,0,41,32,0,20,39,9,0,24,13,0,31,22,18,5,35,0,57,0,44,15,33,0,30,0,55,0,40,38,2,58,60,0,0,43,0,1,0,25,28,53,49,0,45,46,7,48,42,54,0,14,16,
    41,27,3,0,5,49,0,58,0,48,6,14,18,26,15,2,60,0,52,40,44,0,55,0,17,29,0,53,0,11,54,37,0,8,31,34,0,45,36,24,20,39,21,25,0,0,13,19,0,28,56,0,42,0,0,0,59,0,51,33,
    6,25,0,37,53,22,19,0,52,0,46,24,14,2,21,0,20,42,0,15,23,0,0,39,26,31,54,34,0,0,27,0,0,5,44,1,10,0,13,7,48,43,11,60,0,0,41,56,3,0,29,0,0,4,0,30,38,12,50,57,
    0,17,0,5,0,9,38,57,0,20,13,0,0,16,49,24,41,34,0,0,47,0,0,31,0,0,56,40,18,0,0,25,19,15,0,11,52,0,29,51,45,42,55,0,7,53,33,12,26,59,2,36,30,48,8,0,32,3,43,28,
    42,30,57,0,2,41,54,0,18,51,40,0,0,3,19,0,39,28,0,0,0,15,0,60,16,38,17,4,46,0,0,32,8,14,6,10,5,47,33,48,7,44,1,43,27,23,0,31,0,35,21,0,37,11,0,25,0,13,0,0,
    0,39,0,23,0,0,7,0,0,53,14,0,49,38,55,34,0,58,16,0,5,45,59,37,0,41,0,0,19,15,40,31,17,0,32,0,48,22,60,50,26,0,6,56,33,57,4,44,21,11,18,0,36,27,10,0,0,29,25,12,
    5,49,45,33,40,0,0,37,15,32,27,59,34,0,4,30,0,0,42,0,11,60,0,56,55,0,52,21,0,12,29,16,35,24,54,0,1,0,20,23,2,0,36,53,6,0,0,0,46,17,58,50,0,0,22,48,9,14,0,18,
    0,0,12,32,8,36,22,0,31,11,19,33,0,54,37,60,14,21,0,16,0,0,13,47,43,15,20,28,39,18,34,0,49,2,56,9,58,26,0,0,6,4,30,0,50,24,42,0,51,40,0,48,0,0,29,7,25,0,0,0,
    50,38,0,8,57,24,0,0,0,0,59,0,22,32,13,0,0,41,27,47,25,43,6,0,14,10,0,2,26,0,44,29,0,7,52,42,45,46,0,31,18,54,17,21,34,60,0,9,20,0,0,0,23,12,1,40,0,11,3,19,
    15,50,4,54,51,23,47,0,0,28,18,7,46,58,0,0,26,17,9,11,60,0,0,33,41,0,0,6,29,0,35,0,37,1,0,14,0,59,0,0,12,2,3,0,0,21,53,10,39,45,49,31,0,25,43,5,13,19,24,0,
    57,44,0,0,19,30,0,0,36,0,58,13,0,0,1,17,22,23,0,6,0,10,26,21,9,20,59,8,25,7,16,0,53,31,0,3,0,27,0,46,0,0,2,49,4,56,14,52,0,42,40,24,28,32,60,0,55,43,5,0,
    0,40,53,29,33,1,26,27,20,0,52,0,24,50,42,39,16,38,43,0,46,8,4,15,0,21,14,0,5,48,41,0,18,0,35,17,55,0,49,0,0,57,0,0,54,51,58,2,0,60,0,22,45,0,0,0,36,31,34,11,
    22,33,50,0,15,51,0,10,43,36,17,53,19,6,25,0,47,0,60,54,0,0,0,4,0,3,23,1,2,56,57,14,0,0,39,0,41,0,18,0,16,28,38,11,8,45,0,0,0,9,27,0,7,0,31,49,26,44,30,55,
    2,54,0,20,0,40,31,11,0,0,0,0,0,29,7,25,19,52,35,49,48,0,28,6,5,23,24,3,16,8,15,0,41,34,36,37,47,0,4,26,0,17,53,14,55,13,43,0,50,32,44,0,57,0,0,0,0,0,59,58,
    25,42,39,11,38,56,1,45,0,60,30,0,54,8,0,0,55,6,49,41,15,24,0,0,3,12,16,20,13,0,0,0,0,50,48,21,0,40,53,18,27,7,10,31,46,0,52,0,0,0,0,44,0,58,0,47,35,59,17,32,
    0,19,0,21,43,7,39,0,29,12,42,54,2,0,0,31,34,55,37,33,1,59,0,0,44,0,3,11,4,0,0,27,36,0,18,30,0,0,32,13,0,35,58,40,28,0,5,45,0,41,53,57,20,16,0,52,17,0,8,10,
    0,13,27,25,59,14,0,22,16,49,20,46,0,60,47,0,54,44,40,0,0,0,35,2,51,36,18,0,0,26,53,12,0,0,7,39,4,0,3,0,21,34,28,0,45,15,0,0,33,6,11,19,0,38,32,56,52,0,29,31,
    7,0,0,0,0,34,46,0,56,44,41,26,40,20,60,21,57,36,32,25,0,42,19,45,0,58,48,0,0,50,0,6,55,30,24,31,51,39,12,0,0,0,4,3,23,35,15,59,29,0,52,5,1,53,0,43,0,49,0,0,
    0,0,59,42,45,6,37,39,1,0,31,32,12,47,0,52,18,0,0,38,36,17,10,0,20,44,46,33,51,0,0,7,0,0,15,23,29,58,14,35,57,8,0,34,0,0,49,13,11,54,0,40,0,0,55,0,16,4,22,9,
    0,36,60,16,22,3,0,0,55,24,0,51,50,33,11,0,0,35,7,2,57,38,47,34,59,0,40,12,9,0,19,0,0,0,14,48,0,0,0,30,52,26,37,0,25,6,44,58,49,31,0,8,0,46,20,32,29,56,0,13,
    0,0,21,38,0,19,4,56,25,0,0,49,53,31,17,32,42,0,0,50,24,34,18,29,33,0,8,0,0,35,0,23,0,0,26,5,0,57,10,9,28,40,0,45,22,39,59,11,0,15,0,54,0,37,46,36,27,1,6,20,
    16,7,25,55,0,32,50,6,33,0,0,0,52,37,14,0,29,12,5,8,28,11,44,48,0,2,0,49,10,3,0,0,20,17,13,36,0,30,0,19,60,0,0,58,40,41,0,35,54,34,46,27,18,22,0,59,0,0,0,4,
    53,20,32,34,0,43,57,42,0,0,0,36,9,0,30,56,48,11,58,0,0,0,0,51,0,13,47,0,44,1,52,41,54,29,16,46,35,14,17,21,0,0,26,0,5,37,6,0,40,0,0,7,31,8,19,0,2,45,4,3,
    14,0,7,58,0,0,18,24,0,43,0,42,56,45,59,28,0,1,33,35,55,52,0,0,50,0,34,17,0,39,49,40,22,3,10,54,21,16,0,38,0,53,0,44,37,0,47,48,31,0,13,2,0,6,4,15,8,36,0,0,
    0,5,0,0,41,0,40,16,0,15,24,11,44,0,0,45,30,0,38,53,27,14,1,0,49,39,0,32,3,21,25,0,29,0,0,7,46,52,57,2,51,0,31,47,20,59,18,0,12,10,50,0,6,55,34,0,43,37,0,48,
    4,0,41,43,7,0,0,60,26,5,8,58,0,44,20,9,0,0,22,27,0,54,0,14,0,56,2,0,33,47,13,19,59,6,0,18,57,0,0,3,0,25,29,30,0,0,0,0,23,37,24,35,17,31,39,53,12,28,15,42,
    0,29,5,10,46,12,11,55,4,0,21,16,32,0,45,35,1,60,2,57,58,0,37,44,28,17,50,0,0,0,0,39,15,0,9,0,14,31,7,49,0,41,0,0,0,36,54,27,30,33,38,26,59,0,0,0,56,34,20,0,
    0,0,0,0,0,47,16,40,0,0,43,2,58,0,0,11,49,29,13,9,31,48,52,0,27,0,0,18,0,41,6,26,50,44,0,15,59,36,28,33,55,0,12,51,3,34,7,53,32,8,39,30,10,0,0,22,14,46,0,5,
    40,51,58,15,26,0,6,29,24,0,55,0,59,41,0,53,0,5,36,7,0,57,54,0,0,18,1,0,0,0,37,45,16,33,0,25,27,0,8,0,0,21,56,0,35,20,9,34,60,0,22,28,3,43,38,17,48,0,49,47,
    30,0,0,35,42,0,34,19,11,50,60,31,47,15,56,36,53,49,4,10,54,1,21,27,7,9,0,22,48,24,39,0,14,0,57,0,0,0,23,58,0,59,8,2,0,17,28,0,0,0,41,20,16,0,12,38,0,33,0,0,
    18,0,2,51,52,0,23,0,14,0,0,57,28,0,22,1,25,0,0,0,17,4,43,0,12,53,30,39,6,16,3,48,42,56,0,0,7,41,45,0,29,10,0,26,31,38,0,0,59,50,19,15,58,0,24,46,0,35,13,37,
    39,32,19,60,49,0,52,18,44,4,0,0,15,46,40,23,0,31,29,14,2,0,7,25,10,0,11,0,1,0,5,42,0,53,50,47,37,0,6,0,8,51,0,57,0,12,36,22,56,0,3,0,26,21,0,13,0,16,9,0,
    10,41,24,4,37,28,0,31,59,2,0,25,0,5,26,55,0,7,0,0,20,0,30,0,0,48,0,0,34,17,47,38,44,0,53,58,0,13,0,22,0,6,46,8,14,54,3,0,52,43,9,12,27,18,0,0,23,57,40,49,
    28,18,11,6,0,44,9,15,32,19,53,38,0,0,51,37,7,2,20,0,10,16,50,55,48,0,0,24,31,0,0,36,0,41,46,0,54,12,0,29,49,0,0,5,0,0,17,0,43,23,0,0,47,56,57,60,45,8,33,26,
    17,34,0,53,0,16,33,3,0,37,0,10,27,4,46,0,40,15,18,44,0,29,0,58,0,52,55,23,56,19,0,43,9,0,0,41,0,50,47,8,11,31,32,39,21,28,20,0,0,0,5,0,0,49,35,57,6,38,7,0,
    46,43,0,40,30,39,13,0,45,21,56,35,0,10,27,26,59,18,19,28,0,55,3,17,4,0,5,0,52,0,38,20,32,0,12,60,23,0,0,53,0,24,0,41,0,7,0,1,0,48,57,0,8,33,9,51,22,58,0,0,
    0,2,0,14,0,26,0,50,12,22,0,0,55,57,0,44,35,53,25,19,0,56,24,0,0,5,4,48,43,58,60,0,51,49,0,59,42,28,1,0,0,32,41,54,0,30,27,15,36,7,23,0,38,52,0,20,0,6,37,39,
    0,0,0,7,25,50,0,43,2,10,0,27,0,0,23,57,0,0,3,0,6,13,14,54,60,0,44,59,0,53,8,46,0,9,0,52,32,20,0,41,24,58,45,29,11,0,48,38,22,0,31,49,5,28,51,0,15,17,42,21,
    0,0,42,28,44,15,20,0,58,33,0,60,7,48,2,0,9,56,51,13,26,39,57,46,0,14,12,0,0,49,55,8,0,22,30,0,31,29,0,0,3,0,34,0,0,11,1,43,0,25,0,21,32,24,0,18,41,40,16,50,
    58,22,23,39,10,0,0,25,57,7,44,5,0,30,0,0,12,50,0,20,0,49,0,0,2,16,41,46,60,40,43,13,48,38,19,56,0,42,0,52,54,14,33,0,9,47,0,6,28,0,0,32,11,34,37,0,0,0,36,45,
    0,37,20,0,6,0,28,54,53,27,0,41,0,17,0,47,45,4,23,43,51,9,46,0,0,60,39,0,0,42,0,34,12,16,0,0,13,33,56,11,5,52,59,35,0,48,0,29,2,21,1,3,15,0,40,44,0,0,18,22,
    29,15,26,0,54,21,30,0,41,0,36,48,4,0,0,0,43,0,0,22,56,47,23,49,58,50,0,25,11,46,20,28,0,12,59,0,0,17,38,6,14,0,40,16,60,55,45,8,57,13,51,34,0,1,27,35,0,0,0,0,
    56,47,43,0,32,52,5,0,9,0,29,18,16,34,41,0,46,0,0,0,13,6,22,12,0,0,36,57,58,30,31,0,26,60,0,33,17,44,37,42,39,11,0,0,0,25,2,55,1,19,14,51,21,35,50,23,0,0,0,0
  };

  const int d60_1152_bal[] = {
    // Size: 60 x 60
    60,
    // Pre-assigned fields
    8,14,44,0,35,25,0,26,0,6,57,0,41,7,0,40,0,0,31,36,50,0,0,52,38,49,0,56,37,51,24,58,45,59,1,0,15,9,34,43,47,22,16,33,18,19,39,0,10,0,42,0,54,0,0,0,28,23,0,0,
    0,0,17,9,0,0,0,0,3,0,25,55,38,59,16,43,0,14,57,26,0,2,48,13,15,7,0,10,45,0,18,22,11,0,0,29,6,53,0,54,36,46,0,50,58,1,35,33,37,20,47,41,0,23,44,0,21,0,56,0,
    59,45,18,44,58,11,0,41,28,0,26,23,0,0,3,49,0,57,1,48,7,0,31,43,0,37,9,54,38,0,0,0,10,39,0,22,60,25,24,0,50,0,0,42,47,0,32,0,0,16,12,0,33,30,13,0,34,20,46,0,
    1,12,0,46,20,0,41,0,34,26,0,21,3,13,0,19,23,0,0,56,0,36,0,22,31,27,0,42,30,32,48,2,0,43,8,55,0,24,15,37,58,16,9,7,0,0,51,18,0,0,59,10,39,14,0,0,5,60,57,0,
    35,0,10,0,31,0,0,0,0,34,0,15,5,25,38,46,33,0,21,29,53,37,11,0,36,32,27,58,41,0,51,50,43,48,3,13,24,0,0,28,40,18,20,1,19,0,23,16,14,0,0,0,12,0,7,0,49,30,52,0,
    45,10,0,27,0,59,14,48,6,0,37,17,0,52,31,3,51,0,0,0,0,53,36,0,8,26,33,29,42,5,28,54,23,21,0,0,50,19,0,47,0,55,0,0,0,46,11,0,0,22,32,60,13,40,2,39,0,9,0,34,
    27,0,16,0,23,37,8,14,0,0,0,30,51,9,50,20,36,0,6,0,12,25,17,57,22,0,58,13,59,38,1,60,52,4,43,2,0,15,0,40,31,49,44,32,10,0,29,47,24,53,0,0,0,42,5,0,0,0,0,0,
    0,24,0,0,18,38,49,0,7,13,15,22,43,36,33,0,52,16,0,45,21,44,8,0,0,54,0,26,0,0,0,11,0,57,47,35,28,55,5,25,34,29,0,46,53,9,56,42,0,0,60,37,48,17,0,31,0,51,0,40,
    0,60,0,0,29,53,32,33,0,54,22,52,37,24,5,6,2,0,55,0,8,0,0,0,56,1,45,0,57,10,30,47,28,0,38,16,11,0,31,4,59,19,35,0,13,0,0,49,44,18,43,46,41,39,0,34,7,0,0,0,
    0,0,37,13,55,0,10,0,0,8,0,0,0,18,0,22,15,0,45,42,34,7,39,3,0,33,0,0,27,6,23,44,0,0,0,49,9,51,41,17,0,1,0,28,0,16,46,36,58,0,20,11,29,2,14,19,50,54,38,35,
    31,0,0,0,0,29,15,0,21,57,38,39,0,35,10,8,3,33,14,0,0,0,20,30,0,0,60,0,0,9,0,52,0,54,25,45,12,49,0,0,4,5,42,22,24,58,19,37,34,44,36,0,56,51,28,50,18,26,0,23,
    0,58,36,50,17,13,0,0,0,29,39,4,0,27,57,41,0,0,8,32,37,5,42,0,23,0,43,31,21,20,0,53,46,11,55,44,33,38,25,15,0,0,19,0,0,3,0,7,0,0,34,0,0,54,52,14,30,18,10,1,
    54,0,28,31,4,0,0,21,23,16,47,20,48,11,0,0,0,0,34,30,14,0,2,0,29,46,26,35,12,0,10,49,27,0,0,43,0,8,40,0,22,56,51,19,44,0,0,0,17,39,0,25,60,15,41,45,0,50,1,52,
    38,57,0,26,0,27,0,30,8,46,7,0,0,56,0,58,0,48,11,0,39,31,45,23,32,34,22,0,54,29,17,0,0,10,37,12,40,0,52,36,53,33,15,18,0,49,0,20,47,0,4,14,0,0,16,24,42,0,0,59,
    49,0,15,45,0,0,35,51,10,0,4,29,23,0,12,33,44,0,28,0,43,40,0,1,37,22,0,50,7,57,46,0,25,47,41,0,0,32,0,39,0,3,54,13,0,31,0,0,8,58,55,16,0,59,26,2,20,0,48,14,
    26,48,40,2,11,10,51,52,17,0,9,0,36,0,32,0,0,20,39,0,29,58,53,24,0,6,31,0,0,44,0,0,1,45,21,28,25,0,54,0,0,23,13,4,56,22,0,5,15,0,16,42,0,19,0,33,46,0,60,30,
    19,8,0,56,48,0,60,46,13,17,54,34,29,14,58,0,31,0,0,18,35,20,0,11,0,0,25,38,32,28,0,0,30,0,23,50,49,0,9,5,0,0,39,37,0,44,0,51,7,0,0,4,22,57,0,6,47,21,26,53,
    0,6,0,52,0,4,0,17,19,41,32,50,20,0,9,27,24,13,0,31,22,18,5,35,11,57,29,0,15,0,56,30,3,55,51,40,38,0,58,60,23,37,0,0,1,0,25,28,0,49,0,45,46,0,0,0,0,0,14,0,
    0,27,3,0,5,49,0,58,47,0,6,14,18,26,15,2,60,46,52,40,44,35,55,10,17,29,7,0,22,11,0,0,0,0,31,34,0,0,36,24,20,0,21,0,57,4,0,19,9,0,0,23,0,50,30,0,59,32,51,0,
    6,0,47,37,53,22,19,0,52,58,46,24,14,2,21,51,0,42,17,0,23,0,0,39,26,31,54,0,49,0,27,0,40,5,44,1,0,0,13,7,48,43,11,60,16,32,0,0,0,36,29,0,35,0,0,30,0,12,50,0,
    0,17,14,5,50,0,0,57,39,20,0,1,10,0,0,24,41,34,46,60,47,23,0,31,54,35,56,0,0,0,21,0,19,0,22,11,52,0,29,51,45,42,55,0,0,0,33,0,26,59,0,36,30,0,8,58,32,3,43,28,
    0,0,0,49,0,0,54,20,0,51,40,0,26,0,19,50,0,28,59,52,45,15,12,60,16,0,17,0,46,34,22,0,0,0,6,10,0,47,33,0,7,44,1,43,27,23,0,31,55,35,21,9,37,0,36,0,0,13,58,29,
    43,39,46,23,13,54,7,0,0,0,14,0,49,38,55,0,8,58,16,51,5,45,59,37,0,41,42,9,0,0,40,31,17,0,0,0,48,0,60,0,26,20,6,56,0,57,0,44,21,0,18,52,36,0,10,0,3,29,0,12,
    5,49,45,33,40,0,25,0,15,0,27,59,34,0,4,30,13,51,0,39,11,60,0,0,55,47,0,21,28,12,0,0,35,0,54,8,1,0,20,0,2,38,0,53,6,10,57,26,46,17,58,0,19,3,0,48,9,0,0,0,
    3,1,12,0,8,0,22,0,31,0,19,33,35,0,37,60,14,0,0,16,38,0,0,47,0,0,20,0,39,0,34,57,0,2,56,9,58,26,59,0,6,0,30,52,50,0,42,17,51,40,0,48,53,10,29,7,25,0,23,46,
    0,38,35,0,57,0,0,28,49,55,59,0,22,32,13,48,58,41,27,0,25,43,6,5,0,0,51,0,26,4,44,29,0,0,0,0,0,0,16,0,18,54,17,21,34,0,30,9,20,56,15,0,23,12,1,40,33,11,3,0,
    0,50,4,0,51,0,47,34,38,28,18,7,46,58,0,16,26,17,9,0,0,22,40,33,0,55,32,6,29,0,0,56,0,1,42,14,20,0,44,0,12,0,3,0,30,21,53,10,39,45,49,0,52,0,0,0,13,0,0,8,
    57,44,0,47,19,30,29,12,0,0,0,13,11,51,1,17,22,23,48,0,0,0,26,0,9,20,59,0,25,0,0,35,53,31,45,3,34,0,0,46,0,15,2,49,4,56,14,52,0,42,0,24,28,32,60,37,0,43,0,0,
    13,40,0,29,0,1,26,27,20,47,0,0,0,50,42,39,0,38,0,12,46,8,0,15,30,0,14,7,5,48,41,3,18,28,0,0,0,23,0,0,0,57,25,59,54,51,58,0,19,60,0,22,45,0,56,9,36,31,0,11,
    22,33,50,24,15,0,0,10,43,0,17,53,19,0,0,29,47,37,0,0,52,0,34,4,42,3,23,0,0,56,57,14,13,32,0,20,41,21,18,59,16,28,38,0,0,45,12,0,35,9,27,0,0,5,0,0,26,0,30,55,
    0,54,0,20,0,40,31,11,46,0,33,45,42,29,0,25,19,52,0,49,0,12,28,6,5,23,24,3,0,0,15,21,41,34,0,37,0,56,0,26,0,0,53,14,0,13,43,39,0,32,44,38,57,9,18,27,0,0,0,58,
    25,42,39,11,38,56,1,0,5,60,0,19,54,0,34,14,55,6,0,41,15,24,33,0,3,0,0,20,13,22,26,9,0,50,48,0,0,0,53,0,0,0,0,0,46,43,52,57,4,29,37,44,0,0,23,47,35,0,17,32,
    24,19,22,21,43,7,39,38,29,0,42,54,0,49,6,31,0,55,37,0,1,59,0,0,0,51,0,11,4,0,0,0,36,46,0,30,56,60,32,13,0,35,0,40,0,14,5,45,0,41,53,57,0,16,0,52,17,25,8,0,
    9,13,27,0,59,0,17,22,16,49,20,46,0,0,0,0,54,0,40,55,0,50,35,2,51,36,0,5,24,26,53,0,57,58,0,39,0,37,0,1,21,34,28,0,45,15,8,0,33,6,11,19,43,0,32,0,0,41,0,31,
    0,0,38,0,28,34,46,13,56,0,41,26,0,20,0,0,0,36,32,25,9,0,0,45,18,0,48,0,8,0,33,0,55,30,24,31,51,39,12,0,17,47,4,0,23,0,0,59,29,0,52,5,1,53,54,43,0,49,27,2,
    21,28,0,42,45,6,37,39,1,56,31,32,12,47,0,0,18,0,26,38,0,0,0,0,0,44,46,33,0,0,2,7,0,25,15,23,0,58,14,35,0,8,5,34,48,27,49,0,11,54,30,40,50,0,55,0,0,0,22,9,
    0,36,60,16,22,3,42,53,55,24,10,0,50,0,11,15,0,35,7,0,57,38,0,0,59,43,40,12,9,45,19,0,5,27,14,0,0,54,39,0,52,0,37,0,25,0,44,58,0,31,28,0,0,46,20,0,29,56,0,13,
    0,0,21,38,3,19,4,56,25,0,12,49,53,0,0,0,0,43,47,50,24,0,18,0,33,0,0,51,55,35,0,23,60,13,26,5,2,57,10,9,0,0,0,45,22,39,59,11,41,0,48,54,44,0,0,36,27,1,6,0,
    16,0,25,55,56,32,0,6,33,42,1,47,0,37,14,38,29,0,0,8,0,11,44,48,57,0,53,0,0,3,9,24,20,0,0,36,43,30,21,0,60,0,23,58,40,41,26,0,54,0,0,27,18,22,0,59,51,39,0,0,
    53,0,32,0,0,43,0,42,22,59,23,0,0,28,30,56,0,11,58,24,18,33,49,0,39,0,47,0,0,1,0,0,54,0,16,46,35,14,17,21,15,0,26,0,5,37,0,60,40,12,25,7,31,0,19,10,2,45,0,3,
    14,26,7,0,0,0,18,24,0,43,0,0,0,45,59,0,32,1,33,35,0,0,60,19,0,25,34,17,0,39,49,40,22,3,10,0,0,16,30,38,0,53,0,44,37,29,0,48,0,46,13,0,9,6,0,15,8,36,11,51,
    0,5,8,36,0,60,0,0,0,15,24,0,44,0,0,0,30,22,0,53,0,14,0,9,49,0,35,32,0,21,25,17,29,19,58,0,46,52,57,0,51,13,31,47,20,0,18,4,0,10,50,56,6,55,34,26,43,0,54,48,
    0,46,0,0,0,48,21,60,26,5,0,0,0,44,0,9,10,32,0,27,49,54,51,14,0,56,0,52,0,47,13,19,59,6,11,18,57,34,0,0,38,0,29,30,36,0,16,0,23,37,24,35,0,31,0,53,12,28,15,42,
    0,29,5,10,46,12,0,55,0,18,0,16,32,0,45,35,1,0,2,57,58,0,37,44,28,17,0,19,40,52,42,39,0,23,9,51,14,0,0,49,25,0,0,0,0,36,54,0,30,33,38,0,0,13,53,8,0,34,20,6,
    20,0,0,57,24,47,16,40,0,25,0,2,58,0,0,0,49,29,0,9,31,48,52,0,27,4,0,18,35,41,6,26,0,44,0,15,59,36,0,33,55,60,0,51,3,34,0,53,0,0,0,30,10,45,42,22,0,46,21,5,
    40,51,58,0,26,46,0,29,0,23,0,0,59,41,39,53,11,5,36,7,0,57,54,0,13,18,1,30,0,31,37,0,16,33,0,25,27,10,0,0,0,21,56,0,35,20,9,34,0,0,0,28,3,43,38,0,48,52,49,0,
    30,3,52,35,0,0,34,19,11,50,0,31,47,15,56,0,0,49,0,10,54,1,0,27,7,9,6,22,48,24,0,18,0,0,57,0,44,5,0,58,46,59,0,0,32,17,28,0,0,51,41,20,16,0,12,38,0,33,0,0,
    18,9,2,0,0,0,23,47,14,40,49,0,28,21,22,0,0,8,44,5,17,0,0,20,12,53,30,39,6,16,0,48,42,56,27,0,7,0,0,34,29,10,60,0,31,0,55,0,59,0,0,0,58,36,24,46,11,0,13,37,
    39,32,19,60,49,55,52,18,44,0,0,0,15,46,40,23,27,0,29,14,2,30,7,25,10,0,0,45,0,54,5,0,34,53,0,47,0,35,6,20,8,0,0,0,59,0,0,0,56,38,0,0,26,21,0,13,58,16,9,41,
    10,0,0,4,37,28,0,0,0,2,0,25,39,5,0,0,56,7,50,21,0,51,30,16,0,48,15,60,0,17,0,0,0,0,53,0,0,13,42,22,1,6,46,8,14,54,3,32,52,43,9,12,0,0,33,29,0,57,40,49,
    28,18,0,6,34,44,9,0,0,19,0,38,0,42,51,0,7,2,20,59,10,0,50,0,48,40,21,24,31,25,14,36,58,41,0,0,54,12,27,29,0,0,22,5,0,0,0,3,0,23,35,13,0,56,0,60,45,8,33,0,
    17,0,0,53,14,16,33,3,0,37,51,10,27,0,46,0,40,15,18,44,59,0,25,0,45,52,55,23,56,0,12,43,9,26,0,0,0,0,47,8,11,0,0,39,21,0,20,0,42,2,5,1,0,49,35,57,0,38,7,0,
    46,43,31,0,0,0,13,36,45,21,56,0,0,10,27,26,59,18,19,0,0,0,3,17,4,0,0,15,52,14,38,20,32,0,12,60,23,11,0,53,44,24,0,0,0,7,34,1,6,0,57,0,8,0,9,51,0,58,47,54,
    11,2,29,14,9,26,3,50,0,0,16,0,55,57,8,44,35,0,25,0,0,56,24,18,0,5,0,48,0,58,0,10,0,0,0,59,42,28,1,0,13,32,0,54,0,0,27,15,36,0,23,47,38,52,21,0,31,6,37,39,
    47,35,30,0,25,50,55,43,2,0,34,27,0,12,23,0,37,0,3,4,6,0,0,54,60,19,44,0,36,0,0,0,0,0,40,52,32,20,26,41,0,58,45,29,0,18,48,38,22,1,0,0,5,28,0,16,0,17,42,21,
    37,0,42,0,0,15,20,0,0,33,45,60,7,48,2,5,9,56,51,0,0,0,57,46,0,0,12,27,0,49,55,8,47,22,30,0,31,29,35,10,0,0,0,23,52,11,0,43,0,25,0,21,0,24,59,18,41,40,16,50,
    58,0,0,39,10,35,27,25,57,7,0,5,17,0,18,4,12,50,0,0,0,49,0,59,0,16,41,0,0,0,43,13,48,0,0,0,0,42,51,52,54,14,33,55,9,47,31,6,28,24,26,32,11,34,0,21,1,0,0,45,
    55,37,20,30,0,58,0,0,53,27,0,41,0,0,0,47,0,4,0,43,0,9,46,8,0,60,39,36,14,42,0,34,12,16,49,57,13,33,56,11,0,52,0,35,0,48,10,0,2,21,0,3,0,0,40,0,19,7,18,22,
    0,15,0,18,0,21,30,0,41,31,36,48,0,19,0,0,0,3,10,22,56,47,23,0,58,0,0,25,11,46,20,28,0,12,59,0,39,17,0,6,14,9,40,16,60,55,0,0,0,13,51,34,0,1,27,0,24,0,32,7,
    56,0,43,48,32,52,5,0,0,38,29,0,16,34,41,54,0,59,0,0,13,6,22,12,24,0,0,57,58,30,0,4,26,60,28,0,17,0,0,42,39,0,49,20,0,25,2,55,1,19,0,0,21,35,50,0,40,0,45,27
  };

  const int d60_1440[] = {
    // Size: 60 x 60
    60,
    // Pre-assigned fields
    0,14,44,0,35,0,2,26,48,0,0,0,41,0,0,40,0,30,31,36,50,0,32,52,0,0,0,56,37,51,0,58,45,0,1,0,0,0,34,43,47,22,16,33,18,19,0,46,10,0,0,17,0,20,11,4,0,23,53,0,
    34,31,0,9,39,8,12,0,3,0,0,55,38,59,0,43,0,14,57,0,0,0,0,13,0,0,49,10,45,60,0,22,0,0,0,0,6,53,19,0,0,46,0,50,0,0,35,0,37,0,0,41,40,23,0,28,21,0,56,0,
    59,0,0,44,0,11,53,41,28,35,26,0,0,0,3,0,17,0,1,48,0,19,0,43,21,37,0,54,38,36,4,51,10,39,29,22,60,0,24,56,50,27,52,0,47,0,32,14,0,16,12,0,33,30,0,55,34,20,0,15,
    0,0,33,46,20,17,41,49,34,0,0,21,3,0,44,19,23,47,54,0,40,36,38,0,31,0,28,42,30,32,0,2,4,43,8,55,53,24,0,0,0,16,0,7,29,0,0,18,45,52,0,10,0,14,6,11,0,0,57,0,
    0,4,0,59,31,0,56,9,0,0,2,0,5,25,38,46,33,45,21,29,53,0,11,0,36,32,27,58,41,0,0,50,0,48,3,0,24,6,0,28,40,0,20,1,19,8,23,16,0,26,17,39,0,0,0,54,49,0,52,44,
    0,10,49,27,16,0,0,48,0,0,0,17,0,52,31,0,51,24,56,0,0,53,0,7,0,26,0,29,0,5,28,54,0,0,20,38,0,19,43,47,30,55,18,0,12,46,11,41,25,0,32,60,13,0,0,0,44,9,35,34,
    27,21,0,19,23,37,0,14,35,0,3,30,51,9,50,20,36,54,6,0,12,25,17,57,22,11,58,13,59,0,1,60,0,0,0,2,26,0,0,40,31,49,44,0,10,0,0,47,0,53,7,18,0,42,5,0,0,48,0,56,
    32,24,0,0,18,38,49,0,7,0,15,0,0,0,33,12,0,0,0,45,0,44,8,50,19,0,0,26,0,2,59,0,6,0,0,35,0,55,0,25,34,0,14,46,0,0,0,0,27,3,0,37,48,17,58,31,0,0,39,40,
    51,0,9,3,29,0,0,33,0,0,0,0,0,24,0,0,0,0,0,23,8,26,0,36,0,1,0,14,0,0,0,47,28,20,38,16,0,48,0,4,0,19,35,27,13,42,21,49,44,0,43,46,41,0,0,0,7,15,12,0,
    0,56,37,0,55,5,10,59,40,0,0,12,21,18,0,0,15,0,45,0,34,0,0,0,0,33,0,0,0,6,23,44,0,0,4,49,9,51,0,17,32,1,24,0,26,0,46,0,0,0,0,0,0,2,14,0,50,0,38,35,
    31,55,13,0,47,29,0,1,21,0,0,0,6,0,10,8,3,0,14,0,32,41,20,30,0,59,60,0,53,9,11,0,7,54,25,45,0,0,0,27,0,5,0,0,24,58,19,37,0,0,0,43,0,51,0,50,18,26,0,23,
    12,58,36,50,17,13,24,35,0,29,39,0,0,27,57,0,6,0,8,0,37,0,0,0,23,28,0,0,21,20,45,0,0,0,0,44,33,38,25,0,0,48,19,9,2,0,22,7,0,0,34,59,49,54,52,14,30,0,10,1,
    54,0,28,0,0,42,0,21,23,0,47,20,48,11,0,0,38,9,34,30,0,32,2,53,29,46,26,35,0,0,0,49,27,18,0,43,0,0,40,55,0,56,51,0,0,5,0,24,17,0,6,25,60,15,0,0,0,0,1,0,
    38,0,6,26,60,0,44,30,0,46,7,0,0,0,35,58,28,0,11,1,39,0,0,23,32,34,0,0,54,0,17,55,21,0,37,0,0,3,0,0,0,33,0,18,51,49,50,0,0,5,0,0,0,0,16,0,0,0,19,0,
    49,52,15,0,0,0,35,0,10,9,0,0,0,0,0,0,44,0,0,17,43,0,0,0,37,22,0,0,7,0,0,0,25,47,0,6,0,32,11,39,19,0,54,13,42,31,60,21,0,58,0,16,34,0,0,2,0,0,48,14,
    26,48,0,0,11,0,0,52,0,3,9,8,36,0,32,0,0,0,39,37,0,0,0,0,34,6,0,41,47,44,0,59,1,0,21,0,0,43,54,12,35,0,0,4,0,22,38,0,0,0,0,0,14,0,49,0,46,27,60,0,
    0,8,55,0,0,2,60,46,13,17,0,34,29,14,58,59,31,10,0,0,35,20,16,11,52,0,0,38,0,28,0,15,30,42,23,50,0,1,0,5,43,0,0,37,0,44,0,51,0,27,33,0,0,57,3,0,47,21,26,53,
    0,6,0,52,21,4,0,0,19,0,32,50,20,39,0,0,24,13,12,0,22,18,5,35,11,57,29,44,0,33,56,0,3,55,51,40,38,2,58,60,23,37,0,0,1,26,0,0,53,49,8,45,0,7,0,42,0,47,0,16,
    0,27,3,1,0,49,43,58,47,48,0,14,18,26,15,2,0,46,0,40,0,0,55,10,17,0,7,0,22,11,0,37,0,0,31,34,0,0,36,0,0,0,0,25,0,0,13,19,9,28,56,0,42,0,0,0,59,32,0,33,
    6,0,0,37,53,22,0,8,52,0,46,24,14,2,0,0,20,42,17,15,23,0,9,0,26,31,54,34,0,0,27,0,40,5,0,0,10,18,0,7,48,0,11,60,0,0,41,56,3,36,29,55,35,4,0,30,38,0,50,57,
    44,0,0,0,0,0,38,0,0,20,13,1,10,0,49,0,41,0,46,60,47,23,0,31,54,35,56,40,0,0,21,0,19,15,22,11,0,0,29,51,45,0,0,0,7,0,33,12,26,59,2,36,30,48,8,58,0,0,0,0,
    42,30,57,49,0,41,54,20,18,0,40,0,0,0,19,50,0,28,0,52,0,0,12,0,16,38,0,4,0,34,0,0,8,14,0,10,0,47,33,48,7,44,0,0,27,23,24,31,55,35,21,9,0,11,0,25,53,13,58,0,
    0,0,46,0,13,0,7,2,0,53,14,0,49,38,55,0,0,58,0,0,5,45,0,37,47,41,42,9,19,0,40,0,0,0,0,0,0,22,60,0,0,0,6,56,0,57,0,0,0,11,0,52,0,27,0,1,0,0,25,12,
    5,49,45,33,0,31,0,37,15,32,0,0,34,43,0,0,0,0,42,0,0,60,0,0,55,0,52,0,0,12,0,0,0,0,54,8,1,7,20,23,0,38,36,0,0,0,57,26,46,17,0,50,19,3,22,0,9,14,44,0,
    3,1,12,32,8,0,0,0,31,0,19,0,35,0,0,60,14,0,41,0,38,0,0,47,43,15,20,0,39,0,0,0,49,2,0,9,0,26,0,0,0,4,0,0,0,0,42,17,51,40,45,48,53,10,29,7,25,55,23,0,
    0,38,35,8,57,24,36,28,0,55,59,37,22,32,0,48,0,41,27,47,25,43,6,5,0,0,0,2,0,4,44,29,0,0,0,42,45,46,16,0,18,54,0,21,0,60,30,9,20,0,0,0,23,12,0,40,33,11,0,19,
    0,0,0,0,51,0,0,34,38,28,0,0,46,58,48,16,26,17,0,11,60,22,40,0,0,0,32,6,29,0,0,56,0,1,42,0,20,59,0,0,0,0,3,36,0,0,0,0,39,0,0,31,52,25,43,5,13,19,0,8,
    57,44,54,0,19,30,29,0,36,0,58,0,0,51,0,17,0,23,48,6,41,10,0,21,0,20,0,8,25,7,16,35,53,0,45,0,34,27,50,0,33,15,0,49,0,56,0,0,0,42,0,24,0,0,0,0,55,0,0,38,
    0,40,53,29,33,1,26,27,0,47,0,6,0,50,42,0,16,0,43,12,0,0,0,0,0,0,0,0,5,0,0,3,18,28,35,17,55,23,49,32,37,57,25,59,0,0,58,0,19,60,10,0,0,44,0,9,0,31,34,11,
    0,33,50,0,0,0,48,10,43,36,17,53,0,0,0,29,0,0,60,54,52,0,0,4,42,0,23,0,2,0,0,0,0,0,39,20,0,21,0,59,0,28,38,0,0,45,12,0,0,9,0,0,7,0,31,0,26,44,30,0,
    2,54,51,20,1,40,31,0,0,30,0,45,0,29,7,0,19,52,35,0,0,0,0,0,5,23,24,0,0,8,15,21,41,0,0,0,47,56,4,26,0,0,53,0,55,0,43,39,0,32,44,0,57,9,18,0,60,22,59,58,
    25,0,0,0,0,0,1,45,5,0,30,0,0,0,34,0,55,6,0,0,15,24,33,28,0,0,0,0,13,22,0,9,0,50,0,21,0,0,0,0,0,7,10,0,46,43,0,57,4,0,37,44,51,58,23,47,35,0,17,32,
    0,0,22,21,43,0,39,0,29,12,42,0,2,49,0,31,0,55,37,33,1,0,15,26,44,0,0,11,4,0,50,27,36,46,18,30,0,0,32,0,0,35,58,40,0,14,5,45,48,0,0,0,20,16,47,52,17,25,8,0,
    9,13,27,0,59,0,0,0,16,49,20,46,0,0,47,10,0,0,0,55,42,50,35,0,0,36,0,5,24,0,0,0,57,58,7,0,0,0,3,0,21,34,28,48,45,15,8,23,33,0,0,0,43,0,32,56,0,0,29,31,
    0,11,38,22,0,34,46,13,56,0,41,0,40,20,60,21,0,36,0,0,9,0,0,45,18,58,48,0,0,0,33,0,0,30,24,0,51,39,12,16,0,47,0,0,0,0,15,59,29,14,0,5,1,0,54,43,10,0,27,0,
    21,28,0,42,0,6,37,0,0,0,31,32,0,0,0,0,0,19,0,38,0,17,10,0,0,0,0,33,0,43,2,7,24,0,15,0,29,0,14,35,0,0,0,34,48,27,0,13,11,0,0,0,0,60,55,3,16,4,0,9,
    23,0,60,0,0,0,42,0,0,0,0,51,50,33,0,0,0,35,0,2,0,0,47,34,59,43,0,0,9,45,19,0,5,27,14,0,0,54,0,0,52,26,0,0,25,0,44,58,49,31,0,8,4,46,20,0,29,0,41,0,
    0,16,21,38,0,0,4,56,25,14,12,49,0,0,17,0,42,0,47,50,24,0,18,0,33,30,8,51,0,0,58,23,60,13,0,5,2,0,0,9,28,0,7,45,0,39,0,11,0,0,48,0,44,37,0,36,27,0,6,0,
    16,0,0,55,0,32,50,0,33,0,0,0,0,37,0,0,0,12,0,8,0,11,44,0,0,2,53,0,10,3,9,0,0,17,13,0,0,30,0,0,60,45,0,58,0,41,0,35,0,34,46,27,18,0,0,0,51,39,0,0,
    53,20,32,34,27,0,57,42,0,59,0,36,0,0,30,56,0,11,0,24,18,33,0,0,0,13,47,55,0,0,52,41,0,0,0,46,35,14,17,0,0,50,0,38,0,37,0,60,40,12,25,0,31,8,0,0,0,0,0,3,
    0,0,0,0,0,0,18,24,0,0,0,42,56,0,0,0,32,1,33,0,55,0,0,19,0,25,34,0,0,39,0,40,22,3,0,54,0,16,30,38,41,53,0,0,37,0,47,48,0,46,13,0,9,0,0,15,0,0,11,51,
    33,0,8,36,0,0,40,16,0,15,0,11,0,23,0,45,30,22,0,0,0,14,0,9,49,0,0,0,0,21,0,17,0,19,58,0,46,52,0,2,0,13,31,47,20,0,0,4,0,0,0,0,0,55,34,0,43,0,0,0,
    0,0,0,43,0,0,0,0,26,5,0,0,45,0,0,0,0,32,0,27,49,54,51,14,0,56,2,0,0,47,13,19,59,6,11,18,0,34,55,0,0,0,0,30,0,40,0,50,23,0,0,35,17,0,39,53,12,28,15,42,
    48,29,5,10,46,0,11,55,4,0,21,16,32,0,0,35,0,60,0,0,58,3,37,44,0,0,50,19,0,0,0,39,0,0,9,0,0,31,7,0,25,0,47,24,0,36,0,0,0,33,38,26,0,13,0,0,0,34,0,6,
    20,23,56,57,0,0,16,40,0,25,0,2,58,0,54,11,49,0,13,9,0,0,52,38,0,4,19,0,35,41,6,26,50,44,0,15,59,0,0,33,0,60,0,0,3,34,7,53,0,0,39,0,10,0,42,22,14,46,21,5,
    40,51,58,15,0,46,0,29,24,23,55,44,59,41,0,0,11,5,0,0,19,0,54,32,13,18,1,30,50,0,0,45,16,0,2,0,27,10,0,14,0,0,56,0,0,0,9,34,60,4,22,0,3,0,0,17,48,0,49,0,
    30,3,52,35,0,0,0,19,11,0,60,31,47,0,56,0,0,0,0,0,54,1,21,0,0,9,6,0,48,24,39,0,0,0,57,0,0,5,23,0,46,59,0,2,32,17,0,25,0,51,0,0,0,0,0,38,37,0,55,0,
    18,0,0,51,52,0,23,47,0,40,0,0,28,0,22,1,25,0,44,0,0,4,43,0,12,0,30,39,6,16,0,48,42,0,0,32,0,41,45,34,29,10,0,0,31,38,55,0,0,50,19,15,58,36,24,0,11,35,13,37,
    39,0,19,0,49,55,52,18,0,4,28,0,15,46,40,23,27,31,0,14,2,0,7,25,10,0,0,45,1,54,5,42,0,53,0,0,0,35,6,20,8,51,0,0,0,12,36,22,56,38,3,33,26,21,17,13,58,0,9,41,
    10,0,24,0,37,0,45,31,0,2,11,25,39,0,26,55,0,7,0,21,0,51,30,0,35,0,15,60,34,17,0,38,44,36,53,58,19,0,42,22,0,0,0,8,14,54,0,32,0,0,0,0,27,18,33,29,0,0,40,49,
    0,18,0,6,0,44,0,15,0,19,53,38,1,42,0,0,0,0,20,0,10,16,50,0,48,40,0,24,31,0,0,36,58,0,46,4,0,12,0,29,49,0,22,5,0,0,17,3,43,0,0,0,47,56,0,60,45,0,33,26,
    17,34,0,53,14,16,0,0,54,37,51,10,0,0,46,13,40,0,0,0,59,29,25,58,45,0,55,23,0,0,0,0,9,26,60,41,22,50,47,0,0,31,32,39,21,28,20,0,42,2,5,1,0,0,35,57,6,38,7,36,
    46,43,0,0,0,39,0,36,0,0,56,35,25,10,27,0,0,18,19,28,16,0,0,17,4,0,0,15,52,14,38,20,32,37,0,60,23,11,2,0,0,0,0,41,49,7,0,0,6,0,57,29,8,33,0,51,0,58,47,54,
    0,2,29,0,9,26,0,50,0,0,16,0,0,57,8,44,0,53,25,19,33,0,24,18,0,5,0,48,43,58,0,0,51,0,34,0,42,0,1,45,13,32,0,54,0,30,27,0,36,7,23,47,38,0,21,20,31,6,37,39,
    47,0,30,0,25,50,55,43,0,10,34,0,0,0,0,0,0,0,0,4,6,0,0,0,60,19,44,0,0,53,0,0,56,0,0,52,32,20,0,0,24,58,0,29,11,18,48,38,0,1,0,0,5,28,51,16,0,0,42,21,
    37,53,42,0,0,15,20,4,0,0,45,0,7,0,0,5,0,56,51,13,26,39,57,46,0,14,12,27,17,49,55,8,0,0,30,19,0,0,35,10,3,36,34,23,52,11,0,0,0,25,0,21,0,24,0,18,41,0,16,50,
    0,22,23,39,10,0,0,0,57,7,44,5,0,30,18,4,12,50,15,0,0,0,29,59,2,16,41,46,60,0,43,13,48,38,19,0,8,42,0,52,54,0,0,55,9,47,0,6,28,24,26,32,11,34,37,0,1,53,36,45,
    55,37,20,0,6,58,28,0,0,27,50,41,31,0,24,47,45,0,0,43,51,0,46,0,25,60,39,0,0,42,32,34,12,16,0,57,0,0,56,0,5,52,0,35,38,48,10,0,0,0,1,3,15,0,40,44,0,0,0,0,
    29,0,0,18,0,0,30,44,0,0,0,48,0,19,52,0,0,3,10,0,0,47,0,49,58,0,0,0,11,0,20,28,33,12,0,53,0,0,38,6,14,9,40,0,0,0,45,8,57,13,0,34,2,1,27,35,24,5,32,0,
    56,47,0,0,32,52,0,0,0,38,29,18,16,0,41,54,46,0,0,0,13,6,0,12,0,0,0,0,58,0,31,4,0,60,28,33,0,44,37,42,39,11,0,0,0,25,2,55,0,19,0,51,0,35,0,23,0,10,45,27
  };

  const int d60_1620[] = {
    // Size: 60 x 60
    60,
    // Pre-assigned fields
    0,14,0,12,0,25,0,0,48,6,57,3,41,0,29,40,0,0,0,0,50,21,32,0,38,49,13,56,0,0,0,58,45,59,0,0,0,0,0,0,47,0,0,0,18,19,39,46,10,55,42,17,0,20,11,4,0,23,0,60,
    0,31,0,9,39,8,12,0,0,0,25,55,38,59,0,0,4,0,57,0,30,2,48,0,0,7,49,0,0,60,0,0,11,0,5,29,0,0,19,0,36,0,27,0,58,0,35,33,0,20,47,41,40,23,44,0,21,0,0,24,
    59,45,18,0,0,11,53,0,0,35,26,0,8,0,3,49,17,57,0,48,7,19,31,0,0,0,0,0,38,36,0,0,10,0,29,22,60,25,0,0,0,27,0,0,47,0,32,14,0,16,12,6,0,0,0,55,0,20,0,0,
    1,0,0,46,20,17,41,49,0,26,0,21,0,0,0,19,23,47,0,0,0,36,38,0,0,0,28,42,0,32,48,0,0,43,8,55,0,0,15,37,0,16,0,0,29,0,51,0,0,52,0,10,0,0,0,11,0,60,57,0,
    0,4,10,59,0,57,0,0,60,34,0,0,5,25,38,0,33,0,0,29,53,0,0,42,36,0,27,0,41,0,51,50,0,48,3,13,24,6,22,28,40,18,20,1,19,8,23,16,14,26,17,39,12,47,0,54,49,30,52,0,
    0,10,49,0,0,59,0,48,6,1,0,17,57,0,31,3,51,0,56,0,4,53,36,7,8,26,0,29,0,5,28,0,23,21,20,38,50,19,43,0,30,0,0,0,12,46,0,41,0,22,0,60,0,40,0,39,44,0,35,34,
    0,0,16,19,0,37,8,0,35,45,3,30,0,0,0,0,0,0,0,34,0,25,0,57,22,0,58,0,0,38,0,0,52,4,0,2,0,15,0,0,0,49,0,0,10,33,29,47,24,53,0,0,55,0,0,0,0,48,28,56,
    0,24,1,0,18,38,0,0,0,13,15,22,43,36,0,12,52,16,30,0,21,44,8,50,0,0,10,26,20,2,59,0,0,0,0,0,28,55,5,25,34,29,14,46,53,0,0,0,0,3,60,37,0,17,58,31,4,0,39,40,
    51,60,9,3,0,53,32,0,0,54,22,52,37,24,5,0,0,40,55,0,0,0,58,0,56,0,0,0,57,0,30,47,0,20,38,0,0,48,31,4,0,19,0,0,0,0,21,49,44,18,43,0,0,39,25,0,7,15,0,17,
    60,56,37,0,0,5,0,0,40,0,0,12,0,18,43,0,15,0,45,42,34,0,39,3,53,0,57,0,0,0,0,44,31,52,0,0,0,0,0,17,32,0,24,28,26,16,0,0,58,0,20,11,0,0,0,19,50,0,38,35,
    31,55,13,0,47,29,0,0,0,0,38,0,6,35,10,0,0,0,14,46,0,0,20,30,40,0,60,0,0,9,11,0,0,0,25,45,0,0,48,0,4,0,0,22,24,0,19,0,34,0,36,43,0,51,28,50,18,26,0,0,
    0,58,0,0,0,0,24,0,0,29,39,4,60,0,0,0,6,26,8,0,37,0,42,40,0,28,43,31,21,20,45,53,0,0,0,44,0,38,0,15,56,48,19,9,0,0,22,7,0,0,34,59,0,0,0,14,30,18,0,0,
    54,59,28,31,0,0,0,21,23,16,47,0,48,0,36,7,38,0,34,0,14,0,2,53,29,0,26,35,12,13,10,49,27,0,33,43,3,8,0,0,0,0,0,19,44,5,0,24,17,39,6,0,60,0,41,0,0,50,0,52,
    38,0,0,26,0,0,44,30,8,46,7,0,13,56,35,0,28,48,11,1,0,0,45,23,32,0,0,43,0,29,0,55,0,0,0,12,40,0,52,36,53,33,15,0,0,49,50,20,47,0,4,0,0,41,16,24,0,2,19,59,
    49,52,0,45,36,18,0,0,10,9,4,0,0,53,0,33,44,27,0,17,43,40,0,0,37,0,0,0,7,57,46,0,25,0,0,6,0,32,0,39,0,3,0,13,42,31,60,21,0,58,0,16,34,59,26,2,20,0,0,0,
    26,0,40,2,11,0,0,52,17,3,0,8,36,0,32,0,0,20,0,0,29,58,0,0,0,6,0,41,47,44,7,59,1,0,0,0,0,43,54,0,0,23,0,0,56,22,0,5,0,0,0,0,14,19,49,33,46,0,60,0,
    19,0,0,56,0,2,0,46,13,17,54,0,0,0,58,59,31,0,24,0,0,20,0,11,52,45,0,0,32,0,0,15,30,42,23,50,49,1,0,5,43,12,39,0,41,0,40,51,7,27,33,4,0,0,0,6,47,0,0,53,
    0,0,0,0,21,0,59,0,19,0,32,50,20,39,0,27,24,0,12,0,0,0,5,35,0,57,0,0,0,33,56,30,3,0,51,40,38,0,0,0,0,37,43,10,1,0,25,0,0,49,8,0,46,7,48,0,54,0,14,16,
    41,27,0,1,0,0,0,58,47,48,6,0,0,0,0,0,60,46,52,0,44,35,55,0,17,0,7,53,0,11,0,37,38,8,0,34,16,0,36,0,0,0,21,25,57,4,0,19,9,28,0,0,0,50,0,0,59,32,0,33,
    0,25,47,0,53,0,0,8,52,58,0,24,14,2,0,0,0,42,0,15,23,0,0,0,0,31,0,0,0,59,0,33,40,5,44,0,0,0,0,7,48,43,0,0,16,32,41,0,0,0,29,55,35,4,45,30,38,12,0,0,
    0,0,0,5,0,9,38,0,39,20,13,0,0,0,0,0,0,34,0,0,47,23,0,0,54,35,0,40,0,0,0,0,0,0,22,11,52,4,29,51,0,42,0,0,7,53,33,12,0,0,2,0,0,0,8,58,32,0,0,28,
    42,0,57,0,0,41,54,0,18,0,0,56,26,0,19,50,0,28,59,0,45,15,12,0,16,0,0,0,0,34,22,0,0,14,0,10,5,47,33,48,7,44,0,43,27,23,0,31,55,35,0,9,37,11,36,0,0,0,0,29,
    0,0,46,0,0,54,7,0,0,53,0,0,49,0,55,0,0,58,16,0,0,45,0,37,0,41,42,9,0,0,40,31,17,35,0,0,0,22,0,0,26,20,0,56,0,57,0,0,0,11,0,0,0,27,10,0,3,29,25,0,
    5,49,0,33,0,0,0,37,0,32,27,59,34,43,4,0,0,51,42,39,11,60,41,0,55,0,0,0,0,0,0,16,0,0,54,8,0,0,20,0,2,0,36,53,6,10,57,0,46,0,0,0,0,0,0,48,0,14,44,18,
    3,1,12,32,0,36,0,0,0,0,0,33,35,54,37,0,0,0,0,0,0,27,13,47,0,0,0,28,0,18,0,57,0,0,0,0,0,26,59,44,6,0,30,52,50,24,0,0,0,40,45,48,0,0,0,7,0,55,23,46,
    50,0,0,8,57,24,0,0,49,55,0,0,22,32,13,48,0,41,27,47,25,43,0,5,0,10,51,2,26,0,44,0,39,7,52,0,45,0,16,31,0,0,0,21,0,60,30,0,20,56,0,53,23,12,1,40,0,0,3,19,
    15,50,4,0,51,0,0,34,0,28,18,0,46,58,0,0,0,17,9,0,60,0,0,0,0,0,32,0,29,27,0,56,0,1,0,14,20,59,0,0,0,0,3,0,30,21,53,0,0,45,49,0,0,0,43,5,13,19,0,0,
    57,44,54,0,19,30,0,12,36,39,58,13,0,0,1,17,22,0,48,6,0,10,0,0,9,20,59,8,25,7,0,0,0,31,45,0,34,27,0,46,0,0,0,0,4,0,14,52,18,42,0,24,0,32,60,37,55,43,5,38,
    0,40,0,0,0,1,0,0,0,0,0,6,0,0,42,39,0,38,43,0,46,8,4,15,0,21,0,0,0,48,0,0,0,28,35,17,0,0,0,0,37,57,0,59,0,0,58,2,19,60,10,0,0,44,56,9,36,0,34,0,
    0,33,0,0,0,0,48,10,43,36,0,0,19,6,0,29,47,37,0,54,0,0,0,0,0,0,23,1,2,0,0,14,0,32,0,0,41,21,0,59,16,28,38,0,0,0,0,0,0,0,27,0,0,5,0,0,0,44,30,0,
    0,0,0,20,1,0,31,11,0,30,0,0,42,29,7,0,19,52,35,49,48,0,0,6,5,0,24,0,16,0,15,21,41,0,36,0,47,56,4,26,10,0,53,14,0,0,0,39,50,0,0,0,57,0,0,0,60,22,59,0,
    0,42,39,11,0,56,1,45,0,0,30,0,54,8,0,14,55,0,0,0,15,0,0,28,0,12,16,20,13,22,26,9,0,0,0,0,0,0,53,0,27,0,10,0,46,43,0,0,4,29,0,44,0,0,23,0,35,0,0,32,
    0,19,0,21,43,7,0,38,29,12,0,54,0,0,0,0,34,55,0,0,1,0,15,0,44,0,0,0,0,23,0,27,0,0,0,30,0,60,0,13,9,35,58,0,0,14,5,0,48,0,0,57,0,0,0,0,17,0,8,0,
    9,0,27,0,0,0,17,22,16,49,20,46,0,0,0,10,0,44,0,0,0,50,35,2,51,0,18,5,0,0,0,0,57,58,0,0,0,0,0,0,0,34,28,0,45,15,8,23,33,0,11,19,43,0,32,0,0,41,0,0,
    0,0,38,22,28,34,0,0,56,44,0,26,40,0,0,0,0,0,32,25,0,42,0,45,0,0,0,0,0,0,0,0,55,30,24,0,51,0,0,0,17,0,0,0,23,35,15,0,29,0,0,5,0,0,0,43,10,49,0,2,
    0,28,59,42,0,0,37,39,1,0,0,32,0,47,0,0,0,0,26,0,36,17,10,41,20,0,0,0,0,0,0,7,24,0,15,23,0,58,0,35,57,8,5,34,0,27,0,0,0,54,30,0,0,60,55,3,0,4,0,0,
    23,36,0,0,22,3,0,53,55,0,10,51,0,33,0,15,0,35,7,2,57,38,0,34,0,43,0,12,0,45,19,0,0,0,14,0,0,0,0,30,52,26,37,17,25,6,44,0,0,0,28,0,4,0,20,32,0,0,41,0,
    52,16,21,0,3,19,4,56,0,14,0,49,53,31,0,32,0,43,47,0,0,0,0,0,0,30,0,0,55,0,0,0,0,0,0,0,0,0,10,9,0,0,7,45,22,39,59,11,0,0,0,0,0,0,46,0,27,0,6,0,
    16,0,25,55,0,0,50,0,0,42,0,0,52,37,14,38,0,12,0,8,28,0,0,48,57,0,0,0,0,0,9,24,0,17,0,36,0,30,0,0,0,0,23,58,40,0,0,35,54,0,46,0,18,22,15,59,0,0,31,0,
    53,20,0,0,0,43,57,42,22,59,23,36,9,28,30,0,48,11,0,24,0,0,0,51,39,13,0,55,44,1,0,0,54,0,0,46,0,0,17,0,15,50,26,38,0,0,0,60,40,12,25,7,0,8,0,0,0,45,4,3,
    14,26,7,58,12,20,18,24,0,43,5,42,0,0,59,28,0,0,0,0,55,0,0,19,0,0,34,17,23,39,49,40,22,3,10,54,21,0,0,38,0,0,57,44,0,29,47,48,31,46,0,2,9,6,0,15,0,36,11,51,
    33,0,8,0,41,0,0,16,42,0,0,11,44,23,28,0,30,22,0,53,27,14,0,0,49,0,35,0,0,21,25,17,0,0,0,7,0,0,0,0,0,13,31,47,0,0,18,0,0,0,50,0,0,0,0,26,43,37,54,48,
    0,46,41,43,7,0,21,0,26,5,0,58,0,44,0,0,0,32,22,0,49,0,51,0,1,56,2,52,33,47,13,0,59,6,11,18,0,34,0,3,38,25,29,30,0,40,0,50,23,37,24,35,0,31,39,53,12,28,15,42,
    0,29,5,0,46,0,11,55,0,18,21,16,32,0,45,35,1,60,2,0,58,3,37,44,28,0,0,0,0,52,42,0,0,23,9,0,14,0,0,49,0,0,47,24,43,36,54,27,30,0,38,0,59,0,0,8,56,0,20,0,
    0,0,56,57,24,0,0,40,37,0,0,2,0,0,54,0,49,29,13,0,31,48,52,0,27,0,19,0,35,0,6,26,0,0,17,0,0,0,0,33,0,60,12,51,0,34,0,0,0,8,0,30,0,45,0,22,0,46,21,5,
    40,0,58,15,0,46,0,29,0,0,0,44,59,41,0,53,11,5,36,0,19,57,54,32,13,0,1,30,50,31,0,45,16,33,0,0,27,10,0,14,0,0,0,12,0,0,0,0,0,4,22,28,0,0,0,17,0,0,49,47,
    0,3,52,35,0,0,0,0,11,50,0,31,47,0,56,36,0,49,4,0,54,0,21,0,0,9,6,22,48,0,39,0,14,0,57,26,44,5,23,0,46,59,0,0,0,0,28,25,0,0,41,20,16,29,12,0,37,33,55,0,
    0,0,2,51,0,33,0,0,0,40,49,0,28,0,22,1,0,8,44,0,0,0,43,20,0,53,30,0,6,0,3,48,0,0,0,32,7,41,0,0,0,10,0,0,31,0,0,54,59,0,0,0,0,36,24,0,11,0,0,37,
    39,32,0,0,49,0,0,0,0,0,28,0,15,46,0,23,27,31,0,14,0,30,0,0,0,24,0,45,1,54,5,42,34,0,50,0,0,0,0,20,8,51,0,0,59,0,0,22,0,38,0,33,0,21,0,0,0,16,9,0,
    0,41,24,0,37,0,45,0,0,0,0,0,39,5,0,55,0,7,50,21,20,51,30,16,0,0,0,0,0,17,47,38,0,36,0,58,19,0,42,22,0,0,0,8,0,54,3,0,52,0,9,0,0,0,0,29,0,0,0,49,
    0,18,11,6,0,0,0,15,32,0,53,0,1,42,51,0,0,0,20,59,10,0,50,0,0,0,21,24,31,0,14,0,0,41,46,0,0,12,0,29,49,30,22,0,0,0,17,3,0,0,0,13,0,0,57,60,45,8,0,0,
    17,34,48,0,0,16,33,3,54,0,51,10,27,4,46,13,40,0,0,0,0,29,0,0,0,52,0,0,56,0,0,0,0,0,0,41,22,50,47,0,11,31,32,0,0,0,20,0,0,0,5,0,24,0,0,0,6,38,7,36,
    0,0,0,0,30,0,13,36,0,21,56,35,25,0,0,0,59,18,19,0,16,55,0,17,0,42,5,0,52,14,38,0,0,37,12,60,0,11,2,0,44,0,50,41,0,7,34,0,6,48,57,0,8,33,9,0,0,0,47,0,
    0,2,29,0,0,0,0,50,0,22,16,40,0,0,8,44,35,0,25,0,33,0,24,18,46,0,4,48,43,58,60,0,0,49,0,0,0,0,0,0,13,0,0,54,0,30,0,15,36,7,0,0,38,0,0,20,31,6,37,39,
    47,35,30,0,0,50,55,0,2,0,34,0,33,12,0,0,37,0,0,0,6,13,14,54,60,19,44,0,36,53,8,0,56,0,40,52,0,20,0,0,0,0,0,29,11,18,0,38,0,1,0,49,5,28,51,0,15,17,42,21,
    37,53,42,28,44,15,0,4,58,33,0,60,0,0,0,0,0,0,0,13,26,0,57,0,0,0,0,0,17,49,0,8,47,22,30,19,0,29,35,0,3,36,0,0,52,0,1,43,38,0,54,21,32,0,0,0,0,0,16,0,
    58,22,23,39,10,0,0,25,0,0,44,5,17,0,0,4,12,0,0,0,0,0,29,59,2,16,0,46,60,0,0,0,48,0,0,0,8,0,0,52,0,14,33,0,0,47,31,0,0,0,26,32,11,34,0,21,0,53,0,45,
    55,0,0,30,0,0,0,0,53,27,50,41,31,17,24,0,45,0,0,0,51,9,46,0,25,0,0,36,14,0,0,0,12,16,0,57,13,33,0,11,5,0,0,35,38,0,10,29,2,21,1,3,0,0,40,0,0,0,0,0,
    29,0,26,0,0,0,0,0,41,31,0,48,4,0,52,42,0,3,0,0,0,47,0,49,58,0,37,0,11,0,20,28,33,12,0,53,0,17,0,0,0,9,40,16,60,55,45,8,57,13,51,0,2,1,27,0,0,5,0,7,
    0,47,0,48,0,0,0,0,9,0,0,0,16,34,0,54,0,0,53,0,0,6,22,12,0,0,36,57,0,0,0,0,26,60,28,33,0,0,0,42,0,11,49,0,15,0,0,0,1,19,0,0,21,35,0,23,40,0,0,27
  };

  const int d70_2450[] = {
    // Size: 70 x 70
    70,
    // Pre-assigned fields
    0,50,0,38,15,28,51,42,19,52,62,61,0,0,1,0,0,0,64,0,18,10,59,13,0,0,26,0,70,0,0,66,39,9,54,69,0,44,0,55,32,30,58,0,48,0,12,0,22,0,3,0,0,65,0,68,14,0,67,45,0,53,4,8,0,0,0,25,0,0,
    0,40,31,11,66,65,42,48,64,0,3,0,0,0,0,26,0,0,0,0,0,5,9,0,69,8,6,0,19,0,0,46,0,34,10,18,0,0,47,36,22,0,0,0,58,0,70,15,0,30,23,0,0,55,41,63,68,0,57,0,32,0,45,2,0,0,0,59,44,33,
    8,49,0,0,0,0,60,33,0,0,67,0,47,0,0,52,0,0,3,0,41,11,0,22,48,69,0,6,15,10,0,0,65,0,42,0,0,0,0,0,0,0,0,62,68,0,0,43,0,66,18,0,17,23,0,54,0,30,0,0,28,0,7,36,0,0,32,0,29,1,
    0,0,0,0,33,4,2,0,0,7,61,0,0,0,0,5,0,13,32,0,34,0,0,0,0,1,0,0,0,20,65,0,0,0,0,10,60,48,0,67,0,44,54,11,0,69,24,50,0,0,47,0,70,0,12,15,37,0,39,0,43,0,0,27,40,0,0,0,23,62,
    16,0,0,0,0,8,0,0,2,18,5,0,48,0,0,49,15,40,0,66,0,0,23,24,59,44,50,0,32,62,0,0,22,0,0,0,0,0,9,0,0,46,0,0,0,0,29,64,6,33,4,10,0,25,0,51,69,7,58,14,0,60,0,34,0,0,0,0,0,53,
    70,0,45,65,34,0,0,0,7,66,59,18,33,0,23,19,0,36,21,63,0,17,0,0,25,0,0,0,0,41,52,24,0,0,31,3,12,9,0,44,38,0,62,0,5,13,64,39,10,0,0,0,51,56,0,60,0,0,0,35,61,22,0,57,68,1,2,0,0,14,
    67,4,7,0,0,0,65,0,0,24,0,64,19,0,0,15,38,0,0,0,0,20,8,0,62,13,0,0,49,0,0,0,0,42,22,0,0,0,10,0,0,27,0,53,0,0,16,0,0,63,54,0,11,66,0,47,52,0,59,0,0,21,0,14,30,0,0,0,1,50,
    0,32,33,4,16,39,69,44,0,27,57,15,29,0,0,0,0,30,0,0,0,0,0,7,0,17,0,59,63,31,0,8,26,0,0,0,0,0,43,0,35,0,70,41,0,55,50,2,0,0,58,5,47,0,0,46,13,21,0,49,0,48,12,51,0,0,68,0,0,0,
    66,0,17,0,0,0,0,3,61,49,4,0,0,45,36,28,0,0,0,0,0,54,21,0,11,0,64,0,25,0,38,44,14,0,0,0,37,41,0,0,0,67,0,0,29,58,0,27,1,0,0,0,20,9,0,30,0,0,40,0,0,23,0,0,16,62,53,0,35,34,
    19,26,6,0,46,1,28,43,0,0,0,0,24,52,60,0,48,59,0,38,0,13,0,0,0,0,8,67,17,33,47,51,27,31,0,0,0,0,4,0,55,2,45,0,0,50,62,69,14,0,0,0,49,0,40,39,0,65,37,44,0,32,3,15,58,56,10,0,63,9,
    0,6,0,0,0,23,0,0,0,0,7,0,0,58,0,0,52,25,48,32,8,14,0,0,0,0,31,64,43,67,0,36,5,10,0,0,17,28,42,62,0,41,0,66,0,0,38,0,49,0,30,47,0,0,0,19,21,0,15,24,69,0,0,13,0,0,51,29,22,12,
    0,0,0,59,31,0,18,63,0,0,0,58,0,56,47,34,51,0,0,54,0,0,3,0,5,0,0,0,0,0,0,57,13,29,19,0,45,55,20,0,62,0,0,24,66,0,0,70,12,0,0,0,0,60,0,0,0,0,43,16,0,11,0,50,0,44,0,0,8,0,
    26,0,9,0,50,32,37,0,0,0,0,0,31,57,0,0,0,2,56,47,0,0,48,61,0,0,0,63,0,49,21,0,0,0,20,6,0,8,38,70,4,0,19,0,1,0,22,52,5,0,0,17,0,40,0,27,45,15,0,29,60,65,39,10,0,0,0,0,0,59,
    0,53,0,0,38,0,0,65,50,3,43,55,0,41,34,58,0,67,0,21,0,0,39,18,12,0,0,0,2,28,30,63,25,0,0,57,0,0,0,0,0,9,6,0,33,0,0,13,0,22,1,0,45,49,0,14,0,5,0,8,0,0,0,0,19,0,0,62,37,52,
    0,0,13,26,0,15,0,0,28,0,23,0,69,31,0,1,36,0,0,10,62,0,0,0,44,40,0,16,7,0,0,11,0,0,48,61,0,63,0,27,12,0,4,0,0,0,25,0,0,0,45,0,14,0,19,0,0,0,0,0,57,8,37,53,39,2,60,0,0,0,
    0,0,69,10,0,0,15,57,0,58,32,21,0,11,67,16,0,0,70,0,0,0,0,43,38,0,65,0,37,0,0,0,66,35,0,0,47,0,12,45,39,0,56,0,0,18,60,49,61,0,0,0,0,0,0,20,0,0,52,0,0,26,62,0,63,0,22,28,51,0,
    51,0,19,0,25,70,33,49,0,56,69,0,0,0,0,0,62,57,0,24,65,12,0,0,0,2,23,0,14,36,5,53,59,16,55,0,26,54,48,6,0,45,15,47,10,41,0,0,50,27,38,0,31,0,22,11,46,0,0,28,34,35,40,66,0,0,64,0,42,61,
    0,2,35,46,0,0,0,0,0,0,25,14,0,66,0,0,0,0,0,12,0,0,51,0,0,0,61,29,0,32,0,47,0,7,45,23,64,17,0,21,28,0,16,0,0,48,18,0,0,0,65,0,0,0,24,0,0,63,3,53,0,0,0,41,0,0,52,70,0,0,
    2,0,5,0,27,61,0,0,3,0,30,32,50,0,45,62,69,24,41,15,0,18,17,48,52,0,37,0,66,42,0,49,29,47,43,0,0,68,1,0,0,40,65,0,0,0,8,0,0,0,63,21,7,59,0,0,11,58,6,0,0,38,0,20,0,0,4,0,33,13,
    15,41,0,0,47,52,59,1,21,13,34,57,0,25,63,0,22,0,45,26,0,0,0,62,36,29,0,14,55,0,3,42,0,8,24,0,23,0,58,20,10,0,0,4,0,30,11,0,38,19,2,0,0,32,0,66,0,50,0,0,56,17,16,35,0,53,0,33,60,0,
    7,0,0,21,22,24,0,0,58,65,0,0,0,0,0,0,0,0,16,0,0,66,0,68,51,0,0,0,0,0,0,50,0,0,0,0,29,0,0,12,0,57,0,0,45,0,0,33,0,40,13,3,0,47,0,62,0,0,0,60,37,0,0,0,41,23,43,0,69,0,
    0,52,0,0,0,63,19,58,0,9,68,0,0,2,0,0,53,0,0,0,0,36,40,0,15,18,0,0,54,0,29,0,62,0,21,70,50,5,0,0,43,0,0,0,61,0,0,0,0,60,0,1,69,13,3,0,24,0,0,0,0,25,56,33,0,0,0,0,28,46,
    17,19,0,2,0,64,41,61,0,0,0,0,43,18,0,59,0,47,11,0,0,39,42,9,0,32,49,0,0,0,0,0,33,0,0,38,0,60,0,0,0,0,0,0,44,7,0,0,66,56,0,51,0,0,1,0,0,0,22,0,25,0,27,0,4,0,0,12,0,30,
    47,0,37,0,17,22,0,19,0,10,0,20,44,14,0,40,0,5,0,30,0,69,0,0,67,0,0,70,0,0,0,43,3,0,0,0,1,0,31,68,36,0,0,0,12,60,0,0,11,15,0,0,0,0,42,0,25,0,50,0,49,0,0,62,0,26,63,41,0,21,
    0,0,0,62,70,0,38,8,0,14,0,7,15,0,0,66,0,37,55,67,36,0,0,30,22,0,39,56,0,0,0,64,0,23,0,2,0,20,0,40,31,0,60,59,0,0,0,54,29,16,44,24,0,0,0,0,0,11,17,0,47,3,0,0,0,0,34,10,0,0,
    0,0,49,0,13,12,67,6,0,43,22,0,0,39,0,0,0,0,2,0,0,0,26,0,28,24,0,0,0,57,0,18,0,3,0,0,0,38,0,0,9,0,33,0,0,53,32,41,0,14,15,0,59,30,61,0,62,0,0,4,23,0,0,0,64,0,0,69,0,17,
    52,34,4,0,0,2,47,14,0,0,13,0,16,0,58,60,57,0,0,48,25,42,0,0,66,49,29,0,0,17,11,61,0,0,30,0,10,0,53,0,0,54,0,0,0,0,37,0,0,0,36,0,9,0,0,22,59,33,46,0,0,0,0,0,5,0,0,32,0,23,
    63,0,0,48,51,0,6,64,0,0,0,24,0,27,0,54,39,0,0,42,0,8,0,29,0,16,36,0,52,0,49,0,0,0,40,0,0,0,0,0,15,43,41,45,0,56,65,10,0,17,19,18,0,0,0,0,61,0,4,0,5,7,0,37,0,67,0,20,34,26,
    3,0,52,44,42,60,0,0,46,67,0,0,49,48,0,0,0,12,35,0,55,0,0,1,0,7,19,33,0,0,0,17,47,39,0,16,9,51,15,0,65,69,36,18,0,64,0,6,21,10,0,57,63,0,45,53,28,31,0,38,0,66,0,29,70,27,0,0,0,0,
    62,0,22,0,0,0,9,0,66,0,50,33,0,0,0,0,0,65,0,0,19,63,0,3,0,0,52,0,1,0,0,55,43,0,23,59,18,53,0,0,54,0,0,39,8,16,49,4,0,0,0,0,67,26,32,61,36,45,0,2,11,41,60,0,69,70,0,40,25,37,
    45,0,0,61,5,59,17,66,37,0,0,0,0,0,0,31,0,6,47,0,0,0,0,0,0,0,0,0,27,0,40,0,4,53,0,32,0,18,0,15,0,0,57,0,65,36,0,28,41,62,0,0,0,2,29,0,26,44,0,0,0,0,0,0,0,10,0,0,0,60,
    48,58,0,0,61,45,66,31,0,0,0,6,55,59,0,25,0,0,62,0,0,0,0,0,0,0,40,0,47,0,41,39,36,26,0,0,5,52,0,0,0,0,10,3,53,28,0,0,63,0,14,44,33,0,0,0,0,24,0,0,0,0,42,68,54,0,0,0,0,0,
    0,13,0,0,0,0,62,45,0,34,0,1,3,0,0,0,47,0,0,9,53,35,0,0,0,26,46,0,64,0,54,58,40,0,0,0,0,59,0,52,6,4,39,0,0,0,68,0,0,29,0,11,0,43,0,0,0,32,44,0,42,0,38,16,0,69,48,5,0,0,
    37,0,0,0,0,20,70,52,8,0,0,0,0,24,0,21,0,48,38,0,2,43,15,33,0,0,55,0,35,61,56,6,0,0,51,39,28,0,60,10,0,0,0,68,0,0,0,1,58,36,32,49,0,5,16,0,47,66,0,67,0,0,0,0,0,0,18,0,57,0,
    0,0,0,0,67,0,0,11,0,0,37,0,0,35,39,56,31,66,0,14,0,45,0,0,0,0,0,0,0,0,17,0,0,21,0,0,0,40,0,0,1,48,0,12,0,44,27,53,0,0,62,0,0,16,0,0,58,68,0,6,0,43,0,0,0,0,0,0,49,57,
    0,0,0,0,0,0,0,55,0,30,2,48,0,70,0,11,33,0,51,46,0,37,0,0,53,0,0,26,0,0,69,0,0,65,61,56,44,0,52,0,0,17,20,21,0,25,0,5,0,0,40,0,0,0,34,13,50,0,18,0,0,0,0,45,12,0,16,64,39,0,
    0,0,0,15,0,0,27,0,0,70,51,0,0,0,18,0,0,0,0,16,28,0,25,0,0,38,0,61,58,0,0,0,0,69,44,0,0,0,36,65,0,0,0,0,0,0,0,29,0,0,33,0,0,57,0,0,0,0,1,12,0,52,21,47,31,3,0,13,0,45,
    0,0,0,0,0,56,45,10,69,29,0,0,57,0,0,0,0,0,0,35,0,0,0,0,0,5,0,49,23,0,0,67,17,0,50,0,34,30,0,0,37,0,0,0,27,32,0,0,51,12,0,14,68,28,0,0,0,0,0,47,44,54,15,52,20,24,55,0,0,0,
    0,0,0,13,0,0,0,0,0,53,0,16,0,17,0,14,0,0,0,0,0,55,54,19,70,0,0,0,0,0,0,0,0,44,38,0,0,0,25,64,0,15,3,5,52,0,0,56,0,42,26,63,22,0,0,50,0,2,9,0,0,49,0,24,0,0,58,27,48,20,
    0,0,18,0,0,0,29,0,63,0,0,0,0,10,0,47,25,0,0,0,51,26,0,0,0,0,28,0,0,15,0,0,0,0,0,68,0,0,5,0,0,0,49,0,0,1,0,42,70,0,0,23,37,0,31,0,20,0,0,0,0,67,0,0,36,0,19,16,0,0,
    59,64,16,0,0,0,1,4,10,0,52,0,18,0,22,13,55,28,0,0,32,0,0,0,23,56,47,0,0,3,0,33,2,0,0,48,41,21,24,7,5,66,0,57,0,0,39,61,60,58,69,0,0,6,27,67,63,0,0,0,65,0,0,43,0,0,45,54,0,35,
    0,39,26,0,45,38,0,30,0,69,54,0,0,0,21,0,0,0,29,40,3,0,66,0,42,0,0,43,6,0,20,5,34,0,12,41,65,0,44,23,61,0,50,28,64,0,0,0,46,0,37,0,57,0,0,4,0,56,35,18,58,15,0,67,49,0,59,53,0,0,
    43,0,0,30,0,0,23,9,38,60,31,68,0,0,0,22,1,17,5,44,0,32,57,0,0,0,0,54,11,0,51,0,12,0,0,0,0,0,0,0,27,70,35,25,0,0,63,58,0,59,0,0,0,0,66,69,4,0,64,10,0,0,34,0,18,0,61,0,0,39,
    0,0,29,37,56,58,0,7,26,11,0,0,42,65,12,35,18,0,40,0,50,0,0,0,61,46,0,32,0,1,2,38,24,0,28,0,0,19,22,0,0,0,8,0,13,63,0,17,0,0,64,0,0,69,39,9,54,14,0,20,70,0,0,21,47,0,41,60,0,0,
    0,30,3,7,0,0,35,0,47,0,0,36,0,21,70,0,0,52,4,53,0,64,16,0,0,0,59,11,10,9,12,31,8,0,34,0,20,27,0,0,0,0,0,65,54,61,42,0,0,25,39,0,41,0,0,56,0,67,69,0,0,0,0,28,14,22,0,6,68,0,
    0,54,0,0,30,0,25,0,0,5,10,13,51,28,0,50,49,0,14,0,35,29,43,0,0,33,0,3,60,44,0,0,0,0,0,58,0,32,23,63,0,16,0,0,0,0,0,0,36,55,9,26,56,52,70,0,38,0,12,0,67,0,0,61,0,0,0,0,0,0,
    42,27,68,33,57,0,0,0,0,0,0,0,0,7,44,38,0,29,0,0,1,0,61,0,24,0,0,0,53,0,28,0,19,0,37,55,0,0,51,0,30,36,0,2,0,70,0,11,0,0,49,0,46,31,0,65,23,0,45,62,8,16,25,0,0,0,54,15,0,10,
    0,0,12,22,2,35,68,69,24,42,46,59,14,0,0,0,32,45,0,0,0,0,50,66,40,39,3,47,34,0,4,21,57,38,65,51,8,25,55,0,63,56,44,0,43,0,61,62,0,0,0,0,19,17,26,0,41,0,0,5,9,0,48,0,10,0,49,58,67,0,
    0,70,0,16,28,0,0,62,5,0,0,0,0,1,0,4,54,26,0,36,0,47,0,0,0,0,0,69,45,0,9,0,38,20,0,11,0,29,59,17,48,0,40,0,0,6,58,14,0,0,61,0,27,0,33,23,0,25,0,37,39,0,32,49,0,0,67,0,0,0,
    0,0,0,56,0,17,10,0,22,0,0,39,0,0,0,43,0,33,0,70,21,0,49,51,0,36,0,0,41,26,0,0,53,60,35,30,0,0,63,0,0,38,0,0,0,0,0,0,0,8,55,9,3,0,0,0,0,47,0,0,0,0,0,0,0,0,25,0,0,24,
    0,68,0,3,0,0,11,29,0,41,49,0,0,0,15,69,35,53,8,0,9,0,19,0,0,0,56,13,12,0,62,0,23,0,0,34,0,33,0,0,0,42,0,0,0,54,67,59,7,0,0,0,40,0,0,0,65,0,47,63,0,58,0,0,57,66,14,39,0,0,
    13,0,53,18,0,0,39,0,0,0,0,0,0,0,5,63,60,41,0,0,46,0,0,59,0,19,0,4,0,14,33,62,0,0,0,0,0,0,0,0,44,49,43,0,0,40,0,24,65,48,11,58,0,0,30,64,56,0,70,21,0,28,51,0,0,35,0,0,0,16,
    58,8,0,0,0,7,48,26,42,0,0,53,13,37,0,29,0,54,27,0,15,57,0,0,9,0,0,0,3,0,0,0,0,51,0,0,0,0,0,0,24,0,2,0,63,0,0,0,0,18,0,0,0,22,0,31,0,28,38,0,40,0,0,0,52,65,0,30,0,0,
    0,0,62,0,21,33,0,41,49,25,66,11,0,67,48,24,0,0,36,0,61,16,6,15,68,20,0,0,57,0,19,0,50,0,0,0,46,22,8,0,69,0,37,40,0,0,34,0,42,0,0,55,0,10,0,7,0,0,28,32,0,0,30,54,0,0,0,0,0,0,
    0,0,0,0,0,0,12,0,70,1,44,38,0,20,6,17,0,64,0,39,0,4,52,0,3,61,0,42,0,55,31,69,0,14,0,49,33,0,56,13,60,53,0,26,2,59,47,0,57,9,0,0,0,0,15,21,10,19,0,48,0,0,36,7,0,0,0,0,0,8,
    0,0,0,63,0,0,30,0,0,0,58,0,0,61,35,6,8,10,53,51,0,0,0,44,0,65,38,0,0,0,0,0,16,0,0,0,2,0,11,0,0,0,66,0,0,0,0,22,0,0,24,68,0,1,5,0,0,0,54,55,0,0,0,70,0,0,3,0,0,0,
    25,0,0,8,0,44,14,27,54,0,0,26,45,0,55,3,66,0,0,43,0,0,0,0,37,0,0,12,30,0,0,0,0,1,0,0,15,0,0,0,29,64,0,0,0,0,5,18,48,0,51,41,0,0,0,0,7,0,63,69,0,0,0,4,0,0,62,0,40,0,
    0,0,47,0,23,27,0,0,0,40,0,0,34,32,0,0,0,0,25,0,6,0,0,0,64,54,63,0,21,0,45,22,0,0,5,0,4,0,46,0,0,24,13,0,15,51,0,0,0,20,0,0,8,0,35,57,29,36,2,52,0,0,0,0,0,49,70,0,0,0,
    38,12,59,36,53,41,0,28,0,64,0,35,30,0,0,0,0,0,42,0,47,51,4,63,0,0,33,25,20,39,55,0,7,0,0,22,0,0,0,60,21,26,48,9,0,29,0,0,0,32,0,61,2,34,0,0,0,23,10,0,0,57,0,0,43,0,0,0,11,0,
    0,0,32,68,0,0,20,0,56,0,70,54,66,33,19,23,0,0,0,0,0,0,0,35,8,0,45,0,28,0,6,26,61,0,46,0,22,47,17,49,0,10,67,31,4,0,0,55,13,0,0,0,0,0,0,34,2,0,0,64,62,59,43,39,0,0,12,0,15,7,
    21,0,0,47,0,18,40,24,15,55,8,0,65,0,0,44,42,58,0,0,0,33,28,0,17,10,32,23,0,11,0,48,0,0,4,0,0,50,13,0,0,1,29,0,31,22,0,25,54,0,67,0,39,0,0,0,0,52,0,30,0,12,59,0,61,19,35,14,0,0,
    9,22,44,51,0,55,0,0,29,20,47,52,0,0,0,30,0,0,0,0,0,0,0,0,50,60,34,41,13,19,0,10,31,0,59,0,40,0,0,0,53,39,11,64,0,15,0,48,2,21,0,45,0,70,37,0,32,6,0,0,18,0,5,0,0,0,0,36,0,66,
    10,0,63,0,0,49,53,0,68,0,55,0,58,0,0,0,59,0,17,0,33,21,0,0,0,0,0,0,51,0,0,0,56,46,0,0,0,0,0,0,18,28,0,29,7,0,45,23,24,52,0,0,0,0,0,26,12,0,14,0,48,0,0,64,32,30,65,57,19,43,
    31,0,0,17,41,36,0,60,0,33,0,29,62,0,0,12,0,0,0,7,54,0,44,0,0,0,0,0,0,40,42,0,52,43,70,14,27,0,45,0,66,0,0,46,0,9,35,47,26,0,68,53,25,18,21,32,30,0,0,0,55,0,0,0,67,11,0,49,0,15,
    30,0,14,52,36,0,57,0,0,0,0,67,28,6,46,32,70,0,0,0,0,38,0,45,0,3,0,0,18,64,23,0,51,17,0,43,0,0,0,31,7,33,27,15,0,37,48,65,62,0,0,12,0,0,25,0,44,0,55,0,20,0,0,0,66,0,0,0,0,0,
    0,24,0,20,0,0,32,67,0,45,19,12,0,43,0,0,0,9,60,0,0,27,0,46,0,0,0,0,0,70,48,0,0,22,41,0,0,0,28,0,0,0,47,0,0,0,4,0,25,0,10,65,18,0,0,0,0,0,68,61,0,37,29,0,0,0,57,8,0,0,
    0,5,0,39,11,14,0,0,25,17,0,4,63,0,27,0,0,62,12,0,0,70,60,64,2,0,0,8,24,58,1,0,28,0,0,47,69,0,65,0,0,0,38,0,0,0,44,0,3,45,52,0,0,46,53,48,51,0,0,9,0,0,61,0,26,59,0,0,0,0,
    0,0,41,25,68,13,63,20,33,0,21,37,0,3,17,0,12,0,59,55,14,23,35,0,47,48,2,0,26,0,8,28,0,5,67,0,0,65,0,0,0,0,0,61,0,0,0,30,9,54,0,60,24,0,0,0,31,22,66,34,36,39,0,0,0,57,40,43,0,18,
    0,21,0,57,0,48,0,0,18,12,0,0,0,68,0,0,0,70,1,0,60,0,47,39,45,62,35,65,16,0,0,20,58,25,69,67,0,0,26,29,49,0,0,0,0,0,0,0,4,28,43,0,0,0,0,0,0,34,13,0,0,0,41,0,0,0,56,0,10,5,
    0,66,0,0,35,10,46,53,0,2,33,0,17,19,0,0,26,7,50,0,11,31,13,65,27,51,0,24,0,52,22,9,42,0,0,0,0,0,0,8,0,0,23,0,0,4,0,0,69,37,21,54,5,48,43,0,0,0,0,0,0,55,1,0,34,20,44,0,62,0
  };

  const int d70_2940[] = {
    // Size: 70 x 70
    70,
    // Pre-assigned fields
    0,0,40,38,0,0,51,0,0,0,0,61,0,0,0,0,0,0,0,33,18,0,59,0,0,0,0,2,70,0,0,0,0,0,0,69,57,44,0,55,0,0,0,0,0,0,0,34,22,0,0,31,6,0,63,0,0,0,0,0,46,0,0,0,0,0,47,25,20,0,
    56,40,31,11,66,0,0,0,64,0,0,0,0,0,52,26,0,0,24,0,4,0,0,0,0,8,0,0,19,43,13,0,0,0,0,0,0,0,47,36,22,0,25,16,0,0,0,15,0,30,0,0,0,55,41,63,0,0,57,17,0,0,0,0,0,28,27,0,44,0,
    0,0,0,9,12,26,0,0,0,0,0,31,47,34,0,52,0,0,3,20,0,11,0,0,48,0,0,6,15,10,58,0,0,50,0,0,13,4,2,19,56,0,0,0,0,0,0,0,45,0,0,40,0,23,0,0,0,0,27,25,28,24,7,0,59,0,0,0,0,0,
    0,0,30,35,33,4,0,0,52,0,0,0,6,0,0,0,0,13,0,56,0,0,41,0,0,1,0,17,38,0,0,45,0,49,57,0,60,0,0,0,0,44,54,0,0,69,24,0,0,0,47,42,70,0,12,15,0,0,39,59,43,0,0,0,0,0,26,0,0,62,
    16,65,11,0,55,8,0,0,2,0,0,70,48,0,0,0,0,40,0,0,38,0,0,0,59,0,50,0,0,62,0,19,0,0,52,0,0,0,9,0,0,0,0,56,0,0,29,64,0,0,0,0,0,25,0,0,0,0,58,0,35,60,0,0,0,0,17,21,0,0,
    70,0,0,0,0,0,58,37,7,66,59,0,0,0,0,0,0,0,21,63,0,17,0,6,0,27,16,20,0,0,52,24,0,0,31,0,0,0,40,44,0,29,0,43,0,0,64,39,0,0,0,0,0,0,0,0,8,26,49,0,61,22,46,0,68,1,2,55,47,0,
    0,0,7,0,29,0,0,70,0,24,0,64,19,0,31,0,38,0,34,0,0,0,0,0,0,13,0,0,49,0,18,0,0,0,0,0,35,36,0,0,0,0,0,0,32,0,0,0,0,0,54,37,11,0,0,47,52,0,59,26,68,0,0,0,0,0,0,9,0,0,
    34,0,0,0,0,0,69,44,53,27,0,0,0,38,65,61,28,0,0,0,45,62,24,0,10,0,66,59,0,31,0,8,0,0,3,0,0,0,43,1,0,0,0,41,0,0,0,2,0,0,0,5,47,0,0,0,13,0,0,0,0,0,0,0,9,0,0,0,56,0,
    0,48,0,0,0,0,26,0,0,0,4,0,0,0,0,28,0,0,52,50,0,54,21,0,0,15,0,10,25,0,38,0,14,0,0,0,37,41,0,51,0,67,18,0,29,58,0,0,0,0,0,0,0,0,0,0,0,42,40,0,0,23,70,0,16,62,53,0,35,0,
    19,26,0,5,0,0,28,0,0,16,29,66,24,52,0,36,48,0,7,38,70,0,18,57,34,12,0,0,0,33,0,0,0,0,0,0,30,42,0,0,0,0,0,20,0,0,0,69,14,0,0,0,0,68,0,0,0,0,37,0,0,0,0,0,0,0,0,0,63,9,
    0,0,0,0,18,0,0,0,44,0,0,56,0,0,0,0,52,25,0,0,8,14,0,0,1,0,31,64,43,67,0,0,5,0,0,0,17,28,0,62,50,0,0,0,0,0,38,0,49,0,0,0,61,20,0,0,0,0,15,24,0,0,0,0,0,39,0,29,22,12,
    0,0,61,0,31,0,18,63,0,38,0,0,53,0,47,0,51,0,30,0,0,0,3,0,5,0,0,0,67,0,0,0,13,0,19,0,0,0,20,0,62,0,0,24,0,0,0,0,12,23,35,0,0,60,0,0,9,0,0,0,0,0,52,0,0,44,1,37,0,0,
    26,0,9,41,0,32,0,0,0,0,0,0,31,0,0,33,64,0,0,0,0,0,0,61,14,0,0,0,0,49,0,0,0,11,20,6,0,0,0,70,4,0,0,30,0,23,0,0,5,0,42,17,53,0,13,0,45,15,0,29,0,65,0,0,0,54,0,3,0,0,
    29,53,0,0,38,0,16,65,0,3,0,55,0,0,34,0,0,67,0,21,17,0,0,0,0,0,0,46,0,28,30,63,0,48,0,57,70,10,0,0,59,0,6,0,33,0,26,0,0,0,0,32,0,0,7,0,35,0,60,8,0,40,0,11,0,31,0,62,0,0,
    5,0,13,0,6,15,55,0,0,50,0,0,0,0,0,1,0,43,0,0,62,9,0,0,0,0,18,0,7,0,0,0,0,32,0,0,0,0,54,0,12,0,4,0,70,21,25,66,68,35,0,0,14,51,0,0,67,64,65,41,0,0,37,0,39,0,60,0,0,0,
    64,0,0,10,24,19,15,0,34,0,0,21,0,0,0,16,6,55,70,0,31,0,36,0,38,0,0,27,0,54,0,0,0,0,1,44,0,0,0,0,39,0,56,0,25,0,0,49,0,41,0,0,48,42,0,20,0,0,52,0,0,26,62,0,0,0,0,0,0,0,
    51,43,0,0,0,70,0,0,0,0,0,0,52,0,0,7,0,0,0,0,65,0,0,0,18,2,23,60,0,0,5,0,59,16,0,0,0,0,0,0,20,0,0,0,10,41,0,0,50,27,38,0,0,0,0,0,0,0,0,0,0,0,40,66,0,4,0,0,0,0,
    0,0,0,0,0,42,0,0,13,26,0,14,5,0,0,0,0,0,15,12,0,59,0,0,0,30,0,0,62,0,60,0,67,0,0,0,64,0,57,0,0,0,16,37,0,48,0,0,0,34,65,0,58,0,0,38,19,63,0,0,0,0,0,41,0,43,0,0,0,0,
    0,0,0,0,0,61,36,0,3,28,0,32,50,53,45,0,0,0,0,0,23,0,17,0,0,25,37,0,0,42,0,49,0,0,43,0,16,68,1,0,0,0,0,0,56,31,0,0,39,0,0,0,0,59,0,0,0,58,6,0,26,38,0,20,55,12,0,0,0,0,
    15,0,0,0,0,0,0,0,0,13,0,0,0,25,0,0,22,61,45,26,40,46,0,0,0,29,0,14,0,18,3,0,69,8,24,0,0,49,0,20,0,0,31,0,9,30,0,51,0,19,2,48,43,0,0,0,0,50,0,0,56,17,0,35,0,53,37,33,60,0,
    0,0,67,21,22,0,0,17,58,0,0,28,0,0,38,18,0,15,0,8,27,66,0,0,0,0,0,52,0,0,0,50,20,0,0,0,29,0,0,12,42,0,9,0,0,19,0,0,0,0,0,3,30,0,36,62,55,0,0,0,37,4,35,25,0,0,43,56,0,0,
    0,0,23,34,20,0,0,0,0,0,0,0,59,2,0,0,53,49,67,0,7,0,0,47,0,0,14,0,54,0,0,27,0,6,0,0,50,0,37,0,43,0,0,35,0,0,0,0,55,0,0,0,0,0,0,45,0,0,41,0,0,0,0,0,48,64,42,0,0,0,
    0,19,0,0,58,0,0,0,0,0,0,0,43,0,14,59,10,0,0,13,0,39,42,9,0,0,0,0,0,34,36,0,33,28,0,38,0,60,0,0,68,0,0,0,44,7,0,0,0,56,50,0,0,0,0,0,16,70,22,0,0,0,27,0,0,0,23,0,0,0,
    0,55,0,0,0,0,0,0,0,10,0,20,44,14,13,0,23,0,0,0,0,69,0,0,67,0,0,0,65,0,0,43,0,0,0,8,1,16,31,68,0,61,0,0,0,0,0,0,0,0,34,0,28,0,0,33,0,0,0,54,49,0,0,0,0,26,63,41,66,0,
    0,0,0,0,0,5,0,0,57,0,45,0,15,0,42,0,0,0,55,0,36,52,0,0,22,0,39,56,4,48,0,64,63,0,0,2,61,20,0,40,31,0,0,0,0,65,0,54,29,16,44,24,0,0,0,0,0,0,17,27,0,3,0,0,0,68,34,10,0,0,
    0,51,0,0,0,0,0,0,36,43,22,0,25,39,0,0,0,0,0,65,0,58,0,0,0,24,0,0,0,0,0,18,0,3,0,0,21,38,66,34,9,31,0,0,40,0,32,41,0,0,0,0,59,0,0,44,62,0,0,0,23,46,0,0,64,0,11,0,55,17,
    0,0,0,0,3,0,0,14,0,0,0,0,0,12,0,60,57,0,43,0,0,42,0,0,0,0,29,45,0,0,0,0,35,0,30,27,0,0,0,24,0,0,1,0,62,38,37,0,67,0,0,0,0,0,0,22,0,0,46,0,0,0,0,69,0,0,50,0,0,23,
    63,0,0,48,51,0,0,0,14,0,0,0,0,0,3,0,0,11,68,0,0,8,0,0,32,0,36,0,0,0,0,0,0,0,40,0,38,0,0,25,15,43,0,0,0,56,0,10,0,0,0,0,44,33,0,70,0,0,0,0,0,7,2,0,1,0,9,0,0,26,
    3,59,0,0,42,60,0,0,0,0,14,0,49,0,20,0,0,12,35,0,55,0,0,1,54,7,19,33,50,37,0,0,0,0,0,0,0,51,15,0,0,0,0,18,0,0,40,6,0,10,22,57,0,8,45,0,0,0,26,0,0,0,13,29,70,27,0,4,0,0,
    0,10,0,27,44,0,9,0,0,0,50,0,12,30,0,0,0,0,28,29,19,63,0,0,0,6,0,0,0,35,24,0,0,0,0,59,0,0,0,46,54,0,17,0,0,16,0,0,0,13,0,0,0,26,32,0,36,45,42,2,0,0,60,0,69,0,7,40,25,0,
    45,0,0,61,5,0,17,0,0,0,38,0,70,55,8,0,68,6,0,25,0,0,0,0,0,14,69,0,0,7,0,0,4,0,0,0,63,0,19,0,0,0,57,0,0,36,51,0,0,62,12,0,0,0,0,0,0,0,0,0,0,0,0,23,35,0,20,0,16,0,
    48,0,0,0,61,45,66,31,0,0,0,6,0,0,0,25,0,51,0,60,0,0,0,0,0,0,0,0,0,22,0,39,36,0,32,21,5,52,30,69,23,11,0,3,53,0,0,0,63,0,14,0,33,0,0,0,18,24,29,0,0,0,42,68,54,7,0,0,0,2,
    0,0,57,0,0,0,62,0,0,0,41,0,0,0,61,51,47,14,0,0,0,0,63,70,0,26,0,0,64,0,54,0,0,0,0,25,24,0,67,0,6,0,39,33,19,2,0,0,30,0,66,0,21,0,10,55,27,32,44,22,0,20,0,0,0,0,0,0,0,0,
    37,0,0,53,9,20,70,0,0,46,27,0,0,24,7,0,0,0,38,0,0,0,15,33,0,0,55,0,0,61,0,0,0,0,0,39,0,0,0,10,40,63,14,0,0,0,59,0,58,0,0,0,0,5,16,0,47,66,31,0,3,0,0,0,0,0,0,0,57,0,
    55,38,0,0,0,3,0,0,0,0,37,0,0,35,39,0,0,0,0,14,22,0,2,8,29,0,0,0,59,0,0,65,0,0,60,0,0,0,0,0,1,48,63,0,42,0,0,0,0,0,0,52,0,16,0,41,58,0,0,0,30,0,23,0,0,0,69,0,49,57,
    35,3,1,0,0,0,0,55,0,30,2,0,60,70,0,11,0,32,0,46,63,0,0,0,0,4,7,0,0,0,0,0,15,65,0,0,44,0,52,9,47,0,20,0,0,0,0,0,19,0,0,28,0,0,34,0,0,0,0,36,0,0,68,0,12,0,0,0,0,67,
    0,0,0,15,0,0,0,0,40,70,0,42,20,22,0,0,41,50,37,0,28,48,25,17,35,38,24,0,0,0,0,0,0,0,44,0,0,0,0,65,0,62,0,34,0,26,56,0,64,0,0,19,0,0,0,2,0,0,1,12,66,52,21,0,31,0,0,0,30,45,
    0,0,0,0,0,0,0,0,0,29,0,0,57,0,43,70,46,0,19,35,0,0,58,26,0,0,0,0,0,0,66,0,17,0,50,0,0,30,0,0,0,0,0,0,0,32,2,0,51,12,6,14,0,0,4,0,0,0,61,47,0,54,0,0,20,0,0,48,9,38,
    41,0,0,13,10,0,0,39,59,53,0,16,11,17,4,14,0,8,0,61,0,55,54,19,70,47,57,0,0,0,35,1,0,0,0,40,0,0,25,0,0,0,3,0,0,0,30,0,0,42,0,63,22,21,0,50,66,0,0,0,0,0,0,24,0,0,58,0,0,0,
    0,0,18,24,0,0,0,0,63,54,64,50,0,0,0,47,25,0,0,0,0,0,56,0,13,58,28,53,0,0,57,7,0,0,0,0,0,46,0,0,0,6,0,0,34,0,0,0,70,0,60,0,0,14,31,35,20,27,21,0,0,67,0,0,0,0,0,0,59,0,
    59,0,0,40,0,0,1,0,0,0,52,0,0,0,0,13,0,28,0,0,32,30,0,0,23,0,47,0,42,0,0,33,0,0,0,0,41,21,0,0,0,0,53,57,0,0,0,0,60,0,0,0,26,0,0,0,63,0,0,50,0,44,31,43,0,0,0,0,17,35,
    0,39,26,31,0,38,13,30,11,69,0,0,0,0,0,0,0,0,0,40,0,2,0,16,0,0,0,0,0,0,20,0,34,0,12,0,65,0,44,0,61,47,50,28,0,0,0,0,0,0,0,0,0,0,68,4,0,56,35,0,0,0,0,67,0,60,59,0,0,0,
    43,0,48,30,0,0,23,0,0,60,31,0,56,0,0,0,0,17,0,0,0,0,57,0,0,0,20,54,0,0,51,15,0,0,0,0,52,3,0,0,27,70,0,0,0,0,0,58,40,59,29,13,55,53,66,69,0,0,0,0,0,0,0,0,18,0,0,0,6,0,
    49,0,0,37,56,58,0,7,26,0,15,0,0,65,0,0,18,44,0,34,0,0,0,0,61,0,0,0,0,0,2,38,24,0,0,52,0,19,22,0,25,0,8,48,0,63,0,0,0,0,64,67,0,0,39,0,54,14,0,0,0,62,0,21,0,0,41,60,0,0,
    0,0,3,7,26,0,35,0,0,0,0,0,0,0,0,0,5,0,4,0,0,64,0,58,0,63,0,0,0,0,0,0,0,0,0,0,0,27,0,0,0,51,0,0,54,0,42,0,0,25,0,0,41,0,0,0,17,0,69,0,15,0,0,0,14,22,0,0,68,0,
    0,54,8,0,30,0,0,0,0,0,0,13,0,0,0,0,49,0,14,0,35,29,0,0,4,0,0,0,60,0,0,0,0,0,0,0,0,0,0,63,17,16,0,27,41,0,0,20,0,55,0,0,0,0,0,40,38,0,0,0,67,19,0,0,22,0,0,0,31,0,
    0,0,68,33,57,0,4,0,12,63,60,0,67,0,0,0,0,29,0,0,0,6,61,20,0,0,9,0,0,0,0,0,0,0,37,0,0,0,0,0,0,36,0,2,0,70,0,11,0,26,0,22,46,0,0,0,0,0,45,0,8,0,25,0,0,0,0,0,32,0,
    0,37,0,0,2,0,0,0,24,0,46,0,14,0,0,64,0,0,31,6,0,0,50,0,0,39,0,47,34,0,0,21,0,38,0,0,0,25,0,0,0,56,44,0,0,20,61,0,52,0,7,0,19,0,0,29,0,0,16,5,9,0,48,0,0,0,49,0,67,0,
    0,0,0,16,28,0,34,62,0,0,35,0,46,0,51,0,0,0,63,36,0,47,55,53,0,68,21,69,0,50,9,41,0,0,0,0,42,0,0,17,0,0,40,22,0,0,58,14,0,0,61,8,27,0,33,23,60,25,0,0,0,10,0,49,2,13,0,52,65,64,
    0,0,46,56,0,17,0,0,0,59,40,39,0,50,62,43,0,33,0,70,21,0,49,51,0,36,0,0,41,0,0,0,0,60,0,0,67,11,63,0,0,38,0,0,28,42,0,0,0,8,0,0,3,29,52,0,0,47,48,57,0,0,69,12,6,16,25,0,2,24,
    0,68,27,3,0,0,11,29,0,0,0,0,0,64,0,69,0,0,8,28,9,0,0,0,0,45,56,0,12,0,0,0,0,0,0,34,36,33,0,38,0,0,55,50,51,0,67,0,7,0,0,0,0,0,0,37,0,61,0,0,0,58,22,0,0,0,14,0,26,0,
    0,15,53,0,54,6,0,47,0,31,42,9,0,0,0,63,60,41,23,68,0,61,0,0,0,19,0,0,8,14,0,0,0,0,36,1,0,45,0,0,44,49,0,55,0,40,3,24,0,48,0,0,34,0,30,0,56,29,0,21,0,0,51,0,0,0,0,67,52,16,
    58,0,0,43,32,0,0,26,42,0,0,53,13,37,0,29,0,54,0,0,0,0,0,0,9,21,0,35,0,16,0,34,0,51,0,0,0,0,0,0,0,0,0,19,0,12,0,60,0,18,5,0,0,0,62,0,0,28,38,0,0,0,0,17,52,0,0,0,0,0,
    0,0,62,0,21,33,0,41,0,25,66,11,0,0,48,0,58,0,0,0,61,0,6,0,68,0,0,0,57,0,19,2,50,18,0,0,0,0,0,0,0,65,0,0,0,47,0,0,0,0,0,0,0,0,59,0,43,35,0,0,53,0,0,0,51,0,29,0,64,63,
    0,0,28,67,0,0,12,0,70,1,0,0,40,20,6,0,0,0,66,0,16,4,52,0,3,0,0,0,0,0,31,0,41,0,0,0,0,58,0,13,0,0,30,0,0,0,47,0,0,0,0,35,65,0,0,0,0,0,62,0,0,0,0,0,27,0,0,0,0,0,
    57,45,0,63,0,0,30,32,27,0,58,0,0,61,0,0,0,0,53,0,0,50,20,0,60,65,0,0,0,0,7,0,16,0,0,0,2,0,11,0,41,19,66,0,0,14,31,22,0,0,0,0,23,1,0,0,0,49,0,0,0,0,0,0,0,0,0,0,21,0,
    25,57,0,0,0,0,0,27,0,68,36,0,0,23,0,3,66,0,61,0,0,56,0,0,0,0,60,12,30,0,0,52,0,1,2,0,0,34,0,0,29,0,42,0,0,0,0,18,48,0,0,0,0,0,65,0,7,0,0,0,0,9,0,4,17,0,0,22,0,31,
    11,0,0,0,23,0,3,0,0,0,56,0,34,0,0,0,0,39,25,17,0,60,53,0,0,0,0,0,0,0,0,0,18,67,0,0,4,31,0,37,0,0,13,0,15,0,0,26,0,20,0,0,0,61,0,57,0,0,0,0,0,0,0,48,0,49,0,0,50,0,
    0,0,0,0,0,0,0,0,62,64,0,0,0,0,0,8,27,0,42,0,0,0,0,63,65,52,33,25,20,39,55,70,7,0,14,0,0,15,49,0,21,0,0,9,0,29,0,0,0,0,17,0,0,0,0,16,1,23,0,0,0,57,0,0,43,0,31,66,11,0,
    0,0,0,0,0,11,20,0,0,21,0,0,0,0,0,0,63,42,0,0,52,53,14,0,0,9,45,1,28,24,0,0,0,27,0,29,22,47,17,0,0,10,67,0,4,0,0,0,0,69,48,16,0,0,0,0,0,0,30,64,0,0,0,39,65,25,0,38,15,0,
    0,0,51,0,0,0,0,24,0,0,0,0,0,26,53,44,0,58,9,57,20,0,0,0,17,0,32,0,0,11,0,48,0,2,0,0,0,0,13,5,16,0,29,0,0,22,43,25,54,70,67,0,0,0,0,0,64,0,34,0,0,0,59,46,0,0,0,0,0,0,
    0,22,44,51,4,0,0,0,0,20,0,0,8,0,0,0,0,0,0,23,56,0,62,67,0,60,34,0,0,19,0,0,31,0,59,0,0,0,0,0,0,0,0,0,0,0,0,0,0,21,27,0,0,70,0,28,32,0,7,65,18,0,0,0,0,58,0,36,3,0,
    10,0,63,0,37,49,0,54,0,0,55,0,0,40,9,0,59,0,0,1,0,0,22,60,0,0,0,36,0,0,0,0,0,0,0,0,0,0,70,0,0,28,0,0,0,0,0,23,0,0,0,20,35,0,47,0,0,62,0,0,0,0,0,64,0,30,65,57,19,0,
    0,23,50,0,0,0,0,0,0,0,0,0,0,4,0,12,0,0,10,7,0,0,0,0,6,34,0,38,61,0,0,0,0,0,0,14,0,0,0,22,66,0,0,0,0,9,35,47,26,0,0,0,0,18,0,0,0,0,51,0,0,56,0,63,0,11,0,0,0,0,
    30,0,0,52,36,0,0,59,1,0,0,67,0,0,0,0,0,60,0,0,0,38,29,45,19,3,0,0,0,0,23,0,51,0,8,0,68,56,16,0,0,0,0,15,0,0,48,65,0,0,0,12,0,0,25,0,44,0,0,11,20,42,0,58,0,0,21,0,13,0,
    0,24,2,0,0,0,32,0,23,0,0,0,0,0,0,0,21,9,0,11,0,27,33,0,0,66,30,34,0,0,0,13,44,22,0,0,0,35,28,16,0,52,47,0,50,0,0,0,25,49,0,0,0,0,55,59,0,0,68,0,0,0,0,0,0,0,57,8,36,0,
    0,0,0,39,11,14,21,0,0,0,0,4,0,0,0,67,40,0,0,0,68,70,60,64,2,37,0,8,0,0,0,0,0,0,0,0,0,0,0,18,34,35,38,0,0,49,0,0,0,45,0,0,0,0,0,0,51,0,0,0,10,0,61,55,0,0,0,0,0,42,
    0,62,0,0,0,13,63,0,0,0,0,0,7,0,17,0,0,0,0,0,0,0,35,0,0,0,0,0,26,0,0,0,64,0,0,0,0,0,0,0,0,0,0,0,0,45,0,0,9,0,70,60,0,0,0,42,0,22,0,34,36,0,0,38,0,0,0,43,4,0,
    0,0,0,57,52,48,0,0,0,0,17,2,0,0,54,0,7,0,1,0,0,0,47,39,45,62,35,0,0,0,0,20,0,25,0,67,59,0,26,0,49,55,22,0,0,0,0,0,4,0,43,0,66,0,38,8,53,34,0,0,31,0,0,0,0,6,0,0,0,0,
    61,66,0,6,0,0,46,0,0,0,0,30,0,0,0,0,0,7,0,45,11,31,13,65,0,51,41,24,40,52,0,0,42,0,0,0,32,12,3,0,70,0,23,0,0,0,0,0,0,37,0,54,5,0,0,0,0,39,36,0,0,0,1,59,0,0,0,0,62,0
  };


  const int* qcp[] = {
    &d10_67_0[0],  &d10_67_1[0],  &d10_67_2[0],  &d10_67_3[0],  &d10_67_4[0],
    &d10_67_5[0],  &d10_67_6[0],  &d10_67_7[0],  &d10_67_8[0],  &d10_67_9[0],
    &d10_67_10[0], &d10_67_11[0], &d10_67_12[0], &d10_67_13[0], &d10_67_14[0],

    &d15_120_0[0],  &d15_120_1[0],  &d15_120_2[0],  &d15_120_3[0],  &d15_120_4[0],
    &d15_120_5[0],  &d15_120_6[0],  &d15_120_7[0],  &d15_120_8[0],  &d15_120_9[0],
    &d15_120_10[0], &d15_120_11[0], &d15_120_12[0], &d15_120_13[0], &d15_120_14[0],

    &d20_187_0[0],  &d20_187_1[0],  &d20_187_2[0],  &d20_187_3[0],   &d20_187_4[0],
    &d20_187_5[0],  &d20_187_6[0],  &d20_187_7[0],  &d20_187_8[0],   &d20_187_9[0],
    &d20_187_10[0], &d20_187_11[0], &d20_187_12[0], &d20_187_13[0],  &d20_187_14[0],

    &d25_264_0[0],  &d25_264_1[0],  &d25_264_2[0],  &d25_264_3[0],  &d25_264_4[0],
    &d25_264_5[0],  &d25_264_6[0],  &d25_264_7[0],  &d25_264_8[0],  &d25_264_9[0],
    &d25_264_10[0], &d25_264_11[0], &d25_264_12[0], &d25_264_13[0], &d25_264_14[0],

    &d30_316[0], &d30_320[0],

    &d30_374_1[0],  &d30_374_2[0],  &d30_374_3[0],  &d30_374_4[0],  &d30_374_5[0],
    &d30_374_6[0],  &d30_374_7[0],  &d30_374_8[0],  &d30_374_9[0],  &d30_374_10[0],
    &d30_374_11[0], &d30_374_12[0], &d30_374_13[0], &d30_374_14[0], &d30_374_15[0],
    &d30_374_16[0], &d30_374_17[0], &d30_374_18[0], &d30_374_19[0], &d30_374_20[0],
    &d30_375_21[0], &d30_375_22[0], &d30_375_23[0], &d30_375_24[0], &d30_375_25[0],
    &d30_375_26[0], &d30_375_27[0], &d30_375_28[0], &d30_375_29[0], &d30_375_30[0],
    &d30_375_31[0], &d30_375_32[0], &d30_375_33[0], &d30_375_34[0], &d30_375_35[0],
    &d30_375_36[0], &d30_375_37[0], &d30_375_38[0], &d30_375_39[0], &d30_375_40[0],

    &d33_381[0],
    &d35_405[0],
    &d40_528[0], &d40_544[0], &d40_560[0],
    &d50_750_bal[0], &d50_825_bal[0],
    &d60_1080_bal[0], &d60_1152_bal[0], &d60_1440[0], &d60_1620[0],
    &d70_2450[0], &d70_2940[0]
  };

  const char* name[] = {
    "10-67-0",  "10-67-1",  "10-67-2",  "10-67-3",  "10-67-4",
    "10-67-5",  "10-67-6",  "10-67-7",  "10-67-8",  "10-67-9",
    "10-67-10", "10-67-11", "10-67-12", "10-67-13", "10-67-14",

    "15-120-0",  "15-120-1",  "15-120-2",  "15-120-3",  "15-120-4",
    "15-120-5",  "15-120-6",  "15-120-7",  "15-120-8",  "15-120-9",
    "15-120-10", "15-120-11", "15-120-12", "15-120-13", "15-120-14",

    "20-187-0",  "20-187-1",  "20-187-2",  "20-187-3",   "20-187-4",
    "20-187-5",  "20-187-6",  "20-187-7",  "20-187-8",   "20-187-9",
    "20-187-10", "20-187-11", "20-187-12", "20-187-13",  "20-187-14",

    "25-264-0",  "25-264-1",  "25-264-2",  "25-264-3",  "25-264-4",
    "25-264-5",  "25-264-6",  "25-264-7",  "25-264-8",  "25-264-9",
    "25-264-10", "25-264-11", "25-264-12", "25-264-13", "25-264-14",

    "30-316", "30-320",

    "30-374-1",  "30-374-2",  "30-374-3",  "30-374-4",  "30-374-5",
    "30-374-6",  "30-374-7",  "30-374-8",  "30-374-9",  "30-374-10",
    "30-374-11", "30-374-12", "30-374-13", "30-374-14", "30-374-15",
    "30-374-16", "30-374-17", "30-374-18", "30-374-19", "30-374-20",
    "30-375-21", "30-375-22", "30-375-23", "30-375-24", "30-375-25",
    "30-375-26", "30-375-27", "30-375-28", "30-375-29", "30-375-30",
    "30-375-31", "30-375-32", "30-375-33", "30-375-34", "30-375-35",
    "30-375-36", "30-375-37", "30-375-38", "30-375-39", "30-375-40",

    "33-381",
    "35-405",
    "40-528", "40-544", "40-560",
    "50-750-bal", "50-825-bal",
    "60-1080-bal", "60-1152-bal", "60-1440", "60-1620",
    "70-2450", "70-2940",
    nullptr
  };

}

// STATISTICS: example-any

