/* -*-  mode:c; tab-width:8; c-basic-offset:8; indent-tabs-mode:nil;  -*- */
/*
   Copyright (C) 2016 by Ronnie Sahlberg <ronniesahlberg@gmail.com>

Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef _GNU_SOURCE
#define _GNU_SOURCE
#endif

#include <inttypes.h>
#if !defined(__amigaos4__) && !defined(__AMIGA__) && !defined(__AROS__)
#include <poll.h>
#endif
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "smb2.h"
#include "libsmb2.h"
#include "libsmb2-raw.h"

#include <sys/syscall.h>
#include <dlfcn.h>

int alloc_fail = -1;

void *(*real_malloc)(size_t size);
void *(*real_calloc)(size_t nelem, size_t size);

void *malloc(size_t size)
{
        static int call_idx = 0;

        call_idx++;

        if (call_idx == alloc_fail) {
                return NULL;
        }
        return real_malloc(size);
}

void *calloc(size_t nelem, size_t size)
{
        static int call_idx = 0;

        call_idx++;

        if (call_idx == alloc_fail) {
                return NULL;
        }
        return real_calloc(nelem, size);
}

int usage(void)
{
        fprintf(stderr, "Usage:\n"
                "smb2-ls-sync <smb2-url>\n\n"
                "URL format: "
                "smb://[<domain;][<username>@]<host>[:<port>]/<share>/<path>\n");
        exit(1);
}

int main(int argc, char *argv[])
{
        struct smb2_context *smb2;
        struct smb2_url *url;
        struct smb2dir *dir;
        struct smb2dirent *ent;
        char *link;
        int rc = 1;

	if (getenv("ALLOC_FAIL") != NULL) {
		alloc_fail = atoi(getenv("ALLOC_FAIL"));
	}
        /* https://bugzilla.redhat.com/show_bug.cgi?id=2333389 */
        /* skip these test as they are known failures and have been reported */
        if (alloc_fail == 2) {
                alloc_fail = -1;
        }
        if (alloc_fail == 18) {
                alloc_fail = -1;
        }

	real_malloc = dlsym(RTLD_NEXT, "malloc");
	real_calloc = dlsym(RTLD_NEXT, "calloc");
        
        printf("Alloc fail at %d\n", alloc_fail);

        if (argc < 2) {
                usage();
        }

	smb2 = smb2_init_context();
        if (smb2 == NULL) {
                fprintf(stderr, "Failed to init context\n");
                exit(1);
        }

        url = smb2_parse_url(smb2, argv[1]);
        if (url == NULL) {
                fprintf(stderr, "Failed to parse url: %s\n",
                        smb2_get_error(smb2));
                exit(1);
        }

        smb2_set_security_mode(smb2, SMB2_NEGOTIATE_SIGNING_ENABLED);
	if (smb2_connect_share(smb2, url->server, url->share, url->user) < 0) {
		printf("smb2_connect_share failed. %s\n", smb2_get_error(smb2));
                goto out_context;
	}

	dir = smb2_opendir(smb2, url->path);
	if (dir == NULL) {
		printf("smb2_opendir failed. %s\n", smb2_get_error(smb2));
                goto out_disconnect;
	}

        while ((ent = smb2_readdir(smb2, dir))) {
                char *type;
                time_t t;

                t = (time_t)ent->st.smb2_mtime;
                switch (ent->st.smb2_type) {
                case SMB2_TYPE_LINK:
                        type = "LINK";
                        break;
                case SMB2_TYPE_FILE:
                        type = "FILE";
                        break;
                case SMB2_TYPE_DIRECTORY:
                        type = "DIRECTORY";
                        break;
                default:
                        type = "unknown";
                        break;
                }
                printf("%-20s %-9s %15"PRIu64" %s", ent->name, type, ent->st.smb2_size, asctime(localtime(&t)));
                if (ent->st.smb2_type == SMB2_TYPE_LINK) {
                        char buf[256];
                        
                        if (url->path && url->path[0]) {
                                if (asprintf(&link, "%s/%s", url->path, ent->name) < 0) {
                                        printf("asprintf failed\n");
                                        goto out_disconnect;
                                }
                        } else {
                                if (asprintf(&link, "%s", ent->name) < 0) {
                                        printf("asprintf failed\n");
                                        goto out_disconnect;
                                }
                        }
                        if (smb2_readlink(smb2, link, buf, 256) == 0) {
                                printf("    -> [%s]\n", buf);
                        } else {
                                printf("    readlink failed\n");
                        }
                        free(link);
                }
        }

        rc = 0;
        smb2_closedir(smb2, dir);
 out_disconnect:        
        smb2_disconnect_share(smb2);
 out_context:
        smb2_destroy_url(url);
        smb2_destroy_context(smb2);

	return rc;
}
